(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 9.5.00 by Jan
 *
 * Time-stamp: <Friday 2 July 10 10:20:26 tews@blau.inf.tu-dresden.de>
 *
 * Funktor Actions etc.
 *
 * $Id: morphism.mli,v 1.7 2010-07-02 10:55:55 tews Exp $
 *
 *)

open Top_variant_types
open Top_classtypes
open Classtypes

(************************************************************************
 *
 * variance separation is a special substitution:
 * The substitution type contains two substitutes, one for
 * negative occurence (first) and one for positive ones.
 * The substitution is done depending on the variances. Eg, with
 * the substitution type
 *   [ (Self,  (Self_neg, Self_pos)) ]
 * the type
 *   [Self -> Self] -> Self
 * is transformed into
 *   [Self_pos -> Self_neg] -> Self_pos
 *
 *)


type ccsl_variance_separation_type =
  (ccsl_input_types * (ccsl_input_types * ccsl_input_types)) list


(*************************************************************************
 *
 * substitute_variance_types : substitute types
 *
 * Arguments:
 *   subst		the substitution
 *   typ		the type
 *)

val substitute_variance_types :
  ccsl_variance_separation_type -> 
    ccsl_input_types ->
      ccsl_input_types


(*************************************************************************
 *
 * substitute_variance_argument_list : substitute argument lists
 *
 * Arguments:
 *   subst		the substitution
 *   args		the argument list
 *   params		the prameter list (needed for the variance info)
 *)


val substitute_variance_argument_list :
  ccsl_variance_separation_type ->
    ccsl_argument_type list -> 
      ccsl_parameter_type list -> 
	ccsl_argument_type list


(***********************************************************************
 *
 * action of the interface functor 
 *
 * The functor comes in as type expression. What are considered as 
 * the arguments of the functor depends on other parameters.
 * 
 * ccsl_action_type describes the arguments for the functor. For 
 * each argument of the functor two functions are given (as expressions),
 * the first one for contravariant occurences, the second one for 
 * covariant onces. The domain and the codomain of the action of the 
 * functor must be given as two substitutions that generate these 
 * types.
 * 
 * For example: consider H as a type expression that involves Self. 
 * Then the two substitutions
 *   domain_subst : [Self, (Self1,Self1)]
 *   codomain_subst : [Self, (Self1, Self2)] 
 * give H(Self1,Self1) and H(Self1,Self2). Let expr be an 
 * expression of type H(Self1,Self1). To get H(id, f) (expr) you 
 * specify
 *    actions : [Self, (None, Some f)]
 * 
 * Arguments
 *    name_space    	Name space
 *    id_crea_flag	if true identifiers are created before the 
 *    		      	type substitution, otherwise after
 *    actions		the arguments of the functor as an assoc list
 *    variance		current variance, true: positive, false : negative
 *    domain_subst	substitution for getting the domain
 *    codomain_subst	substitution for getting the codomain
 *    expression	the expression to which the action is applied
 *    typ		Type describing the functor
 *    
 * There is the following invariant:
 * 
 * `expression' is an expression of domain type, ie. typ[domain_subst]
 * 
 * the result is an expression of codomain type, ie. typ[codomain_subst]
 * corresponding to  typ[actions] (expression)
 *)

type ccsl_action_type =
    (ccsl_input_types * 
     (ccsl_expressions option * ccsl_expressions option)) list

val functor_action :
    ccsl_name_space_type ->				(* name_space *)
      bool ->						(* id_crea_flag *)
	ccsl_action_type ->				(* actions *)
	  bool ->					(* variance *)
	    ccsl_variance_separation_type ->		(* domain_subst *)
	      ccsl_variance_separation_type ->		(* codomain subst *)
		ccsl_expressions ->			(* expression *)
		  ccsl_input_types ->			(* typ *)             
		    ccsl_expressions


(************************************
 * functor action for argument lists
 *)

val functor_action_argument_list :
    ccsl_name_space_type ->				(* name_space *)
      bool ->						(* id_crea_flag *)
	ccsl_action_type ->				(* actions *)
	  bool ->					(* variance *)
	    ccsl_variance_separation_type ->		(* domain_subst *)
	      ccsl_variance_separation_type ->		(* codomain subst *)
		ccsl_argument_type list ->		(* the arguments *)
		  ccsl_parameter_type list ->		(* the parameters *)
		    ccsl_expressions list




(* the main part of the condition that has to hold for coalgebra 
 * morphisms:
 *  
 *    X ---c---->H(X,X)
 *    |               \
 *    |               H(X,f)
 *    |                 \
 *    |                 \/
 *    f                 H(X,Y)
 *    |                 /\
 *    |                 /
 *    |               H(f,Y)             
 *    \/             /
 *    Y --d---->H(Y,Y)
 *   
 *    generates the equation(s) for the commuting diagram above, with respect 
 *    to a given list of methods. This is done by calling the REAL morphism
 *    action function after massaging the domain of the type of each function.
 *
 * parameters
 *     morphism            -> the morphism between the state spaces of the
 *                            coalgebras that we check
 *     name_space          -> name space for name creation
 *     coalgebra1          -> the domain coalgebra   (as TERM)
 *     coalgebra2          -> the codomain coalgebra (as TERM)
 *     ancestor_list       -> list of ancestors, they are in the form of a 
 *                            ('class * 'class top_pre_argument list ) list
 *     anc_morphism        -> function to get the 
 *     method_list         -> the list of methods    (as 'mem list)
 *     selfvar             -> the variable of Self, to which we apply
 *                            the equation (as TERM)
 *     self_type_argument  -> the types of Self1 and Self2, as an argument, to 
 *                            instanciate the ancestors right
 *     self1_typ           -> type for covariant Self
 *     self2_typ           -> type for contravariant Self
 *     direction           -> direction of the equation:
 *                            true yields   H(X,f) o c  =  d o f
 *                            false yields       d o f  =  H(X,f) o c
 * 
 *)

val coalgebra_morphism:
  ccsl_expressions ->                                           (* morphism *)
    ccsl_name_space_type ->				      (* name_space *)
      ccsl_expressions ->					(* coalg 1 *)
   	ccsl_expressions ->					(* coalg 2 *)
	  (ccsl_iface_type * ccsl_argument_type list) list ->	(* anc list *)
	    (Classtypes.ccsl_iface_type -> string) ->		(* anc_morph *)
	      Classtypes.ccsl_member_type list ->		(* methods *)
		ccsl_argument_type list ->			(* self args *)
		  ccsl_top_types ->				(* self1_typ *)
		    ccsl_top_types ->				(* self2_typ *)
                      bool ->		                        (* direction *)
			ccsl_formulas



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

