(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 8 October 01 17:58:00 tews@ithif51>
 *
 * utility functions for pvs proofs
 *
 * $Id: pvs_proof_util.ml,v 1.2 2002/01/23 16:00:23 tews Exp $
 *
 *)

open Top_variant_types;;


let simple_command s = LList([LAtom s])

let apply_command c l = LList([LAtom c] @ l)

(*******************************************************************
 * PVS proof commands in alphabetical order 
 *)

let apply c = apply_command "APPLY" [c]

let auto_rewrite_defs = simple_command "AUTO-REWRITE-DEFS"

let auto_rewrite_theories thl = 
  apply_command "AUTO-REWRITE-THEORIES" 
    (List.map (fun th -> LAtom ("\"" ^ th ^ "\"")) thl)

let pvs_assert = simple_command "ASSERT"

let branch step steplist = apply_command "BRANCH" [step;LList(steplist)]

let case ex = LList( [ LAtom("CASE "); 
			       LAtom("\"" ^ ex ^ "\"") ])

let expand name = LList( [ LAtom("EXPAND "); 
			       LAtom("\"" ^ name ^ "\"") ])

let expand_num name i = LList( [ LAtom("EXPAND "); 
				       LAtom("\"" ^ name ^ "\"");
				       LAtom(string_of_int i)])

let expand_nums name i j =
  if j=0 then expand_num name i else
  	LList( [ LAtom("EXPAND "); 
		 LAtom("\"" ^ name ^ "\"");
		 LAtom(string_of_int i);
		 LAtom(" ");
		 LAtom(string_of_int j)])

let extensionality = simple_command "APPLY-EXTENSIONALITY :hide? t"

let fail = simple_command "FAIL"

let flatten = simple_command "FLATTEN"

let grind = simple_command "GRIND"

let hide sequent = apply_command "HIDE" [LAtom(string_of_int sequent)]

let hide_nums num_list = LList(LAtom("HIDE")
			       ::
			       (List.map
				  (fun i ->
				     LAtom(" " ^ (string_of_int i)))
				  num_list))

let induct name = LList( [ LAtom("INDUCT"); 
			       LAtom("\"" ^ name ^ "\"") ])

let inst_question = simple_command "INST?"

let inst_num i = LList( [ LAtom("INST? "); 
			       LAtom( string_of_int i ) ])

let inst_val i name = LList( [ LAtom("INST "); 
			       LAtom( string_of_int i );
			       LAtom(" " ^ name)])


let lemma name = LList( [ LAtom("LEMMA "); 
			       LAtom("\"" ^ name ^ "\"") ])

let liftif = simple_command "LIFT-IF"

let liftif_num i = LList( [ LAtom("LIFT-IF "); 
			       LAtom( string_of_int i ) ])

let name shortname longname = LList( [ LAtom("NAME "); 
			               LAtom("\"" ^ shortname ^ "\"");
                                       LAtom("\"" ^ longname ^ "\"") ])

let split = simple_command "SPLIT"

let split_num i = LList( [ LAtom("SPLIT "); 
			       LAtom( string_of_int i ) ])

let postpone = simple_command "POSTPONE"

let propax = simple_command "PROPAX"

let reduce = simple_command "REDUCE"

let repeat c = apply_command "REPEAT" [c]

let replace_num i = LList( [ LAtom("REPLACE "); 
			       LAtom( string_of_int i ) ])

let rewrite name = LList( [ LAtom("REWRITE "); 
			       LAtom("\"" ^ name ^ "\"") ])

let rewrite_left name = LList( [ LAtom("REWRITE "); 
				 LAtom("\"" ^ name ^ "\"");
				 LAtom(":DIR");
				 LAtom("RL")])

let skip = simple_command "SKIP"

let skolem_typepred = simple_command "SKOLEM-TYPEPRED"

let skosimp = simple_command "SKOSIMP"

let skosimp_star = simple_command "SKOSIMP*"

let then_affe (* steplist *) = apply_command "THEN@" (* steplist *)

let type_pred name = LList( [ LAtom("TYPEPRED "); 
			      LAtom("\"" ^ name ^ "\"") ])

let pvs_try c then_ else_ = apply_command "TRY" [c;then_;else_]

let use name = LList( [ LAtom("USE "); 
			       LAtom("\"" ^ name ^ "\"") ])



(************************************************************************
 * utility functions to built proofs
 *)

let make_simple_proof command_list =
  PTree( command_list, [] )

(************************************************************************
 * more complicated strategies
 *)

(* super_inst
 *
 * instanciates a list in the antecedent until we hit the right 
 * one. If something goes wrong we try the next sequent
 *
 * 	(APPLY (REPEAT
 * 		(TRY (BRANCH (INST? -1) ((SKIP) (FAIL))) (HIDE -1)
 * 		 (HIDE -1)))))
 *)

let super_inst = 
  apply( repeat( pvs_try (branch (inst_num (-1)) [skip; fail])
		       (hide (-1)) (hide (-1))))




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** End: ***)

