(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 25.6.99 by Hendrik
 *
 * Time-stamp: <Sunday 19 May 02 19:52:45 tews@ithif56.inf.tu-dresden.de>
 *
 * Substitution and class instanciation
 *
 * $Id: substitution.ml,v 1.15 2002/05/22 13:42:43 tews Exp $
 *
 *)

open Util
open Top_variant_types
open Iface_class 
open Member_class
open Symbol
open Types_util


(***********************************************************************
 *
 * class and adt instanciation
 *
 *)

(**********************
 * instanciation is split into an inernal and an external version:
 * 
 * The external version (instanciate) takes an iface and an 
 * argument list. From that it computes a substitution (that is two 
 * assoc lists) and calls the internal version.
 * 
 * The internal version (internal_instanciate) takes an iface,
 * a pair of eqality functions and a substitution, it uses both 
 * arguments to call substitute and substitute_argument.
 *
 * The internal version is recursive and works down all ancestors.
 *)

let rec internal_instanciate iface subst top_args method_filter =
  let _ = assert(iface#inheritance_ready) in
  let inst = new ccsl_pre_iface_class 
	       iface#get_token (new_local()) false in
  let do_ancestor = function
    | Resolved_renaming(ancestor, args, renamed_methods, inst_anc, loc) ->
	let subs_args =		
	  ccsl_substitute_arguments subst args in
	let inst_anc = internal_instanciate inst_anc subst
			 subs_args method_filter in
	let _ = inst#inherit_locals (inst_anc#get_inherited_locals) 
	in
	  (Resolved_renaming(
	       ancestor,
	       subs_args,
	       renamed_methods,
	       inst_anc,
	       loc
	   ))
    | _ -> assert(false)
  in let add_member m sort =
      let m' = new ccsl_pre_member_class 
		 inst
		 m#get_token
		 m#get_old_names
		 (ccsl_substitute_types subst m#get_domain)
		 (ccsl_substitute_types subst m#get_codomain)
		 m#get_visibility
		 sort
      in
	inst#add_member m';
	m'
  in let attr_list = ref []
  in let do_member m = match m#get_sort with
    | Proper_Attribute (Some u) ->
	let m' = add_member m (Proper_Attribute None) 
	in
	  attr_list := (u#get_name, m') :: !attr_list

    | Adt_Recognizer 
    | Adt_Accessor 
    | Normal_Method
    | Defined_Method
    | Var_Constructor
    | Const_Constructor
    | Adt_Var_Constructor
    | Adt_Const_Constructor
    | Adt_Reduce
    | Class_Coreduce
    | Class_Special
    | GroundTerm
    | InfixGroundTerm -> ignore( add_member m m#get_sort )

    | Update_Method -> 
	let u = add_member m Update_Method in
	let m = List.assoc u#get_name !attr_list
	in
	  m#register_update_method u
	  

    | Proper_Attribute None -> assert(false)

  in
    (* the instanciated class does neither have parameters, 
     * nor imports, nor other fancy stuff
     *)
    inst#become_instanciated;
    inst#set_arguments top_args;
    (match iface#get_kind with
       | Spec_adt -> inst#become_adt 
       | Spec_class -> inst#become_class
       | Spec_sig -> inst#become_sig
       | Spec_Spec -> assert(false)
    );
					(* add ancestors *)
    inst#set_ancestors (List.map do_ancestor iface#get_ancestors);
					(* add methods *)
    List.iter do_member (List.filter method_filter iface#get_members);

    inst#inheritance_done;		(* set inheritance flag *)

(* components are added in the component pass
 *					   (* add components *)
 *     (List.iter (fun (c,args) -> 
 * 		     inst#add_component (c, ccsl_substitute_arguments subst 
 * 					   args))
 * 	  (List.rev iface#get_components));
 *)
    inst


let instanciate iface args method_filter =
  let _ = assert(check_parameters iface#get_parameters args) in
  let subst = make_substitution iface#get_parameters args
  in
    internal_instanciate iface subst args method_filter


(***********************************************************************
 *
 * Instanciation feature
 * 
 *)

let store_instanciated_iface iface args = 
  let rec search = function
    | [] -> raise Table.Not_defined
    | (_,sym, block) :: syms -> (match sym with
			| AdtSymb adt -> adt, block
			| ClassSymb cl -> cl, block
			| SigSymb _
			| CCSL_GroundTypeSymb _ 
			| TypevarSymb _ 
			| VarSymb _ 
			| MemberSymb _
			| InstClassSymb _
			| InstAdtSymb _ -> search syms
				)
  in
  let (orig, block) = 
    search (Table.find_all_with_block ccsl_gst CCSL_ID iface#get_name) in

  let _ = assert(orig#get_kind = iface#get_kind) in
  let _ = assert(check_parameters orig#get_parameters args) in
  let shape = Instanciation(args) in
  let symbol = match iface#get_kind with
    | Spec_class -> InstClassSymb(iface)
    | Spec_adt -> InstAdtSymb(iface)
    | Spec_sig
    | Spec_Spec -> assert(false)
  in
    Table.local_overload block InstAdtOrClass iface#get_name shape symbol



let find_instanciated_iface iface args =
  assert(iface#inheritance_ready);
  try 
       (* instanciated classes and adts are stored
	* in name space InstAdtOrClass
	*)
    match Table.find_overloaded ccsl_gst InstAdtOrClass 
      iface#get_name (Instanciation args) with
					(* instanciations *)
      | InstClassSymb cl -> cl
      | InstAdtSymb adt -> adt
				(* the rest should not be in InstClasses *)
      | CCSL_GroundTypeSymb _ 
      | TypevarSymb _ 
      | ClassSymb _ 
      | AdtSymb _ 
      | SigSymb _
      | VarSymb _ 
      | MemberSymb _ -> assert(false)
  with
      Table.Not_defined -> 
	let _ = assert( check_parameters iface#get_parameters args ) in
	let inst_orig = instanciate iface args (fun _ -> true) in
	let _ = store_instanciated_iface inst_orig args in
	  inst_orig


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)





