(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Tuesday 16 July 02 10:45:04 tews@ithif51>
 *
 * Utility functions for ccsl input types
 *
 * $Id: types_util.ml,v 1.24 2002/07/18 13:43:25 tews Exp $
 *
 *)

open Util
open Top_variant_types
open Top_variant_types_util (* for resolution_of *)
open Logic_util
open Classtypes
;;

(***********************************************************************
 *
 * counting selfs and checking for the various functor classes
 *
 *)

(* count self's, works only for ccsl_input_types *)
let rec count_self = function
  | Groundtype(_, args) 	-> count_self_args args
  | BoundTypeVariable _ 	-> 0
  | Self			-> 1
  | Carrier 			-> 0
  | Bool 			-> 0
  | Function( dom, codom) 	-> count_self dom + count_self codom
  | Product( types ) 		-> 
      List.fold_left (fun n typ -> n + count_self typ) 0 types
  | Class(cl, args )		-> count_self_args args
  | Adt(cl, flag, args )	-> count_self_args args
		(* do records as an exception *)
  | Record label_list 		->
      List.fold_left (fun n (_,typ) -> n + count_self typ) 0 label_list
					(* not in ccsl input types *)
  | TypeConstant _
  | IFace _
  | FreeTypeVariable _
  | Array _
  | Predtype _
  | SmartFunction _ -> assert(false)


and count_self_args args = List.fold_left
			(fun n arg -> match arg with 
			   | TypeArgument(typ) -> count_self typ + n
			)
			   0 args

(* count carrier's, works only for ccsl_input_types *)
let rec count_carrier = function
  | Groundtype(_, args) 	-> count_carrier_args args
  | BoundTypeVariable _ 	-> 0
  | Self			-> 0
  | Carrier 			-> 1
  | Bool 			-> 0
  | Function( dom, codom) 	-> count_carrier dom + count_carrier codom
  | Product( types ) 		-> 
      List.fold_left (fun n typ -> n + count_carrier typ) 0 types
  | Class(cl, args )		-> count_carrier_args args
  | Adt(cl, flag, args )	-> count_carrier_args args
		(* do records as an exception *)
  | Record label_list 		->
      List.fold_left (fun n (_,typ) -> n + count_carrier typ) 0 label_list
					(* not in ccsl input types *)
  | TypeConstant _
  | IFace _
  | FreeTypeVariable _
  | Array _
  | Predtype _
  | SmartFunction _ -> assert(false)


and count_carrier_args args = List.fold_left
			(fun n arg -> match arg with 
			   | TypeArgument(typ) -> count_carrier typ + n
			)
			   0 args


    (****************************
     * check_for_params walks through the type 
     * and returns true, if in typ, there is no 
     * typeparameter from param_p, if one parameter 
     * from param_p is found, it returns false.
     *
     * val check_for_params : 
     * 	 (('cl,'mem) top_pre_identifier_record_type -> bool) ->
     * 	   ('cl,'mem) top_pre_types -> bool
     * 
     *)
let rec check_for_params param_p = function
  | Groundtype (_, args) 	-> check_for_params_args param_p args
  | BoundTypeVariable id 	-> not(param_p id)
  | Self			-> true
  | Carrier 			-> true
  | Bool 			-> true
  | Function( dom, codom) 	-> (check_for_params param_p dom 
 				    && check_for_params param_p codom)
  | Product( types ) 		-> 
      List.for_all (fun typ -> check_for_params param_p typ) types
  | Class(cl, args )		-> check_for_params_args param_p args
  | Adt(cl, flag, args )	-> check_for_params_args param_p args
		(* do records as an exception *)
  | Record label_list 		->
      List.for_all (fun (_,typ) -> check_for_params param_p typ) label_list
					(* not in ccsl input types *)
  | TypeConstant _
  | FreeTypeVariable _
  | IFace _
  | Array _
  | Predtype _
  | SmartFunction _ -> assert(false)


and check_for_params_args param_p args = 
  List.for_all
    (fun arg -> match arg with 
       | TypeArgument(typ) -> check_for_params param_p typ
    ) args


    (****************
     * checks if type corresponds to a constant functor, 
     * in which certain parameters (given by param_p) do not occur
     * needed in predicate lifting
     * 
     * val constant_type : ccsl_input_types -> bool
     *)
let constant_type param_p typ = 
  (count_self typ = 0) && (count_carrier typ = 0)
  &&
  check_for_params param_p typ


let constant_arg_list param_p argumentlist =
  List.for_all 
    (function TypeArgument t -> constant_type param_p t)
    argumentlist


(**********************************************************************
 * returns true if argument contains a free type variable
 * 
 * val type_is_nonground : ('cl, 'mem) top_pre_types -> bool
 *)

let rec type_is_nonground = function
  | Groundtype(_, args) 	-> args_are_nonground args
  | BoundTypeVariable _ 	-> false
  | Self			-> false
  | Carrier 			-> false
  | Bool 			-> false
  | Function( dom, codom) 	-> 
      (type_is_nonground dom) or (type_is_nonground codom)
  | Product( types ) 		-> 
      List.exists (fun typ -> type_is_nonground typ) types
  | Class(cl, args )		-> args_are_nonground args
  | Adt(cl, flag, args )	-> args_are_nonground args
  | FreeTypeVariable _ 		-> true

					(* not in ccsl input types *)
  | TypeConstant _
  | Record _
  | IFace _
  | Array _
  | Predtype _
  | SmartFunction _ -> assert(false)

and args_are_nonground args =
  List.exists
    (function | TypeArgument(typ) -> type_is_nonground typ)
    args



    (****************
     * returns true if the type variable occurs in the type
     *
     * val occur_check : Type_variable.t -> ('cl, 'mem) top_pre_types  -> bool
     *
     *)

let rec occur_check tv t = 
  let recurse = occur_check tv 
  in
    match t with
      | FreeTypeVariable otv 	-> Type_variable.eq otv tv
      | Groundtype(_, args) 	-> (List.exists 
				      (function TypeArgument t -> 
					 recurse t)
				      args)
      | BoundTypeVariable _ 	-> false
      | Self			-> false
      | Carrier 		-> false
      | Bool 			-> false
      | Function( dom, codom) 	-> (recurse dom) or (recurse codom)
      | Product( types ) 	-> List.exists recurse types
      | Class(cl, args )	-> (List.exists 
				      (function TypeArgument t -> 
					 recurse t)
				      args)
      | Adt(cl, flag, args )	-> (List.exists 
				      (function TypeArgument t -> 
					 recurse t)
				      args)

					(* not in ccsl input types *)
      | TypeConstant _
      | Record _
      | IFace _
      | Array _
      | Predtype _
      | SmartFunction _ 	-> assert(false)




(***********************************************************************
 *
 * check if this type yields a successor state in SELF
 * needed for enumeration type of successor states
 *
 *)

let rec has_successor_state = function
      | Groundtype(_, args) -> args_have_successor_state args
      | BoundTypeVariable _ -> false
      | Self -> true
      | Bool -> false
      | Function (_,l) -> has_successor_state l	      
      | Product l -> List.exists has_successor_state l
      | Class(_, args) -> args_have_successor_state args
      | Adt (_,_,args) -> args_have_successor_state args
					(* may not occur in method types *)
      | Carrier
      | Predtype _
      | Record _
      | IFace _
      | Array _ 	      
      | FreeTypeVariable _
      | TypeConstant _
      | SmartFunction _ -> assert(false)
and args_have_successor_state args = 
  List.exists
    (function 
       | TypeArgument typ -> (has_successor_state typ)
    )
    args


(*******************************************************************
 *
 * curry an Adt constructor type to convert it for Isabelle 
 *
 *)


let curry_constructor_type typ = match typ with
  | Function (Product(args), res) -> 
      List.fold_right 
	(fun arg accu -> Function(arg, accu))
	args
	res

  | Function _
  | Groundtype _ 
  | BoundTypeVariable _ 
  | Self 
  | Bool 
  | Product _
  | Class _ 
  | Adt _
  | Carrier
  | Predtype _
  | Record _
  | IFace _
  | Array _ 	      
  | FreeTypeVariable _
  | TypeConstant _
  | SmartFunction _ 
    -> typ


(*******************************************************************
 *
 * replace SmartFunction with iterated Function's or Products,
 * depending on Global.output_mode
 * 
 * needed for normalizing types during typechecking
 *)


let rec unsmart_type typ = match typ with
  | Groundtype(idrec, args) 	-> Groundtype(idrec, unsmart_args args)
  | BoundTypeVariable _ 	-> typ
  | FreeTypeVariable _		-> typ
  | Self			-> typ
  | Carrier 			-> typ
  | Bool 			-> typ
  | Function( dom, codom) 	-> 
      Function(unsmart_type dom, unsmart_type codom)
  | Product( types ) 		-> Product(List.map unsmart_type types)
  | Class(cl, args )		-> Class(cl, unsmart_args args)
  | Adt(adt, flag, args )	-> Adt(adt, flag, unsmart_args args)
		(* do records as an exception *)
  | Record label_list 		->
      Record( List.map (fun (name,typ) -> (name, unsmart_type typ)) label_list)
					(* not in ccsl input types *)
  | SmartFunction( doml, codom) ->
      let ndoml = List.map unsmart_type doml in
      let ncodom = unsmart_type codom 
      in
	(match ndoml with
	   | [] -> assert false
	   | [t] -> Function(t, ncodom)
	   | _ -> 
	       if !Global.output_mode = Isa_mode 
	       then
		 List.fold_right
		   (fun t accu -> Function(t, accu))
		   ndoml
		   ncodom
	       else
		 Function(Product(ndoml), ncodom)
	)

  | TypeConstant _
  | IFace _
  | Array _
  | Predtype _
    -> assert(false)

and unsmart_args args = 
  List.map (function TypeArgument t -> TypeArgument( unsmart_type t)) args

    
(*******************************************************************
 *
 * utility functions for ground types (more below)
 *
 *)

let get_ground_type_parameters idrec = match idrec.id_origin with
  | CCSL_GroundTypeDef si
  | CCSL_GroundTypeDecl si -> si#get_parameters @ idrec.id_parameters
  | _ -> assert(false)


let get_definition_parameters def =
  def.defined_method#hosting_class#get_parameters @
  def.defined_method#get_local_parameters


let is_type_def idrec = match idrec.id_origin with
  | CCSL_GroundTypeDef _ -> true
  | CCSL_GroundTypeDecl _ -> false
  | _ -> assert(false)

let sig_of_typedef idrec = 
  match idrec.id_origin with
    | CCSL_GroundTypeDef si -> si
    | CCSL_GroundTypeDecl si -> si
    | _ -> assert(false)


(************************************************
 *
 * extract type parameters of a member, taking local parameters 
 * into account
 *)

let get_member_parameters m =
  m#hosting_class#get_parameters @ m#get_local_parameters




(***********************************************************************
 *
 * iter_components walks through the type and applies action 
 * to all components
 * the intention is, that action is applied exactly in the order in 
 * which the importings must be generated
 *
 *)

let rec iter_components variance action typ = match typ with
  | Groundtype(id, args)-> 
      iter_component_arglist variance action args 
	(get_ground_type_parameters id);
      action variance typ
  | BoundTypeVariable id 	-> ()
  | Self 		-> ()
  | Carrier 		-> ()
  | Bool 		-> ()
  | Function(dom,codom) -> 
      iter_components (variance_subst variance Neg) action dom;
      iter_components variance action codom
  | Product(type_list) 	-> 
      List.iter (iter_components variance action) type_list
  | Class(cl, args) 	-> 
      iter_component_arglist variance action args cl#get_parameters;
      action variance typ
  | Adt(adt,flag,args) 	-> 
      iter_component_arglist variance action args adt#get_parameters;
      action variance typ
				(* not in ccsl input, but occurs for reduce *)
  | FreeTypeVariable _  -> ()

					(* not in ccsl input types *)
  | TypeConstant _
  | Predtype _
  | Record _
  | IFace _
  | Array _ 	      
  | SmartFunction _ -> assert(false)


and iter_component_arglist variance action arguments parameters = 
  List.iter2 (function TypeArgument(typ) -> 
		(function TypeParameter id ->
		   iter_components (variance_subst variance 
				      id.id_variance)
		     action typ))
    arguments parameters


let do_add_component iface v = function
  | Class(cl,args) -> iface#add_component (v, cl, args)
  | Adt(adt,_,args) -> iface#add_component (v, adt, args)
  | Groundtype(id,args) ->
      let osi = sig_of_typedef id 
      in
	if osi#get_name <> iface#get_name
	then
	  iface#add_component (v, osi, args)
  | _ -> assert(false)


let iter_components_add_component iface typ =
  iter_components Pos (do_add_component iface) typ

let iter_component_arglist_add_component iface arguments parameters =
  iter_component_arglist Pos (do_add_component iface) arguments parameters


(***********************************************************************
 *
 * equality on ccsl output types
 *
 *)

(* to avoid mistakes, I don't use polymorphic structural equality 
 * in the following, instead, I redefine:
 *)

let eq_string (s1 : string) (s2 : string) = s1 = s2

let eq_int (s1 : int) (s2 : int) = s1 = s2

let eq_option (eq_fun) o1 o2 = match o1,o2 with
  | Some x1, Some x2 -> eq_fun x1 x2
  | None, _
  | _, None -> false

let rec eq_ccsl_types t1 t2 = t1 == t2 or match t1,t2 with
   | Groundtype(id1,args1), Groundtype(id2,args2) -> 
       eq_string id1.id_token.token_name id2.id_token.token_name
       && (List.length(args1) = List.length(args2))
       && (List.for_all2 eq_ccsl_args args1 args2)
   | TypeConstant(name1,flag1,args1), TypeConstant(name2,flag2,args2) ->
       (* flags are ignored *)
       (eq_string name1 name2)
       && (List.length(args1) = List.length(args2))
       && (List.for_all2 eq_ccsl_args args1 args2)
   | BoundTypeVariable id1, BoundTypeVariable id2 -> 
       eq_string id1.id_token.token_name id2.id_token.token_name
   | FreeTypeVariable t1, FreeTypeVariable t2 ->
       Type_variable.eq t1 t2
   | Self,Self -> true
   | Carrier, Carrier -> true
   | Bool, Bool -> true
   | Function(dom1,codom1),Function(dom2,codom2) -> 
       (eq_ccsl_types dom1 dom2) && (eq_ccsl_types codom1 codom2)
   | SmartFunction(doml1,codom1),SmartFunction(doml2,codom2) -> 
       (List.length doml1 = List.length doml2)
       && (List.for_all2 (fun t1 t2 -> eq_ccsl_types t1 t2) doml1 doml2)
       && (eq_ccsl_types codom1 codom2)
   | Product(type_list1),Product(type_list2) ->
       (List.length type_list1 = List.length type_list2)
       && (List.for_all2 eq_ccsl_types type_list1 type_list2)
   | Class(cl1, args1),Class(cl2, args2) ->
       (eq_string cl1#get_name cl2#get_name)
       && (List.length args1 = List.length args2)
       && (List.for_all2 eq_ccsl_args args1 args2)
   | Adt(adt1,flag1,args1),Adt(adt2,flag2,args2) ->
       (eq_string adt1#get_name adt2#get_name)
       (* ignoring flags *)
       && (List.length args1 = List.length args2)
       && (List.for_all2 eq_ccsl_args args1 args2)
   | Predtype(formula1),Predtype(formula2) ->
       eq_ccsl_formulas formula1 formula2
   | Record(label_list1),Record(label_list2) ->
       (List.length label_list1 = List.length label_list2)
       && (List.for_all2 
	     (fun (label1,typ1) (label2,typ2) -> 
		(eq_string label1 label2)
		&& (eq_ccsl_types typ1 typ2))
	     label_list1 label_list2)
   | IFace(cl1, flag1, args1),IFace(cl2, flag2, args2) ->
       (* ignoring flags *)
       (eq_string cl1#get_name cl2#get_name)
       && (List.length args1 = List.length args2)
       && (List.for_all2 eq_ccsl_args args1 args2)

     (* not allowed constructors in t1 *)
  | Array _,_



     (* not allowed constructors in t2 *)
  | _, Array _          -> assert(false)



     (* cases where the constructors mismatch *)
  | Groundtype _ ,_
  | TypeConstant _ ,_
  | BoundTypeVariable _ ,_
  | FreeTypeVariable _,_
  | Self ,_
  | Carrier ,_
  | Bool, _
  | Function _ ,_
  | SmartFunction _,_
  | Product _ ,_
  | Class _ ,_
  | Adt _ ,_
  | Predtype _ ,_
  | Record _ ,_
  | IFace _ ,_

  | _, Groundtype _
  | _, TypeConstant _
  | _, BoundTypeVariable _
  | _, FreeTypeVariable _
  | _, Self
  | _, Carrier 
  | _, Bool
  | _, Function _
  | _, SmartFunction _
  | _, Product _
  | _, Class _
  | _, Adt _
  | _, Predtype _
  | _, Record _
  | _, IFace _ -> false

and eq_ccsl_typ_option o1 o2 = o1 == o2 or eq_option eq_ccsl_types o1 o2

and eq_ccsl_formulas f1 f2 = f1 == f2 or match (f1,f2) with 
  | True,True -> true
  | False,False -> true
  | Not f1,Not f2 -> eq_ccsl_formulas f1 f2
  | And f_list1,And f_list2 ->
      (List.length f_list1 = List.length f_list2)
      && (List.for_all2 eq_ccsl_formulas f_list1 f_list2)
  | Or f_list1,Or f_list2 ->
      (List.length f_list1 = List.length f_list2)
      && (List.for_all2 eq_ccsl_formulas f_list1 f_list2)
  | Implies(prem1,concl1), Implies(prem2,concl2) ->
      (eq_ccsl_formulas prem1 prem2)
      && (eq_ccsl_formulas concl1 concl2)
  | Iff(prem1,concl1), Implies(prem2,concl2) ->
      (eq_ccsl_formulas prem1 prem2)
      && (eq_ccsl_formulas concl1 concl2)
  | Equal(ex_a1,ex_b1),Equal(ex_a2,ex_b2) ->
      (eq_ccsl_expressions ex_a1 ex_a2)
      && (eq_ccsl_expressions ex_b1 ex_b2)
(*   | LessOrEqual(ex_a1,ex_b1),LessOrEqual(ex_a2,ex_b2) ->
 * 	 (eq_ccsl_expressions ex_a1 ex_a2)
 * 	 && (eq_ccsl_expressions ex_b1 ex_b2)
 *)
  | Forall(quant_list1, f1),Forall(quant_list2, f2) ->
      (List.length quant_list1 = List.length quant_list2)
      && (List.for_all2 
	    (fun (name1,typ1) (name2,typ2) ->
	       (eq_string name1 name2)
	       && eq_ccsl_types typ1 typ2)
	    quant_list1 quant_list2)
      && eq_ccsl_formulas f1 f2      
  | Exists(quant_list1, f1),Exists(quant_list2, f2) ->
      (List.length quant_list1 = List.length quant_list2)
      && (List.for_all2 
	    (fun (name1,typ1) (name2,typ2) ->
	       (eq_string name1 name2)
	       && eq_ccsl_types typ1 typ2)
	    quant_list1 quant_list2)
      && eq_ccsl_formulas f1 f2      
  | ConstantPredicate name1,ConstantPredicate name2 ->
      eq_string name1 name2
  | Formula ex1,Formula ex2 ->
      eq_ccsl_expressions ex1 ex2
  | MetaImplies(prem1,concl1),MetaImplies(prem2,concl2) ->
      (eq_ccsl_formulas prem1 prem2)
      && (eq_ccsl_formulas concl1 concl2)
  | MetaForall(quant_list1, f1),MetaForall(quant_list2, f2) ->
      (List.length quant_list1 = List.length quant_list2)
      && (List.for_all2 
	    (fun (name1,typ1) (name2,typ2) ->
	       (eq_string name1 name2)
	       && eq_ccsl_types typ1 typ2)
	    quant_list1 quant_list2)
      && eq_ccsl_formulas f1 f2      
  | Bisim(typ1,exl1,exr1), Bisim(typ2,exl2,exr2) -> 
      (eq_ccsl_types typ1 typ2)
      && (eq_ccsl_expressions exl1 exl2)
      && (eq_ccsl_expressions exr1 exr2)
  | Obseq(t1,exl1,exr1), Obseq(t2,exl2,exr2) ->
      (match (t1,t2) with
	   None,None -> true
	 | Some(n_1,t_1), Some(n_2,t_2) -> 
	     (eq_string n_1 n_2) && (eq_ccsl_types t_1 t_2) 
	 | Some _, _
	 | None, _ -> false
      )
      && (eq_ccsl_expressions exl1 exl2)
      && (eq_ccsl_expressions exr1 exr2)

  | FormLoc(f1,l), f2 -> eq_ccsl_formulas f1 f2
  | f1, FormLoc(f2,l) -> eq_ccsl_formulas f1 f2

	(* cases where constructors do not match *)
  | True ,_
  | False ,_
  | Not _ ,_
  | And _ ,_
  | Or _ ,_
  | Implies _ ,_
  | Iff _ ,_
  | Equal _ ,_
(*   | LessOrEqual _ ,_
 *)
  | Forall _ ,_
  | Exists _ ,_
  | ConstantPredicate _ ,_
  | Formula _ ,_
  | MetaImplies _ ,_
  | MetaForall _ ,_
  | Bisim _, _
  | Obseq _, _

  | _, True
  | _, False
  | _, Not _
  | _, And _
  | _, Or _
  | _, Implies _
  | _, Iff _
  | _, Equal _
(*   | _, LessOrEqual _
 *)
  | _, Forall _
  | _, Exists _
  | _, ConstantPredicate _
  | _, Formula _
  | _, MetaImplies _
  | _, MetaForall _ 
  | _, Bisim _
  | _, Obseq _
      -> false

and eq_ccsl_basic_expressions be1 be2 = be1 == be2 or match be1,be2 with 
	(* don't compare optional classes, class names are 
	 * unique and they are compared via the hosting classes of the members
	 *)
  | Member(typeopt1,c1), Member(typeopt2,c2) ->
      let c1' = resolution_of c1 in
      let c2' = resolution_of c2 in
	eq_string (c1'#get_name) (c2'#get_name)
	&& eq_string (c1'#hosting_class#get_name) (c2'#hosting_class#get_name) 

(* Hendrik: Don't know if the following is inconsistent, because
 *  different variables might have the same name ?
 *)
  | TermVar idc1, TermVar idc2 ->
      let id1 = resolution_of idc1 in
      let id2 = resolution_of idc2 in
	eq_string id1.id_token.token_name id2.id_token.token_name

	(* cases where constructors do not match *)
  | Member _,_
  | TermVar _,_
  | _, Member _
  | _, Member _
      -> false


and eq_ccsl_expressions e1 e2 =   e1 == e2 or match e1,e2 with
  | BasicExpr(be1), BasicExpr(be2) ->
      eq_ccsl_basic_expressions be1 be2
  | Term(name1,flag1,args1), Term(name2,flag2,args2) ->
      (* ignoring flags *)
      (eq_string name1 name2)
      && (List.length args1 = List.length args2)
      && (List.for_all2 eq_ccsl_args args1 args2)
  | TypedTerm(ex1,typ1), TypedTerm(ex2,typ2) ->
      eq_ccsl_expressions ex1 ex2 
      && eq_ccsl_types typ1 typ2
  | TypeAnnotation(ex1,typ1), TypeAnnotation(ex2,typ2) ->
      eq_ccsl_expressions ex1 ex2 
      && eq_ccsl_types typ1 typ2
  | QualifiedTerm(th1, flag1, args1, name1), 
    QualifiedTerm(th2,flag2,args2,name2) ->
      (*ignoring flags *)
      (eq_string th1 th2)
      && (List.length args1 = List.length args2)
      && (List.for_all2 eq_ccsl_args args1 args2)
      && (eq_string name1 name2)

	(* don't compare optional classes, class names are 
	 * unique and they are compared via the hosting classes of the members
	 *)
  | MethodSelection(obj1, cl1, m1), MethodSelection(obj2, cl2, m2) ->
      let m1' = resolution_of m1 in
      let m2' = resolution_of m2 in
      	eq_ccsl_expressions obj1 obj2 
      	&& eq_string (m1'#get_name) (m1'#get_name)
	&& eq_string (m1'#hosting_class#get_name) (m2'#hosting_class#get_name) 
  | Tuple ex_list1, Tuple ex_list2 ->
      (List.length ex_list1 = List.length ex_list2)
      && (List.for_all2 eq_ccsl_expressions ex_list1 ex_list2)
  | Projection(i1,n1), Projection(i2,n2) ->
      assert(n1 = n2);
      (eq_int i1 i2)
  | RecordTuple label_list1, RecordTuple label_list2 ->
      (List.length label_list1 = List.length label_list2)
      && (List.for_all2 (fun (name1,ex1) (name2,ex2) ->
			   (eq_string name1 name2)
			   && eq_ccsl_expressions ex1 ex2)
	    label_list1 label_list2)
  | RecordSelection(name1,ex1), RecordSelection(name2,ex2) ->
      (eq_string name1 name2)
      && (eq_ccsl_expressions ex1 ex2)
  | RecordUpdate(ex1, update_list1),RecordUpdate(ex2, update_list2) ->
      (eq_ccsl_expressions ex1 ex2)
      && (List.length update_list2 = List.length update_list2)
      && (List.for_all2 (fun (name1,ex1) (name2,ex2) ->
			   (eq_string name1 name2)
			   && eq_ccsl_expressions ex1 ex2)
	    update_list1 update_list2)
  | List ex_list1, List ex_list2 ->
      (List.length ex_list1 = List.length ex_list2)
      && (List.for_all2 eq_ccsl_expressions ex_list1 ex_list2)
  | Abstraction(decl_list1, ex1), Abstraction(decl_list2, ex2) ->
      (List.length decl_list1 = List.length decl_list2)
      && (List.for_all2 (fun (name1,typ1) (name2,typ2) ->
			   (eq_string name1 name2)
			   && eq_ccsl_types typ1 typ2)
	    decl_list2 decl_list2)
      && (eq_ccsl_expressions ex1 ex2)
  | SmartAbstraction(decl_list1, ex1), SmartAbstraction(decl_list2, ex2) ->
      (List.length decl_list1 = List.length decl_list2)
      && (List.for_all2 (fun (name1,typ1) (name2,typ2) ->
			   (eq_string name1 name2)
			   && eq_ccsl_types typ1 typ2)
	    decl_list2 decl_list2)
      && (eq_ccsl_expressions ex1 ex2)
  | Application(f1, ex1),Application(f2, ex2) ->
      (eq_ccsl_expressions f1 f2)
      && (eq_ccsl_expressions ex1 ex2)
  | InfixApplication(ex1a,iif1,m1,ex1b), InfixApplication(ex2a,iif2,m2,ex2b) ->
      let m1' = resolution_of m1 in
      let m2' = resolution_of m2 in
      	(eq_ccsl_expressions ex1a ex2a)
      	&& (eq_ccsl_expressions ex1b ex2b)
	&& (eq_string m1'#get_name m2'#get_name)
	&& eq_string (m1'#hosting_class#get_name) 
	  (m2'#hosting_class#get_name) 
  | SmartApplication(f1, exl1),SmartApplication(f2, exl2) ->
      (eq_ccsl_expressions f1 f2)
      && (List.length exl1 = List.length exl2)
      && (List.for_all2 (fun ex1 ex2 ->
			   eq_ccsl_expressions ex1 ex2)
	    exl1 exl2)
  | FunUpdate(ex1, update_list1), FunUpdate(ex2, update_list2) ->
      (eq_ccsl_expressions ex1 ex2)
      && (List.length update_list1 = List.length update_list2)
      && (List.for_all2 (fun (ex_a1,ex_b1) (ex_a2, ex_b2) ->
			   (eq_ccsl_expressions ex_a1 ex_a2)
			   && (eq_ccsl_expressions ex_b1 ex_b2))
	    update_list2 update_list2)
  | Let(define_list1, ex1), Let(define_list2, ex2) ->
      (List.length define_list1 = List.length define_list2)
      && (List.for_all2 (fun (id1,typ1,ex1) (id2,typ2,ex2) ->
(* Hendrik: Don't know if the following is inconsistent, because
 *  different variables might have the same name ?
 *)
			   (eq_string id1.id_token.token_name
			      id2.id_token.token_name)
			   && (eq_ccsl_typ_option typ1 typ2)
			   && (eq_ccsl_expressions ex1 ex2))
	    define_list2 define_list2)
      && (eq_ccsl_expressions ex1 ex2)
  | If( cases_list1,ex1), If( cases_list2,ex2) ->
      (List.length cases_list1 = List.length cases_list2)
      && (List.for_all2 (fun (form1,ex1) (form2,ex2) ->
			   (eq_ccsl_formulas form1 form2)
			   && (eq_ccsl_expressions ex1 ex2))
	    cases_list2 cases_list2)
      && (eq_ccsl_expressions ex1 ex2)
  | Case( ex1, match_list1), Case( ex2, match_list2) ->
      (eq_ccsl_expressions ex1 ex2)
      && (List.length match_list1 = List.length match_list2)
      && (List.for_all2 (fun (ex_a1,ex_b1) (ex_a2, ex_b2) ->
			   (eq_ccsl_expressions ex_a1 ex_a2)
			   && (eq_ccsl_expressions ex_b1 ex_b2))
	    match_list1 match_list2)
  | CCSL_Case( ex1, match_list1), CCSL_Case( ex2, match_list2) ->
      (eq_ccsl_expressions ex1 ex2)
      && (List.length match_list1 = List.length match_list2)
      && (List.for_all2 
	    (fun (mc1,var_list1,ex_b1) (mc2,var_list2,ex_b2) ->
	       let m1 = resolution_of mc1 in
	       let m2 = resolution_of mc2 in
		 eq_string (m1#get_name) (m2#get_name)
		 && eq_string (m1#hosting_class#get_name) 
		   (m2#hosting_class#get_name) 
		 && (List.length var_list1 = List.length var_list2)
		 && (List.for_all2
		       (fun id1 id2 ->
			  eq_string id1.id_token.token_name 
			    id2.id_token.token_name)
		       var_list1 var_list2)
		 && (eq_ccsl_expressions ex_b1 ex_b2))
	    match_list1 match_list2)
  | Box(typ1,pred1,tlist1),
    Box(typ2,pred2,tlist2) ->
      (eq_ccsl_types typ1 typ2)
      && (eq_ccsl_expressions pred1 pred2)
      && (List.for_all2 (fun t1 t2  ->
			   (eq_string t1.token_name t2.token_name)) 
	    tlist1 tlist2)
  | Diamond(typ1,pred1,tlist1),
    Diamond(typ2,pred2,tlist2) ->
      (eq_ccsl_types typ1 typ2)
      && (eq_ccsl_expressions pred1 pred2)
      && (List.for_all2 (fun t1 t2 -> (eq_string t1.token_name t2.token_name)) 
	    tlist1 tlist2)
  | Every(typ1, form_list1), Every(typ2, form_list2) ->
      (eq_ccsl_types typ1 typ2) &&
			(* equal types imply equal parameter lists 
			 * imply equal length of formula lists 
			 *)
      (assert((List.length form_list1) = (List.length form_list2));
       List.for_all2 eq_ccsl_expressions form_list1 form_list2)
  | RelEvery(typ1, form_list1), RelEvery(typ2, form_list2) ->
      (eq_ccsl_types typ1 typ2) &&
			(* equal types imply equal parameter lists 
			 * imply equal length of formula lists 
			 *)
      (assert((List.length form_list1) = (List.length form_list2));
       List.for_all2 eq_ccsl_expressions form_list1 form_list2)
  | Map(typ1, expr_list1), Map(typ2, expr_list2) ->
      (eq_ccsl_types typ1 typ2) &&
			(* equal types imply equal parameter lists 
			 * imply equal length of formula lists 
			 *)
      (assert((List.length expr_list1) = (List.length expr_list2));
       List.for_all2 eq_ccsl_expressions expr_list1 expr_list2)      
  | Expression(formula1), Expression(formula2) ->
      eq_ccsl_formulas formula1 formula2
  | Comprehension(name1, typ1, fo1),Comprehension(name2, typ2, fo2) ->
      (eq_string name1 name2)
      && (eq_ccsl_types typ1 typ2)
      && (eq_ccsl_formulas fo1 fo2)

  | ExprLoc(ex1,l1), ExprLoc(ex2,l2) -> eq_ccsl_expressions ex1 ex2
  | ExprLoc(ex1,l), ex2 -> eq_ccsl_expressions ex1 ex2
  | ex1, ExprLoc(ex2,l) -> eq_ccsl_expressions ex1 ex2

	(* and now the cases where the constructors differ *)
  | BasicExpr _,_
  | Term _ ,_
  | TypedTerm _,_
  | TypeAnnotation _,_
  | QualifiedTerm _ ,_
  | MethodSelection _,_
  | Tuple _ ,_
  | Projection _ ,_
  | RecordTuple _ ,_
  | RecordSelection _ ,_
  | RecordUpdate _ ,_
  | List _ ,_
  | Abstraction _ ,_
  | SmartAbstraction _ ,_
  | Application _ ,_
  | InfixApplication _,_
  | FunUpdate _ ,_
  | Let _ ,_
  | If _ ,_
  | Case _ ,_
  | CCSL_Case _ ,_
  | Box _,_
  | Diamond _,_
  | Every _, _
  | RelEvery _, _
  | Map _, _
  | Expression _ ,_
  | Comprehension _ ,_

  | _, BasicExpr _
  | _, Term _
  | _, TypeAnnotation _
  | _, TypedTerm _
  | _, QualifiedTerm _
  | _, MethodSelection _
  | _, Tuple _
  | _, Projection _
  | _, RecordTuple _
  | _, RecordSelection _
  | _, RecordUpdate _
  | _, List _
  | _, Abstraction _
  | _, SmartAbstraction _
  | _, Application _
  | _, InfixApplication _
  | _, FunUpdate _
  | _, Let _
  | _, If _
  | _, Case _
  | _, CCSL_Case _
  | _, Box _
  | _, Diamond _
  | _, Every _
  | _, RelEvery _
  | _, Map _
  | _, Expression _
  | _, Comprehension _ 
      -> false

and eq_ccsl_expressions_option o = eq_option eq_ccsl_expressions o

and eq_ccsl_args a1 a2 = match a1,a2 with

  | TypeArgument typ1, TypeArgument typ2 ->
      eq_ccsl_types typ1 typ2

let eq_ccsl_params pl1 pl2 = 
  try
    List.for_all2 
      (fun p1 p2 -> match p1,p2 with
	 | TypeParameter id1, TypeParameter id2 -> 
	     eq_string id1.id_token.token_name id2.id_token.token_name)
      pl1 pl2
  with
    | Invalid_argument _ -> false

(***********************************************************************
 ***********************************************************************
 *
 * substitution
 *
 * the substitution is given as two assoc lists, 
 * the first gives a substitution on types, 
 * with elements top_types * top_types 
 * the second gives a substitution on values.
 * 
 * Substitution on values is important because of value
 * parameters (and arguments). Currently, only substitution on values 
 * works only on very restricted set of expression (Term, Termvar)
 *
 * the main machinery is in logic_util, here are only ccsl 
 * specific instanciations
 *)


(************
 * specialize even further to use eq_ccsl_types
 * 
 * val ccsl_substitute_types :
 *   (('cl, 'mem) top_pre_types * ('cl, 'mem) top_pre_types) list ->
 *     ('cl, 'mem) top_pre_types -> 
 * 	 ('cl, 'mem) top_pre_types 
 *)

let ccsl_substitute_types =
  substitute_types_only eq_ccsl_types

   (* utility function for argument lists *)
let ccsl_substitute_arguments typ_subst 
  = substitute_arguments (eq_ccsl_types, (==)) (typ_subst,[])



(***********************************************************************
 * 
 * Formula uncurrying
 * 
 * !!! for proper functioning formulas must not contain any !!!
 * !!! FormLoc, ExprLoc                                     !!!
 *)

(* merge_ands uses associativity of and to make one top_level
 * And from both arguments. 
 * It is used as utility function in uncurry_formula
 *)

let merge_ands f g = match (f,g) with
  | (And(fl), And(gl)) -> And(fl @ gl)
  | (f, And(gl)) -> And( f :: gl)
  | (And(fl), g) -> And(fl @ [g])
  | (f,g) -> And([f;g])

(* uncurry_formula performs the rewrite 
 * A IMPLIES (B IMPLIES C) ===> (A AND B) IMPLIES C
 *)

let rec uncurry_formula formula = 
  match formula with
    | FormLoc (f,l) -> formula
    | True -> formula
    | False -> formula
    | Not f -> formula
    | And f_list -> formula
    | Or f_list -> formula
    | Implies(a,b_impl_c) -> 
	(match uncurry_formula b_impl_c with
	  | Implies(b, c) -> Implies(merge_ands a b, c)
	  | concl -> formula
	)
    | Iff _ -> formula
    | Equal(ex_a,ex_b) -> formula
(*     | LessOrEqual(ex_a, ex_b) -> formula
 *)
    | Forall(quant_list, f) -> formula
    | Exists(quant_list, f) -> formula
    | ConstantPredicate name -> formula
    | Formula ex -> formula
	  (* don't know, if one can move ordinary quantifiers over
	   * those meta level things. Moving not, is at least correct 
           *)
    | MetaImplies(prem,concl) -> formula
    | MetaForall(quant_list, f) -> formula
    | Bisim(typ,ex1,ex2) -> formula
    | Obseq(t,ex1,ex2) -> formula

(* curry_formula performs the rewrite 
 * (A AND B) IMPLIES C  ===>  A IMPLIES (B IMPLIES C)
 *)

let rec curry_formula formula = 
  match formula with
    | FormLoc _ -> formula
    | True -> formula
    | False -> formula
    | Not f -> formula
    | And f_list -> formula
    | Or f_list -> formula
    | Implies(And(al), b) -> 
	List.fold_right (fun a res -> Implies(a, res))
	  al b
    | Implies _ -> formula
    | Iff _ -> formula
    | Equal(ex_a,ex_b) -> formula
    | Forall(quant_list, f) -> formula
    | Exists(quant_list, f) -> formula
    | ConstantPredicate name -> formula
    | Formula ex -> formula
	  (* don't know, if one can move ordinary quantifiers over
	   * those meta level things. Moving not, is at least correct 
           *)
    | MetaImplies(prem,concl) -> formula
    | MetaForall(quant_list, f) -> formula
    | Bisim(typ,ex1,ex2) -> formula
    | Obseq(t,ex1,ex2) -> formula

(* meta_implies converts Implies into MetaImplies, this is better for
 * Isabelle resolution
 *)

let rec meta_implies formula = 
  match formula with
    | FormLoc _ -> formula
    | True -> formula
    | False -> formula
    | Not f -> formula
    | And f_list -> formula
    | Or f_list -> formula
    | Implies(a, b) -> MetaImplies(a, meta_implies b)
    | Iff(a,b) -> And([MetaImplies(a, meta_implies b); 
		       MetaImplies(b, meta_implies a)])
    | Equal(ex_a,ex_b) -> formula
    | Forall(quant_list, f) -> formula
    | Exists(quant_list, f) -> formula
    | ConstantPredicate name -> formula
    | Formula ex -> formula
    | MetaImplies(prem,concl) -> formula
    | MetaForall(quant_list, f) -> formula
    | Bisim(typ,ex1,ex2) -> formula
    | Obseq(t,ex1,ex2) -> formula

(***********************************************************************
 *
 * quantifier utilities
 * 
 * Attention!! These functions do not check for the free variable 
 * condition!. They do no alpha conversion. It is the responsibility
 * of the user, to ensure that no free variables are bound and 
 * no variable is bound twice (using flat name spaces for instance)
 *
 * val pull_forall : ('cl, 'mem) top_pre_formula -> ('cl, 'mem) top_pre_formula
 * 
 * val pull_exits : ('cl, 'mem) top_pre_formula -> ('cl, 'mem) top_pre_formula
 * 
 * Both functions are implemented by pull_quant_formula, which uses 
 * a boolean argument to distigish wether it runs in forall or exists 
 * mode.
 *)

    (* we do forall and exits at once, 
     * argument forall is true if we do forall 
     * and false, if we do exists
     *)
let rec pull_quant_formula forall formula = 
  match formula with
    | FormLoc _ -> [], formula
    | True -> [], formula
    | False -> [], formula
    | Not f -> 
	let (q,f') = pull_quant_formula (not forall) f in
	  if q = [] then [],formula
	  else (q, Not f')
    | And f_list -> 
	let (q,f_list') = List.fold_right  
			    (fun f (qs,fs) -> 
			       let (nq,nf) = pull_quant_formula forall f
			       in (qs @ nq, nf :: fs))
			    f_list ([],[]) in
	  if q = [] then ([], formula)
	  else (q, And(f_list'))
    | Or f_list -> 
	let (q,f_list') = List.fold_right 
			    (fun f (qs,fs) -> 
			       let (nq,nf) = pull_quant_formula forall f
			       in (qs @ nq, nf :: fs))
			    f_list ([],[]) in
	  if q = [] then ([], formula)
	  else (q, Or(f_list'))
    | Implies(prem,concl) -> 
	let (qp,fp) = pull_quant_formula (not forall) prem in
	let (qc,fc) = pull_quant_formula forall concl in
	  if (qp = []) && (qc = []) then [],formula
	  else (qp @ qc, Implies(fp,fc))
	(* Iff is really hard. The only thing, one could do is:  
	 * pull_quant_formula forall And([Implies(p1,p2);Implies(p2,p1)]),
	 *)
    | Iff(p1,p2) -> ([], formula)
    | Equal(ex_a,ex_b) -> ([], formula)
    | Forall(quant_list, f) -> 
	if forall 
	then 
	  let (q,f') = pull_quant_formula forall f in
	    if q = [] then 
	      (quant_list, f)
	    else 
	      (quant_list @ q, f')
	else ([], formula)
    | Exists(quant_list, f) -> 
	if forall 
	then ([], formula)
	else
	  let (q,f') = pull_quant_formula forall f in
	    if q = [] then 
	      (quant_list, f)
	    else 
	      (quant_list @ q, f')
    | ConstantPredicate name -> ([], formula)
    | Formula ex -> ([], formula)
	  (* don't know, if one can move ordinary quantifiers over
	   * those meta level things. Moving not, is at least correct 
           *)
    | MetaImplies(prem,concl) -> ([], formula)
    | MetaForall(quant_list, f) -> ([], formula)
    | Bisim(typ,ex1,ex2) -> ([],formula)
    | Obseq(t,ex1,ex2) -> ([],formula)

let pull_forall f = 
  let (q,f') = pull_quant_formula true f in
    if q = [] then f
    else Forall(q, f')

let pull_exits f = 
  let (q,f') = pull_quant_formula false f in
    if q = [] then f
    else Exists(q, f')


(***********************************************************************
 *
 * make_rewrite_form takes a formula and tries to transform it into 
 * a PVS rewrite formula
 *
 * !!! for proper functioning formulas must not contain any !!!
 * !!! FormLoc, ExprLoc                                     !!!
 *)


let make_pvs_rewrite_form f =
  if !Global.optimize_expressions 
  then
    match pull_forall f with
      | Forall(q,f') -> Forall(q, uncurry_formula f')
      | f' -> uncurry_formula f'
  else f

let make_isa_rewrite_form f =
  if !Global.optimize_expressions 
  then
    match pull_forall f with
      | Forall(q,f') -> meta_implies( curry_formula f')
      | f' -> meta_implies( curry_formula f')
  else f


(***********************************************************************
 *
 * one small utility: types_from_parameters is needed soon
 *
 *)

let types_from_parameters params =
  List.map (function
	      | TypeParameter id -> BoundTypeVariable(id)
	   )
    params

let types_from_arguments args =
  List.map (function
	      | TypeArgument t -> t
	   )
    args


let arguments_from_parameters params = 
  List.map (function
	      | TypeParameter id -> TypeArgument(BoundTypeVariable(id))
	   )
    params


let make_substitution param_list arg_list =
  List.combine
    (types_from_parameters param_list)
    (types_from_arguments arg_list)


let make_substitution_param_param param_list_from param_list_to =
  List.combine
    (types_from_parameters param_list_from)
    (types_from_parameters param_list_to)


(*******************************************************************
 *
 * more utility functions for ground types
 *
 *)

let rec expand_type_def eq_types typ = match typ with
  | Groundtype(id, args) ->
      if is_type_def id 
      then
	expand_type_def eq_types
	  (substitute_types_only eq_types 
	     (make_substitution (get_ground_type_parameters id) args)
	     id.id_type
	  )
      else
	typ
				(* expand the second level for functions *)
  | Function(dom, codom) ->
      let ndom = expand_type_def eq_types dom in
      let ncodom = expand_type_def eq_types codom 
      in
	if dom == ndom && codom == ncodom 
	then
	  typ
	else
	  Function(ndom, ncodom)
					(* do nothing in all other cases *)
  | BoundTypeVariable _
  | Self
  | Carrier
  | Bool
  | Product _
  | Class _ 
  | Adt _
  | Record _
  | TypeConstant _
  | IFace _
  | FreeTypeVariable _
  | Array _
  | Predtype _
  | SmartFunction _         -> typ


(***********************************************************************
 *
 * export method / constructor types
 * 
 * export_member class/adt member 
 * substitutes each BoundType with a uniqe free type variable,
 * self/carrier is substituted with the adt/class type
 *
 * val export_member : 
 *   ccsl_iface_type -> ccsl_input_types -> 
 *     ccsl_inst_iface_type * ccsl_input_types
 * 
 * val export_member_with_args : 
 *   ccsl_iface_type -> ccsl_argument_type list -> ccsl_input_types -> 
 *     ccsl_inst_iface_type * ccsl_input_types
 * 
 *)

let export_member_with_args iface member args memtype =
  let _ = assert(List.length 
		   (get_member_parameters member) = List.length args)
  in
  let types = types_from_arguments args in
  let bounds = types_from_parameters (get_member_parameters member) in
  let iiface = 
    match iface#get_kind with
      | Spec_adt -> Adt(iface, Always, args)
      | Spec_class -> Class(iface, args)
      | Spec_sig -> Product []		(* this is never used *)
      | Spec_Spec -> assert(false)
  in
  let subst = 
    (match iface#get_kind with
       | Spec_adt -> [ (Carrier, iiface) ]
       | Spec_class -> [ (Self, iiface) ]
       | Spec_sig -> []
       | Spec_Spec -> assert(false)
    )
    @ List.combine bounds types 
  in
    (InstIface(iface,args,None),
     ccsl_substitute_types subst memtype)


let export_member iface member memtyp =
  let args = List.map 
	       (fun _ -> TypeArgument(
		  FreeTypeVariable(Type_variable.fresh())))
	       (get_member_parameters member)
  in 
    export_member_with_args iface member args memtyp
     
let export_member_local iface member memtyp = 
  let global_args = arguments_from_parameters iface#get_parameters in
  let localparams = member#get_local_parameters
  in let local_args = List.map 
		     (fun _ -> TypeArgument(
			FreeTypeVariable(Type_variable.fresh())))
		     localparams
  in
    export_member_with_args iface member (global_args @ local_args) memtyp

(***********************************************************************
 *
 * substitute all type variables with fresh ones
 * 
 * same constraint on Predtype as in generalize_type_parameter
 *
 * val uniquify_type_variables
 *   ('cl, 'mem) top_pre_types -> ('cl, 'mem) top_pre_types
 * 
 *)


let uniquify_type_variables typ =
  let subst_list = ref ([] : (Type_variable.t * Type_variable.t) list) in
  let subst tv = 
    try
      Util.assoc Type_variable.eq tv !subst_list
    with
      | Not_found ->
	  let ntv = Type_variable.fresh () in
	  let _ = subst_list := (tv,ntv) :: !subst_list
	  in
	    ntv
  in 
  let rec doit_typ typ = match typ with
    | Groundtype(id,args) -> Groundtype(id, doit_args args)
    | TypeConstant(name,flag,args) -> TypeConstant(name,flag, doit_args args)
    | BoundTypeVariable id -> typ
    | FreeTypeVariable t -> FreeTypeVariable(subst t)
    | Self -> typ
    | Carrier -> typ
    | Bool -> typ
    | Function(dom,codom) -> Function( doit_typ dom, doit_typ codom)
    | SmartFunction(doml,codom) -> 
	   SmartFunction( List.map doit_typ doml, doit_typ codom)
    | Product(type_list) 	-> Product(List.map doit_typ type_list)
    | Class(cl, args) -> Class(cl, doit_args args)
    | Adt(adt,flag,args) -> Adt(adt,flag, doit_args args)
    | Record(labels) -> Record(List.map (fun (l,t) -> (l, doit_typ t)) labels)
    | IFace(cl,arg_flag,args) -> IFace(cl,arg_flag, doit_args args)
    | Array(cl,elemtyp,size) -> Array(cl, doit_typ elemtyp, size)
					   (* not allowed here *)
    | Predtype(formula) -> 
	   assert(false)
  and doit_args args = 
    List.map (function TypeArgument(t) -> TypeArgument(doit_typ t))
	 args
  in
    doit_typ typ


(***********************************************************************
 *
 * small utilities
 *
 *)


(* computing the list of argument types from a method *)

let member_arg_list c =
  match c#get_domain with
    | Product(tl) -> tl
    | t -> [t]

let method_arg_list m =
  match m#get_domain with
      | Product(Self::tl) -> tl
      | Self -> [] 
      | _ -> assert(false) 

  (**************
   * check if the arguments suit the parameters in number kind and type
   *)
let check_parameters params args = 
  (List.length params) = (List.length args)


let id_record_from_string name = 
  { id_token  = 
      { token_name = name;
	loc = None
      };
    id_type = TypeConstant("Irrelevant field", Never, []);
    id_parameters = [];
    id_origin = CCSL_Output;
    id_variance = Unset;
    id_sequence = -1;
  }


  (**************
   * utility functions for features
   * at the moment groundtypes have a fixed feature set
   *)

let groundtype_feature id = function 
  | HasRelLiftFeature
  | HasMapFeature
      -> true
  | BuildinFeature
  | NeedsMapFeature
  | FinalSemanticsFeature
  | MixedSelfInstFeature
  | ComponentNoRelLiftFeature
  | ComponentNoFullRelLiftFeature
  | ComponentNoMapFeature
  | ComponentNoFullMapFeature
  | ClassComponentFeature
  | CarrierClassComponentFeature
  | HasBisimulationFeature
  | HasFullRelLiftingFeature
  | HasGreatestBisimFeature
  | HasMorphismFeature
      -> assert(false)

  (*
   * supplied type must be one of Class / Adt / Groundtype
   *
   * val type_has_feature : 
   *   feature_type ->
   *     (< has_feature : feature_type -> bool; .. > as 'cl,
   * 	'mem) top_pre_types ->
   * 	 bool
   *)

let type_has_feature feature = function
  | Class(cl,_) -> cl#has_feature feature
  | Adt(adt,_,_) -> adt#has_feature feature
  | Groundtype(id,args) -> groundtype_feature id feature
  | _ -> assert(false)


  (**************
   * the next two functions are for extracting the location field
   * !! they fail if the top constructor is not a location !!
   * 
   * val get_ex_loc : ('cl, 'mem) top_pre_expressions -> location_type
   * 
   * val get_form_loc : ('cl, 'mem) top_pre_formulas -> location_type
   *)

let rec get_ex_loc = function
  | ExprLoc(_,l) -> l
  | _ -> assert false

let rec get_form_loc = function
  | FormLoc(_,l) -> l
  | _ -> assert(false)




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
