(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 8.5.02 by Hendrik
 *
 * Time-stamp: <Wednesday 8 May 02 13:49:08 tews@ithif51>
 *
 * Put a shell around the Lexer for handling include directives
 *
 * $Id: abstract_lexer.ml,v 1.2 2002-05-22 13:42:37 tews Exp $
 *
 *)

open Util
open Global
open Grammar
open Error

exception Include_error

let empty s = 
  try
    ignore(Stack.top s); false
  with
    | Stack.Empty -> true


let d s =
  if debug_level _DEBUG_LEXER
  then begin
    prerr_endline s;
    flush stderr
  end


(* the variable current_top_level_input is in global.ml
 * nevertheless it is maintained here.
 *)

    (* extract the directory from current_top_level_input *)
let get_current_directory () = match !current_top_level_input with
  | None -> assert(false)
  | Some f -> Filename.dirname f


type lexing_pos = 
    {
      lexbuf : Lexing.lexbuf;
      util_state : Parser_util.state_type;
      closing_action : unit -> unit
    }

    (* the stack of lexers and file positions for recursive includes
     *)
let lexer_stack = (Stack.create () : lexing_pos Stack.t)

    (* initialize this lexer shell with a top level input file
     *)
let initialize top_file_name =
  current_top_level_input := Some top_file_name;
  Parser_util.reset_line top_file_name;
  Stack.clear lexer_stack


    (* read a token from the current lexer
     * might raise Stack.Empty if there is no current lexer
     *)
let token_from_top () = 
  Lexer.token (Stack.top lexer_stack).lexbuf


let divert lexbuf file closing_action =
  let _ = d ("Diverting into " ^ file ) in
  let _ =
    if not (empty lexer_stack) then
      let including_lex_pos =
	{ (Stack.pop lexer_stack) with
	    util_state = Parser_util.get_state()
	}			
      in 
	Stack.push including_lex_pos lexer_stack 
  in
  let _ = Parser_util.reset_line file in
  let included_lex_pos =
    { lexbuf = lexbuf;
      util_state = Parser_util.get_state();
      closing_action = closing_action
    }
  in
    Stack.push included_lex_pos lexer_stack
  


    (* read a token
     * handle includes and EOF's of included files
     *)
let rec token lexbuf = 
  try
    (match token_from_top() with
       | INCLUDE(filename, loc) ->
	   let relocated_name = 
	     Filename.concat (get_current_directory()) filename in
	   let included_channel = 
	     try
	       open_in relocated_name
	     with
	       | Sys_error msg -> 
		   begin
		     error_message loc msg;
		     raise Include_error
		   end
	   in 
	     begin
	       divert (Lexing.from_channel included_channel)
		 relocated_name (fun () -> close_in included_channel);
	       token lexbuf
	     end
       | EOF ->
	   begin
	     (Stack.pop lexer_stack).closing_action ();
	     if empty lexer_stack 
	     then EOF
	     else 
	       let top = Stack.top lexer_stack in
	       let top_state = top.util_state in
	       let (line, line_start, file) = top_state 
	       in begin
		   Parser_util.set_state top_state;
		   d ("Continuing lexing in file " ^ file ^
		      " at line " ^
		      (string_of_int line) ^
		      " char " ^
		      (string_of_int 
			 ((Lexing.lexeme_end top.lexbuf) - line_start)));
		   token lexbuf
	     end
	   end
       | othertoken -> othertoken
    )
  with
    | Stack.Empty ->
	begin
	  divert lexbuf 
	    (remove_option !current_top_level_input)
	    (fun () -> ());
	  token lexbuf
	end




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
