(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Wednesday 30 June 10 11:26:00 tews@blau.inf.tu-dresden.de>
 *
 * check assertions/creation conditions for behavioural invariance
 *
 * $Id: behavioural.ml,v 1.5 2010-06-30 09:38:00 tews Exp $
 *
 *)
 

open Util
open Error
open Global
open Top_variant_types
open Types_util
;;

(***********************************************************************
 ***********************************************************************
 *
 * Error messages / debug messages
 *
 *)

let d s = 
  if debug_level _DEBUG_STRICT then
    print_verbose s

let ds s = d( "    " ^ s)


(***********************************************************************
 ***********************************************************************
 *
 * behavioural invariance for formulas/expressions
 *
 *)


let rec pedantic_expression loc expr = 
  let _ = ds ("at loc " ^ (string_of_loc loc)) in
  let recurse_exp = pedantic_expression loc in
  let recurse_form = pedantic_formula loc in
    match expr with
      | ExprLoc(ex,loc) -> pedantic_expression loc ex

      | TypeAnnotation(BasicExpr(Member(instiface, 
					Resolved(m))), typ) ->
	  (match instiface with
					(* member of this class *)
	     | CurrentIface -> 
		 if m#is_action 
		 then ()		(* a method *)

		  (* Constructors are not behaviourally invariant, but here
		   * we only need to assure that the final model exists. For 
		   * that purpose in creation conditions 
		   * constructors on the current class are ok.
		   *)
		 else if m#is_constructor  (* a constructor *)
		 then ()
		 else
		   assert false

	     | InstIface(ocl, args, oloc) ->
		 (* some other function, 
		  * only if it is essentially monomorphic 
		  *)
		 if not (constant_arg_list (fun _ -> false) args)
		 then
		   pedantic_error loc
		     ("Polymorphic functions can only be used with " ^
		      "constant types"
		     )

	     | NoIface ->
		 assert(false)
	  )

					    (* we are in pedantic_expression *)

					(* a variable *)
      | BasicExpr(TermVar(Resolved(id_rec))) -> ()

				(* other BasicExpr cannot appear here *)
      | BasicExpr(TermVar(Unresolved _))
      | BasicExpr(Member _) -> 
	  assert(false)
					(* a natural number *)
      | TypedTerm(Term _, typ) -> ()

      | TypedTerm(ex, typ) -> recurse_exp ex

      | MethodSelection(TypeAnnotation(ex,ex_typ), typeopt, Resolved m) ->
	  (match ex_typ with
	     | Self ->
		 assert(m#is_action)
	     | Class(cl, args) ->
		 (* some other function, 
		  * only if it is essentially monomorphic 
		  *)
		 if not (constant_arg_list (fun _ -> false) args)
		 then
		   pedantic_error loc
		     ("Polymorphic functions can only be used with " ^
		      "constant types"
		     )

	     | _ -> assert false 
	  )

					    (* we are in pedantic_expression *)

				(* MethodSelection needs TypeAnnotation *)
      | MethodSelection _ ->
	  assert(false)

      | Tuple(ex_list) -> List.iter recurse_exp ex_list
      | Projection(i,_) -> ()
      | Abstraction(decl_list,ex) -> recurse_exp ex
      | Application(ex1,ex2) -> recurse_exp ex1; recurse_exp ex2
      | InfixApplication(ex1, iface, tokcontainer, ex2) ->
	  recurse_exp ex1; recurse_exp ex2 

	    (* standard translation of function update involves equality
	     * and indeed it can break behavioural invariance: Consider
	     * a non-minimal model and a cluster of bisimilar states. 
	     * updating a function for just one of these object results 
	     * in !@#$
	     *)
      | FunUpdate(fex, changes) -> 
	  pedantic_error loc "function update is not behaviourally equivalent";

	  (* let translates in abstraction and application *)
      | Let(decl_list, ex) ->
	  List.iter (fun (id_rec, typopt, ex) -> recurse_exp ex)
	    decl_list;
	  recurse_exp ex
      | If(conds,ex) -> 
	  List.iter (fun (f,ex) -> (recurse_form f; recurse_exp ex)) conds;
	  recurse_exp ex
      | CCSL_Case(ex,variants) -> 
	  recurse_exp ex;
	  List.iter (fun (mem,ids,exp) -> recurse_exp exp) variants

					    (* we are in pedantic_expression *)

	    (* holds for polynomial functors *)
      | Modality(modal,typ,pred,tlist) -> recurse_exp pred

      | Expression form -> recurse_form form
      | TypeAnnotation(exp,typ) -> recurse_exp exp
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
	
and pedantic_formula loc formula =
  let _ = ds ("at loc " ^ (string_of_loc loc)) in
  let recurse_exp = pedantic_expression loc in
  let recurse_form = pedantic_formula loc in
    match formula with
      | FormLoc(f,loc) -> pedantic_formula loc f
      | True -> ()
      | False -> ()
      | Not f -> recurse_form f
      | And f_list -> List.iter recurse_form f_list
      | Or f_list -> List.iter recurse_form f_list
      | Implies(assum,concl) -> 
	  recurse_form assum; 
	  recurse_form concl
      | Iff(assum,concl) -> 
	  recurse_form assum;
	  recurse_form concl
					(* allow equality for constant types *)
      | Equal( TypeAnnotation(ex_a,eq_typ), ex_b) -> 
	  recurse_exp ex_a; 
	  recurse_exp ex_b;
	  if count_self eq_typ <> 0 
	  then
	    pedantic_error loc "Equality is not behaviourally invariant"

					(* Equal needs TypeAnnotation *)
      | Equal _ -> assert(false)
	  
				(* Quantification over constant types only *)
      | Forall(decll, f) -> 
	  List.iter
	    (fun (var,typ) -> 
	       if count_self typ <> 0 
	       then
		 pedantic_error loc
		   "Quantification over Self is not behaviourally invariant"
	    )
	    decll;
	  recurse_form f

      | Exists(decll, f) -> 
	  List.iter
	    (fun (var,typ) -> 
	       if count_self typ <> 0 
	       then
		 pedantic_error loc
		   "Quantification over Self is not behaviourally invariant"
	    )
	    decll;
	  recurse_form f

      | Formula ex -> recurse_exp ex
      | Obseq(Some(obsname, eq_typ), ex1,ex2) -> 
	  recurse_exp ex1;
	  recurse_exp ex2;

      | Obseq _ 
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | Bisim _
      | ConstantPredicate _
	-> assert(false)







(***********************************************************************
 ***********************************************************************
 *
 * top level items
 *
 *)

let pedantic_assertion ccl assertion =
  if not assertion.is_generated 
  then
    match assertion.assertion_formula with
      | Symbolic(FormLoc(formula,loc)) -> 
	  let _ = d ("  * assertion " ^ assertion.assertion_name.token_name)
	  in
	    pedantic_formula loc formula
	      
      | Symbolic _ -> assert(false)
	  
      | Pvs_String _
      | Isa_String _ -> 
	  pedantic_error (remove_option (ccl#get_token).loc)
	    "Inlined formulae not allowed"


let pedantic_definition ccl def =
  match def.definition with 
    | Symbolic(FormLoc(Formula(TypeAnnotation(defex, righttyp)), loc)) ->
	let mem = def.defined_method in
	let _ = d ("  * definition " ^ mem#get_name) 
	in
	  pedantic_expression loc defex
	      
      | Symbolic _ -> assert(false)
	  
      | Pvs_String _
      | Isa_String _ -> 
	  pedantic_error (remove_option (ccl#get_token).loc)
	    "Inlined definitions not allowed"



let pedantic_class ccl =
  d (" ** TC " ^ ccl#get_name);
  List.iter (pedantic_definition ccl) ccl#get_definitions;
  List.iter (pedantic_assertion ccl) ccl#get_assertions;
  List.iter (pedantic_assertion ccl) ccl#get_creations


let pedantic_ast = function
  | CCSL_class_dec cl -> pedantic_class cl
  | CCSL_adt_dec adt -> ()
  | CCSL_sig_dec si -> ()


let ccsl_pedantic_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter pedantic_ast ast;;




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
