(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 9.3.02 by Hendrik
 *
 * Time-stamp: <Thursday 1 July 10 0:32:53 tews@gromit.tews.net>
 *
 * The class method pass : coreduce and friends
 * requires that feature pass has been done
 * 
 * $Id: class_methods.ml,v 1.6 2010-07-02 10:55:55 tews Exp $
 *
 *)

open Util
open Error
open Top_variant_types
open Types_util
open Member_class
;;


(*********************************************************
 *
 * Utility functions
 *
 *)


let check_for_clash loc cl name = 
  try 
    ignore(cl#find_member name); 
    warning_message loc
      (name ^ " not defined because of name clash.");
    false
  with 
    | Member_not_found -> true


let make_class_suit_type_list cl self_subst =
  let type_list = 
    List.map (fun m -> m#get_full_type) 
      cl#get_all_sig_actions 
  in
  let subst = [ Self, self_subst  ] 
  in
    List.map (ccsl_substitute_types subst) type_list


let make_adt_suit_type_list adt carrier_subst = 
  let typ_list = 
    List.map 
      (fun const -> const#get_full_type)
      adt#get_adt_constructors 
  in
  let subst = [ Carrier, carrier_subst  ] 
  in
    List.map (ccsl_substitute_types subst) typ_list


let add_iface_method cl name typ sort =
  let _ = assert(sort = Class_Sig_Special 
		 || sort = Class_Coreduce
		 || sort = Class_Map
		 || sort = Adt_Special) 
  in
  let m_tok = {token_name = name; loc = None};
  in
  let mem = new ccsl_pre_member_class cl m_tok []
	      (Product[]) typ NoVisibility sort
  in
    cl#add_member mem



(*********************************************************
 *
 * invariance
 *
 *)

let add_invariance cl inv_name = 
  let tv = FreeTypeVariable(Type_variable.fresh()) in
  let class_suit_type_list = make_class_suit_type_list cl tv in
  let inv_type = SmartFunction( class_suit_type_list, 
			   Function( Function(tv, Bool), Bool))
  in
    add_iface_method cl inv_name inv_type Class_Sig_Special


(*********************************************************
 *
 * bisimulation
 *
 *)

let add_bisimulation cl bisim_name =
  let tv1 = FreeTypeVariable(Type_variable.fresh()) in
  let tv2 = FreeTypeVariable(Type_variable.fresh()) in
  let suit_1 = make_class_suit_type_list cl tv1 in
  let suit_2 = make_class_suit_type_list cl tv2 in
  let bisim_type =
    SmartFunction
      (suit_1,
       SmartFunction(suit_2,
		Function( 
		  Function(Product[tv1;tv2], Bool), Bool)))
  in
    add_iface_method cl bisim_name bisim_type Class_Sig_Special


(*********************************************************
 *
 * morphism
 *
 *)

let add_morphism cl morph_name = 
  let tv1 = FreeTypeVariable(Type_variable.fresh()) in
  let tv2 = FreeTypeVariable(Type_variable.fresh()) in
  let suit_1 = make_class_suit_type_list cl tv1 in
  let suit_2 = make_class_suit_type_list cl tv2 in
  let morph_type =
    SmartFunction
      (suit_1,
       SmartFunction(suit_2,
		Function( 
		  Function(tv1,tv2), Bool)))
  in
    add_iface_method cl morph_name morph_type Class_Sig_Special


(*********************************************************
 *
 * coreduce
 *
 *)

(* add coreduce to cl *)
let add_coreduce cl coreduce_name =
  let tv = FreeTypeVariable(Type_variable.fresh()) in 
  let class_suit_type_list = make_class_suit_type_list cl tv in
  let coreduce_type = SmartFunction( class_suit_type_list, Function(tv, Self))
  in
    add_iface_method cl coreduce_name
      coreduce_type Class_Coreduce


(*********************************************************
 *
 * class map
 *
 *)

let add_class_map cl map_name =
  (* do variance separation; 
   * see also ccsl_adt_map_theory, ccsl_map_struct_theory, 
   *    create_ids_with_variance
   *)
  let param_ids = List.map (function TypeParameter id -> id) cl#get_parameters
  in let source_types = 
      List.map (fun _ -> FreeTypeVariable(Type_variable.fresh())) param_ids
  in let target_types = 
      List.map (function id -> BoundTypeVariable(id)) param_ids
  in let pre_pre_fun_types = 
      Util.map3 (fun source target id -> 
		   (Function(source, target),
		    Logic_util.make_simple id.id_variance))
	source_types target_types param_ids
  in
  let pre_fun_types = 
    List.map
      (fun (t,v) -> match v with
	 | Unused
	 | Pos -> (None, Some(t))
	 | Neg -> 
	     (match t with 
		| Function(dom,codom) -> (Some( Function(codom,dom) ), None)
		| _ -> assert(false)
	     )
	 | Mixed -> 
	     (match t with 
		| Function(dom,codom) -> (Some( Function(codom,dom) ), Some(t))
		| _ -> assert(false)
	     )
	 | _ -> assert(false)
      )
      pre_pre_fun_types
  in let fun_types = Logic_util.variance_flatten pre_fun_types
  in
    (* source/target types *)
  let source_typ = 
    Class(cl, List.map (fun t -> TypeArgument t) source_types)
  in 
  let target_typ = Self
  in let map_type =
      SmartFunction(fun_types, Function(source_typ, target_typ))
  in
    add_iface_method cl map_name map_type Class_Map


(*********************************************************
 *
 * reduce
 *
 *)

(* add reduce to adt *)
let add_reduce adt reduce_name =
  let tv = FreeTypeVariable(Type_variable.fresh()) in 
  let adt_suit_type_list = make_adt_suit_type_list adt tv in
  let reduce_type = SmartFunction( adt_suit_type_list, Function(Carrier, tv))
  in
    add_iface_method adt reduce_name
      reduce_type Adt_Special


(*********************************************************
 *
 * adt map
 *
 *)

let add_adt_map adt map_name =
  (* do variance separation; 
   * see also ccsl_adt_map_theory, ccsl_map_struct_theory, 
   *    create_ids_with_variance
   *)
  let param_ids = List.map (function TypeParameter id -> id) adt#get_parameters
  in let source_types = 
      List.map (function id -> BoundTypeVariable(id)) param_ids
  in let target_types = 
      List.map (fun _ -> FreeTypeVariable(Type_variable.fresh())) param_ids
  in let pre_pre_fun_types = 
      Util.map3 (fun source target id -> 
		   (Function(source, target),
		    Logic_util.make_simple id.id_variance))
	source_types target_types param_ids
  in
  let pre_fun_types = 
    List.map
      (fun (t,v) -> match v with
	 | Unused
	 | Pos -> (None, Some(t))
	 | Neg -> 
	     (match t with 
		| Function(dom,codom) -> (Some( Function(codom,dom) ), None)
		| _ -> assert(false)
	     )
	 | Mixed -> 
	     (match t with 
		| Function(dom,codom) -> (Some( Function(codom,dom) ), Some(t))
		| _ -> assert(false)
	     )
	 | _ -> assert(false)
      )
      pre_pre_fun_types
  in let fun_types = Logic_util.variance_flatten pre_fun_types
  in
    (* source/target types *)
  let source_typ = Carrier in
  let target_typ = 
    Adt(adt, Always, List.map (fun t -> TypeArgument t) target_types)
  in let map_type =
      SmartFunction(fun_types, Function(source_typ, target_typ))
  in
    add_iface_method adt map_name map_type Adt_Special


(*********************************************************
 *
 * the pass
 *
 *)


let do_class cl  =
  let loc = remove_option (cl#get_token).loc 
  in
    (let coreduce_name = Names.name_of_class_coreduce cl in
       if (cl#has_feature FinalSemanticsFeature) &&
	 (check_for_clash loc cl coreduce_name)
       then
	 add_coreduce cl coreduce_name
    );

    (let map_name = Top_names.name_of_class_map cl in
       if (cl#has_feature HasMapFeature) &&
	 (check_for_clash loc cl map_name)
       then
	 add_class_map cl map_name
    );

    (let inv_name = Names.name_of_private_struct_invariance cl 
     in
       if check_for_clash loc cl inv_name
       then
	 add_invariance cl inv_name
    );

    (let morph_name = Names.name_of_full_morphism_struct_pred cl in
       if (cl#has_feature HasMorphismFeature) &&
	 (check_for_clash loc cl morph_name)
       then 
	 add_morphism cl morph_name
    );

    (let bisim_name = Names.name_of_private_struct_bibisimulation cl in
       if (cl#has_feature HasBisimulationFeature) &&
	 (check_for_clash loc cl bisim_name)
       then
	 add_bisimulation cl bisim_name
    )


let do_adt adt =
  let loc = remove_option (adt#get_token).loc
  in
    (let reduce_name = Names.name_of_adt_reduce adt in
       if check_for_clash loc adt reduce_name
       then
	 add_reduce adt reduce_name
    );

    (let map_name = Top_names.name_of_adt_map adt in
       if (adt#has_feature HasMapFeature) &&
	 (check_for_clash loc adt map_name)
       then
	 add_adt_map adt map_name
    )


let do_ast = function
  | CCSL_class_dec cl -> do_class cl
  | CCSL_adt_dec adt -> do_adt adt
  | CCSL_sig_dec si -> ()


let ccsl_class_methods_pass (ast: Classtypes.ccsl_ast list) = 
    List.iter do_ast ast;;


(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

