(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 20.8.03 by Hendrik
 *
 * Time-stamp: <Wednesday 30 June 10 11:31:51 tews@blau.inf.tu-dresden.de>
 *
 * bottom up type derivarion; see comments in newtypecheck.ml
 *
 * $Id: derive.ml,v 1.6 2010-06-30 09:38:01 tews Exp $
 *
 *)


open Util
open Global
open Error
open Top_variant_types
open Top_variant_types_util
open Ccsl_pretty
open Classtypes
open Types_util
;;



(***********************************************************************
 ***********************************************************************
 *
 * Error messages
 *
 *)

exception Typecheck_error

let mismatch loc type1 type2 =
  error_message loc
    ("Type mismatch. Type " ^
     (string_of_ccsl_type type1) ^
     "\nis not compatible with type " ^
     (string_of_ccsl_type type2) ^ "."
    )

let wrong_object_type loc objtype quali =
  error_message loc
    ("Type mismatch in qualified method selection. Derived type\n" ^
     (string_of_ccsl_type objtype) ^
     " is not compatible with " ^
     (string_of_ccsl_type quali) ^ "."
    )

let undefined_method cl token loc = begin
  error_message loc ("Method " ^ token.token_name ^
		       " not defined in class " ^
		       cl#get_name ^ ".");
  if Global.debug_level _DEBUG_DUMP_SYMBOL_TABLE then begin
    prerr_string( "Symboltable: " );
    print_verbose (Symbol.symbol_table_dump 
		     Top_variant_types_util.dump_symbol_brief);
  end;
  raise Typecheck_error
end

let proj_i_error projexpr loc producttype =
  error_message loc
  ("Type mismatch. Derived " ^
   (string_of_ccsl_type producttype) ^ "\n" ^
   "for the domain of " ^
   (string_of_ccsl_expression projexpr) ^ "."
  )

let proj_dom_error loc domtype =
  error_message loc
    ("Type mismatch. Derived " ^
     (string_of_ccsl_type domtype) ^
     " for the domain of a projection.\n" ^
     "This is not compatible with a product type."
    )

let is_not_fun end_message loc typ constr =
  error_message loc
    ("This expression has type " ^
     (string_of_ccsl_type typ) ^ ". " ^
     end_message ^ "."
    )

let dom_mismatch loc domtype argtype =
  error_message loc
    ("Type mismatch in application. Function domain type\n" ^
     (string_of_ccsl_type domtype) ^
     " is not compatible with type of argument\n" ^
     (string_of_ccsl_type argtype) ^ "."
    )

let types_dont_match loc typ constrainttyp =
  error_message loc
    ("This expression has type " ^
     (string_of_ccsl_type typ) ^
     "\nbut is expected to have type " ^
     (string_of_ccsl_type constrainttyp) ^ "."
    )
	  
let if_mismatch if_or_case loc type1 type2 =
  error_message loc
    ("Branches of " ^
     if_or_case ^ 
     " have different types. Type\n" ^
     (string_of_ccsl_type type1) ^
     "\nis not compatible with type\n" ^
     (string_of_ccsl_type type2)
    )

let case_mismatch loc typex adttype =
  error_message loc
    ("This case expression matches values of " ^
     (match adttype with
	| Adt(adt,_,_) -> adt#get_name
	| _ -> assert(false)
     ) ^
     ". Type\n" ^
     (string_of_ccsl_type typex) ^
     " is not compatible with\n" ^
     (string_of_ccsl_type adttype) ^ "."
    )

let eq_not_equal loc typeleft typeright =
  error_message loc
    ("Both sides of the equation must have equal type. Type\n" ^
     (string_of_ccsl_type typeleft) ^
     " is not compatible with type\n" ^
     (string_of_ccsl_type typeright)
    )

let d s = 
  if debug_level _DEBUG_TYPECHECK then
    print_verbose s

let dd s = d( "    TD " ^ s)

let du s = 
  if debug_level _DEBUG_UNIFICATION then
    print_verbose ( "    Unify " ^ s)



(***********************************************************************
 ***********************************************************************
 *
 * type variable management
 *
 *)


exception Occur_check

type tv_binding_type = (Type_variable.t * ccsl_output_types) list ref

let new_tv_binding () : tv_binding_type = ref []

(* the following invariant is kept on new_tv_binding:
 * type variables that have a binding must not occur in the types 
 * for bindings of other type variables. 
 * the function add_new_binding keeps this assertion by 
 * performing various substitutions before adding a binding.
 *)

let string_of_tv_bindings (tv_bindings : tv_binding_type) = 
  List.fold_right (fun (tv,typ) res -> 
		     (Type_variable.string_of_tv tv) ^ " => " ^
		     (string_of_ccsl_type typ) ^ "; " ^ res)
    !tv_bindings ""

let find_tv (tv_bindings : tv_binding_type) tv = 
  Util.assoc Type_variable.eq tv !tv_bindings

let normalize_type (tv_bindings : tv_binding_type) t = 
  ccsl_substitute_types 
    (List.map (fun (tv,t) -> (FreeTypeVariable tv, t))
       !tv_bindings)
    t

let normalize_args (tv_bindings : tv_binding_type) arglist = 
  ccsl_substitute_arguments
    (List.map (fun (tv,t) -> (FreeTypeVariable tv, t))
       !tv_bindings)
    arglist


let add_new_binding (tv_bindings : tv_binding_type) tv t = 
  let nt = normalize_type tv_bindings t in
  let _ = dd ("bind " ^
	     (Type_variable.string_of_tv tv) ^
	     " to " ^ (string_of_ccsl_type t)) 
  in
  let _ = if occur_check tv nt then raise Occur_check 
  in let subst = [ FreeTypeVariable tv, nt] in
  let ntv_bindings = 
    List.map (fun (tv,t) -> (tv, ccsl_substitute_types subst t))
      !tv_bindings
  in
    tv_bindings := (tv,nt) :: ntv_bindings


(***********************************************************************
 ***********************************************************************
 *
 * Type Unification
 *
 *)

exception Unify

let rec unify tv_bindings t1 t2 = 
  let recurse = unify tv_bindings in
  let recurse_arg = unify_arg tv_bindings 
  in
    (match t1,t2 with
       | Groundtype(id1,args1), Groundtype(id2,args2) when
	   id1.id_token.token_name = id2.id_token.token_name ->
	     begin
(* HENDRIK: this assertion can be triggered with overloading groundtypes *)
	       assert((List.length args1) = (List.length args2));
	       Groundtype(id1, List.map2 recurse_arg args1 args2)
	     end
       | BoundTypeVariable id1, BoundTypeVariable id2 ->
	   if id1.id_token.token_name = id2.id_token.token_name 
	   then
	     BoundTypeVariable id1
	   else
	     raise Unify
							  (* we are in unify *)
       | FreeTypeVariable t1, FreeTypeVariable t2
	   when Type_variable.eq t1 t2 
	     ->
	        FreeTypeVariable t1
       | Self, Self -> Self
       | Carrier, Carrier -> Carrier
       | Bool, Bool -> Bool
       | Function(dom1,codom1), Function(dom2,codom2) -> 
	   let ndom = recurse dom1 dom2 in
	   let ncodom = recurse codom1 codom2 
	   in
	     Function(ndom, ncodom)
       | Product(type_list1),Product(type_list2) ->
	   if (List.length type_list1) = (List.length type_list2) 
	   then
	     Product( List.map2 recurse type_list1 type_list2 )
	   else
	     raise Unify
							  (* we are in unify *)
       | Class(cl1, args1),Class(cl2, args2) ->
	   if cl1#get_name = cl2#get_name 
	   then
	     begin
	       assert((List.length args1) = (List.length args2));
	       Class(cl1, List.map2 recurse_arg args1 args2)
	     end
	   else
	     raise Unify
       | Adt(adt1,flag1,args1),Adt(adt2,flag2,args2) ->
	   if adt1#get_name = adt2#get_name 
	   then
	     begin
	       assert((List.length args1) = (List.length args2));
	       Adt(adt1, flag1, List.map2 recurse_arg args1 args2)
	     end
	   else
	     raise Unify
							  (* we are in unify *)

					(* new bindings *)
       | FreeTypeVariable tv1, FreeTypeVariable tv2 ->
	   (try
	      let binding = find_tv tv_bindings tv1
	      in
		recurse binding t2
	    with
	      | Not_found ->
		  try
		    let binding = find_tv tv_bindings tv2
		    in
		      recurse t1 binding
		  with
		    | Not_found ->
			(try
			   add_new_binding tv_bindings tv2 t1;
			   t1
			 with
			   | Occur_check -> raise Unify
			)
	   )		  

							  (* we are in unify *)
       | FreeTypeVariable tv, t ->
	   (try
	      let binding = find_tv tv_bindings tv
	      in 
		recurse binding t
	    with
	      | Not_found -> 
		  (try
		     add_new_binding tv_bindings tv t;
		     t
		   with 
		     | Occur_check -> raise Unify
		  )
	   )
       | t, FreeTypeVariable tv ->
	   (try
	      let binding = find_tv tv_bindings tv
	      in 
		recurse t binding
	    with
	      | Not_found -> 
		  (try
		     add_new_binding tv_bindings tv t;
		     t
		   with 
		     | Occur_check -> raise Unify
		  )
	   )
							  (* we are in unify *)
       | Groundtype(id1,args1), t2 when is_type_def id1 ->
	   let subst = 
	     make_substitution (get_ground_type_parameters id1) args1
	   in
	     recurse (ccsl_substitute_types subst id1.id_type) t2

       | t1, Groundtype(id2,args2) when is_type_def id2 ->
	   let subst = 
	     make_substitution (get_ground_type_parameters id2) args2
	   in
	     recurse t1 (ccsl_substitute_types subst id2.id_type)

	   
					(* not allowed cases *)
       | SmartFunction _,_
       | TypeConstant _,_
       | Predtype _,_
       | Record _,_
       | IFace _,_
       | Array _,_ 
	   
       | _, SmartFunction _
       | _, TypeConstant _
       | _, Predtype _
       | _, Record _
       | _, IFace _
       | _, Array _          -> assert(false)
	   
	   
				(* cases where the constructors mismatch *)
       | Groundtype _, _
       | BoundTypeVariable _ ,_
       | Self ,_
       | Carrier ,_
       | Bool, _
       | Function _ ,_
       | Product _ ,_
       | Class _ ,_
       | Adt _ ,_
							  (* we are in unify *)
(* double cases
 * 	  | _, Groundtype _
 * 	  | _, BoundTypeVariable _
 * 	  | _, Self
 * 	  | _, Carrier 
 * 	  | _, Bool
 * 	  | _, Function _
 * 	  | _, Product _
 * 	  | _, Class _
 * 	  | _, Adt _ 
 *)
          -> raise Unify
    )
    
and unify_arg tv_bindings arg1 arg2 = 
  (match arg1,arg2 with
     | TypeArgument t1, TypeArgument t2 ->
	 TypeArgument( unify tv_bindings t1 t2 )
  )


let unify_types tv_bindings t1 t2 error_msg_fun =
  try
    let _ = du ((string_of_ccsl_type t1) ^ " & " ^
		(string_of_ccsl_type t2)) in
    let nt = unify tv_bindings t1 t2 in

    let _ = du ("==> " ^ (string_of_ccsl_type nt))
    in let _ = du ("bound vars: " ^(string_of_tv_bindings tv_bindings))
    in
      nt
	       
  with
    | Unify ->
	let nt1 = (normalize_type tv_bindings t1) in
	let nt2 = (normalize_type tv_bindings t2) in
	  error_msg_fun nt1 nt2;
	  raise Typecheck_error


let types_must_unify t1 t2 =
  let tv_bindings = new_tv_binding () 
  in
    try
      ignore(unify tv_bindings t1 t2);
      !tv_bindings = []
    with
      | Unify -> false


(***********************************************************************
 ***********************************************************************
 *
 * Type derivation
 *
 *)

let rec derive_expression ccl loc tv_bindings expr = 
  let recurse_exp = derive_expression ccl loc tv_bindings in
  let recurse_form = derive_formula ccl loc tv_bindings in
  let do_unify = unify_types tv_bindings in
  let nexpression, ntype = 
    match expr with
      | ExprLoc(ex,l) -> 
	  let _ = dd ("ExprLoc " ^ (string_of_loc l)) in
	  let nex, typ_ex = derive_expression ccl l tv_bindings ex 
	  in
	    (ExprLoc(nex, l),
	     typ_ex)
      | TypeAnnotation(BasicExpr(Member _), typ) ->
	  expr, typ
      | BasicExpr(TermVar(Resolved(id_rec))) -> 
	  expr, id_rec.id_type
				(* other BasicExpr cannot appear here *)
      | BasicExpr(TermVar(Unresolved _))
      | BasicExpr(Member _) -> 
	  assert(false)

      | TypedTerm(Term _, typ) -> 
	  let _ = dd "typed ground term" 
	  in
	    (expr, typ)

      | TypedTerm(ex, typ) -> 
	  let _ = dd "TypedTerm" in
	  let nex, typ_ex = recurse_exp ex in
	  let utyp_ex = do_unify typ_ex typ (mismatch loc)
	  in
	    (TypedTerm(nex, utyp_ex),
	     utyp_ex)

					      (* we are in derive_expression *)
      | MethodSelection(ex, instiface, m) ->
	  let _ = dd "MethodSelection" in
	  let mtoken = match m with
	    | Unresolved t -> t
	    | Resolved _ -> assert(false) in
	  let nex, typ_ex = recurse_exp ex in
	  let utyp = match instiface with
	    | InstIface (iface,uargs,instloc) -> 
		assert(iface#is_class);

		(* generate arguments if necessary *)
		let args = match uargs, iface#get_parameters with
		  | [], (_ :: _ as params) ->     (* user left out arguments *)
		      List.map (fun _ -> 
				  TypeArgument(FreeTypeVariable(
						 Type_variable.fresh())))
		      params
		  | _, params -> 		  (* user gave arguments *)
		      (if not (check_parameters params uargs) 
		       then
			 begin
			   Parser.instantiation_error (remove_option instloc) 
			     (iface#get_name ^ "::" ^ mtoken.token_name)
			     params uargs;
			   raise Typecheck_error
			 end 
		      );
		      uargs
		in
		  do_unify typ_ex (Class(iface,args)) (wrong_object_type loc)

	    | CurrentIface ->
		do_unify typ_ex Self (wrong_object_type loc)
	    | NoIface -> typ_ex 
	  in
	  let nutyp = normalize_type tv_bindings utyp in
	  let _ = match nutyp with
	    | Self			
	    | Class _			(* all ok *)
		-> ()

	    (* a type variable might be ok, but the type checker is not
	     * smart enough.
	     *)
	    | FreeTypeVariable _ -> 
		begin
		  error_message loc(
		    "Cannot determine type for subject of method selection. "^
		    "Derived " ^
		    (string_of_ccsl_type nutyp) ^
		    "\nPlease use a qualified method name or " ^
		    "add type constraints."
		  );
		  raise Typecheck_error
		end

					(* incompatible *)
	    | _ ->
		begin
		  error_message loc
		    ("Subject of method selection has type\n" ^
		     (string_of_ccsl_type nutyp) ^
		     "\nThis type is not compatible with any object type."
		    );
		  raise Typecheck_error
		end
					      (* we are in derive_expression *)

	  (* leave nonground arguments ... they are checked in newtypecheck
	   *
           * in let _ = 
	   *     if type_is_nonground nutyp 
	   *     then
	   * 	begin
	   * 	  error_message loc(
	   * 	    "Cannot determine type for subject of method selection. "^
	   * 	    "Derived " ^
	   * 	    (string_of_ccsl_type nutyp) ^
	   * 	    "\nPlease use a qualified method name or " ^
	   * 	    "add type constraints."
	   * 	  );
	   * 	  raise Typecheck_error
	   * 	end
           *)
	  in let cl = match nutyp with
	    | Self -> ccl
	    | Class(cl,_) -> cl 
	    | _ -> assert(false)

	  (* debugging
           * in let _ = 
	   *     (print_verbose( "Method selection Symboltable: " );
	   *      print_verbose (Symbol.dump_scope cl#get_local
	   * 	     Top_variant_types_util.dump_symbol_brief)
	   *     )
           *)

	  in let m = 
	    try cl#find_member mtoken.token_name
	    with | Member_not_found -> undefined_method cl mtoken 
		     (remove_option mtoken.loc) in
	  let _ = (if not m#is_action 
		   then undefined_method cl mtoken
		     (remove_option mtoken.loc)) in
					      (* we are in derive_expression *)
	  let typ,ninstiface = match nutyp with
	    | Self -> (m#get_curried_type, CurrentIface)
	    | Class(cl,args) -> 
		let locopt = 		(* preserve user loc if given *)
		  match instiface with
		    | InstIface(_,_,locopt) -> locopt
		    | _ -> None
		in
		  (snd( export_member_with_args cl m args m#get_curried_type ),
		   InstIface(cl, args, locopt))
	    | _ -> assert(false)
	  in let _ = dd ("Resolve " ^ m#get_name ^ ": " ^
			 m#hosting_class#get_name)
	  in
	    (MethodSelection(
	       TypeAnnotation(nex, nutyp),
	       ninstiface, Resolved m),
	     typ)

      | Tuple(ex_list) -> 
	  let _ = dd "Tuple" in
	  let nex_list, typ_list = 
	    List.split (List.map recurse_exp ex_list)
	  in
	    (Tuple(nex_list), Product(typ_list))

		(* hack for projections, 
		 * this branch is taken when the projection appears 
		 * in an application 
		 *)
					      (* we are in derive_expression *)
      | Application(ExprLoc(Projection(i,_), ploc) as proj_ex, ex2) ->
	  let _ = dd "ProjAppl" in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let ntyp_ex2 = normalize_type tv_bindings typ_ex2 in
	  let expand_ntyp_ex2 = expand_type_def eq_ccsl_types ntyp_ex2
	  in
	    (match expand_ntyp_ex2 with
	       | FreeTypeVariable _ -> 
		   let res_type = FreeTypeVariable(Type_variable.fresh()) in
		   let proj_type = Function(expand_ntyp_ex2, res_type) 
		   in
		     (Application(
			TypeAnnotation( proj_ex, proj_type ),
			nex2),
		      res_type)
	       | Product domtlist -> 
		   let _ = if i > List.length domtlist then
		     begin
		       proj_i_error proj_ex loc ntyp_ex2;
		       raise Typecheck_error
		     end
		   in let res_type = (List.nth domtlist (i - 1)) in
		   let proj_type = Function(expand_ntyp_ex2, res_type) 
		   in
		     (Application(
			TypeAnnotation( proj_ex, proj_type ),
			nex2),
		      res_type)
	       | _ -> 
		   begin
		     proj_dom_error loc ntyp_ex2;
		     raise Typecheck_error
		   end
	    )

					      (* we are in derive_expression *)
      | Projection(i,_) -> 
	  let _ = dd "Projection" in
	  let typ = 
	    Function(
	      FreeTypeVariable(Type_variable.fresh()),
	      FreeTypeVariable(Type_variable.fresh()))
	  in
	    (expr,
	     typ)	    
      | Abstraction(decl_list,ex) ->
	  let _ = dd "Abstraction" in
	  let nex,typ_ex = recurse_exp ex in
	  let decl_types = List.map snd decl_list in
	  let dom_typ = match decl_types with
	    | [t] -> t
	    | tl -> Product(tl) 
	  in
	    (Abstraction(decl_list, nex),
	     Function(dom_typ, typ_ex))
					      (* we are in derive_expression *)
      | Application(ex1,ex2) -> 
	  let _ = dd "Application" in
	  let nex1, typ_ex1 = recurse_exp ex1 in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let ntyp_ex1 = do_unify typ_ex1
			   (Function(
			      FreeTypeVariable(Type_variable.fresh()),
			      FreeTypeVariable(Type_variable.fresh())))
			   (is_not_fun "It cannot be applied" 
			      (get_ex_loc ex1))
	  in let dom_type, codom_type = match ntyp_ex1 with
	    | Function(dt,ct) -> dt,ct
	    | _ -> assert(false)
		(* unify to bind free type variables, don't use result *)
	  in let _ntyp_ex2 = do_unify dom_type typ_ex2 (dom_mismatch loc)
	  in
	    (Application(
	       TypeAnnotation(nex1, ntyp_ex1),
	       nex2),
	     codom_type)
					      (* we are in derive_expression *)
      | TypeAnnotation(InfixApplication(ex1,instiface,mem,ex2), memtype) ->
	  let _ = dd "Infix" in
	  let nex1, typ_ex1 = recurse_exp ex1 in
	  let nex2, typ_ex2 = recurse_exp ex2 in
	  let lefttype, righttype, resulttype =
	    (match memtype with
	       | Function(Product [t1;t2], t3) -> t1, t2, t3
	       | Function(t1, Function(t2,t3)) -> t1, t2, t3
	       | _ -> assert(false)
	    )
		(* unify to bind free type variables, don't use result *)
	  in let _nlefttype = do_unify typ_ex1 lefttype 
			       (mismatch (get_ex_loc ex1))
		(* unify to bind free type variables, don't use result *)
	  in let _nrighttype = do_unify typ_ex2 righttype 
			       (mismatch (get_ex_loc ex2))
	  in
	    (TypeAnnotation(
	       InfixApplication(nex1,instiface,mem,nex2),
	       memtype),
	     resulttype)

			(* InfixApplication is always under TypeAnnotation *)
      | InfixApplication _ -> assert(false)

					      (* we are in derive_expression *)
      | FunUpdate(fex, changes) -> 
	  let _ = dd "FunUpdate" in
	  let nfex, typ_f = recurse_exp fex in
	  let ntyp_f = do_unify typ_f
			 (Function(
			    FreeTypeVariable(Type_variable.fresh()),
			    FreeTypeVariable(Type_variable.fresh())))
			 (is_not_fun "It cannot occur in an update expression"
			    (get_ex_loc fex))
	  in let dom_type, codom_type = match ntyp_f with
	    | Function(dt,ct) -> dt,ct
	    | _ -> assert(false)
	  in let nchanges = 
	      List.map (fun (ex1,ex2) ->
			  let nex1,typ_ex1 = recurse_exp ex1 in
			  let nex2,typ_ex2 = recurse_exp ex2 in
			    (* unify to bind free type variables, 
			       don't use result *)
			  let _ntyp_ex1 = do_unify typ_ex1 dom_type
					   (mismatch (get_ex_loc ex1)) in
			    (* unify to bind free type variables, 
			       don't use result *)
			  let _ntyp_ex2 = do_unify typ_ex2 codom_type
					   (mismatch (get_ex_loc ex2)) in
			    (nex1, nex2)
		       ) changes 
	  in
	    (FunUpdate( nfex, nchanges), 
	     ntyp_f)

					      (* we are in derive_expression *)
      | Let(decl_list, ex) ->
	  let _ = dd "Let" in
					(* Let binds sequentially *)
	  let ndecl_list =
	    List.map (fun (id_rec, typopt, ex) ->
			let nex, typ_ex = recurse_exp ex in
			let ntyp_ex = do_unify typ_ex id_rec.id_type
					(mismatch (get_ex_loc ex)) in
			let _ = id_rec.id_type <- ntyp_ex
			in
			  (id_rec, typopt, nex)
		     ) decl_list 
	  in let nex,typ_ex = recurse_exp ex 
	  in
	    (Let(ndecl_list, nex),
	     typ_ex)

					      (* we are in derive_expression *)
      | If(conds,else_ex) -> 
	  let _ = dd "If" in
	  let res_type = FreeTypeVariable(Type_variable.fresh()) in
	  let nconds = 
	    List.map 
	      (fun (cond,ex) -> 
		 let ncond,typ_cond = recurse_form cond in
		 let nex, typ_ex = recurse_exp ex in
		   (* typecheck via unify, don't use result *)
		 let _ntyp_cond = do_unify typ_cond Bool 
				   (types_dont_match (get_form_loc ncond)) in
		   (* typecheck via unify, don't use result *)
		 let _ntyp_ex = do_unify typ_ex res_type
				 (if_mismatch "if" (get_ex_loc nex))
		 in
		   (ncond, nex)
	      ) conds
	  in
	  let nelse_ex, typ_else_ex = recurse_exp else_ex in
	  let ntyp_else_ex = do_unify typ_else_ex res_type
			       (if_mismatch "if" (get_ex_loc nelse_ex)) 
	  in
	    (If(nconds, nelse_ex), ntyp_else_ex)

					      (* we are in derive_expression *)
      | CCSL_Case(TypeAnnotation(ex,ex_typ), variants) -> 
	  let _ = dd "CCSL_Case" in
	  let nex, nex_typ = recurse_exp ex in
	  let nnex_typ = do_unify nex_typ ex_typ 
			   (case_mismatch loc) in
	  let res_type = FreeTypeVariable(Type_variable.fresh()) in
	  let nvariants = 
	    List.map (function 
			| (Resolved(m) as const, args, cex) ->
			    let _ = dd ("case " ^ m#get_name) in
			    let ncex, typ_cex = recurse_exp cex in
			      (* typecheck via unify, don't use result *)
			    let _ntyp_cex = do_unify res_type typ_cex 
					(if_mismatch "case" (get_ex_loc cex))
			    in
			      (const, args, ncex)
			| _ -> assert(false)
		     ) variants
	  in
	    (CCSL_Case(TypeAnnotation(nex,nnex_typ), nvariants),
	     res_type)

					(* CCSL_Case needs a TypeAnnotation *)
      | CCSL_Case _ -> assert(false)

					      (* we are in derive_expression *)
      | Modality(modal,typ,pred,tlist) ->
	  let _ = dd "Modality" in
	  let npred, typ_pred = recurse_exp pred in
	  let ntyp_pred = do_unify typ_pred (Function(typ,Bool)) 
			    (types_dont_match (get_ex_loc npred))
	  in
	    (Modality(modal,typ,npred,tlist), ntyp_pred)


      | Expression form -> 
	  let _ = dd "Expression" in
	  let nform, typ_form = recurse_form form 
	  in
	    (Expression(nform), typ_form) 

					      (* we are in derive_expression *)

			(* type annotations are already catched above *)
      | TypeAnnotation _
					(* not allowed in ccsl_input_types *)
      | Term _
      | QualifiedTerm _
      | RecordTuple _
      | RecordSelection _
      | RecordUpdate _
      | Case _
      | List _
      | Every _
      | RelEvery _
      | Map _
      | SmartAbstraction _
      | SmartApplication _
      | Comprehension _ ->
	  assert(false)
  in 
  let nntype = (* normalize_type tv_bindings *) ntype 
  in let _ = 
      if not (is_expr_loc nexpression) then
	dd ("derive " ^ (string_of_ccsl_type nntype) ^ " for " ^
	    (string_of_ccsl_expression nexpression))
  in
    (nexpression, nntype)


and derive_formula ccl loc tv_bindings formula =
  let recurse_exp = derive_expression ccl loc tv_bindings in
  let recurse_form = derive_formula ccl loc tv_bindings in
  let do_unify = unify_types tv_bindings in
  let nformula,ntype = 
    match formula with
      | FormLoc(f,l)    -> 
	  let _ = dd ("FormLoc " ^ (string_of_loc l)) in
	  let nf,typ_f = derive_formula ccl l tv_bindings f 
	  in
	    (FormLoc(nf, l),
	     typ_f)
      | True 		-> (True, Bool)
      | False 		-> (False, Bool)
      | Not f -> 
	  let _ = dd "Not" in
	  let nf,typ_f = recurse_form f in
	    (* typecheck via unify, don't use result *)
	  let _utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc nf))
	  in
	    (Not nf, Bool)

						 (* we are in derive_formula *)
      | And f_list -> 
	  let _ = dd "And" in
	  let nftype_list = List.map recurse_form f_list in
	    (* typecheck via unify, don't use result *)
	  let _utyp_f_list = 
	    List.map 
	      (fun (f,t) -> do_unify t Bool (types_dont_match (get_form_loc f)))
	      nftype_list 
	  in
	  let nf_list, typ_f_list = List.split nftype_list 
	  in
	    (And nf_list, Bool)
      | Or f_list -> 
	  let _ = dd "Or" in
	  let nftype_list = List.map recurse_form f_list in
	    (* typecheck via unify, don't use result *)
	  let _utyp_f_list = 
	    List.map 
	      (fun (f,t) -> do_unify t Bool (types_dont_match (get_form_loc f)))
	      nftype_list 
	  in
	  let nf_list, typ_f_list = List.split nftype_list 
	  in
	    (Or nf_list, Bool)

						 (* we are in derive_formula *)
      | Implies(assum,concl) -> 
	  let _ = dd "Implies" in
	  let nassum, typ_assum = recurse_form assum in
	  let nconcl, typ_concl = recurse_form concl in
	    (* typecheck via unify, don't use result *)
	  let _utyp_assum = do_unify typ_assum Bool
			     (types_dont_match (get_form_loc assum)) in
	  let _utyp_concl = do_unify typ_concl Bool 
			     (types_dont_match (get_form_loc concl))
	  in
	    (Implies( nassum, nconcl), Bool)
      | Iff(assum,concl) -> 
	  let _ = dd "Iff" in
	  let nassum, typ_assum = recurse_form assum in
	  let nconcl, typ_concl = recurse_form concl in
	    (* typecheck via unify, don't use result *)
	  let _utyp_assum = do_unify typ_assum Bool 
			     (types_dont_match (get_form_loc assum)) in
	  let _utyp_concl = do_unify typ_concl Bool 
			     (types_dont_match (get_form_loc concl))
	  in
	    (Iff(nassum, nconcl), Bool)
      | Equal(ex_a,ex_b) -> 
	  let _ = dd "Equal" in
	  let nex_a, typ_a = recurse_exp ex_a in
	  let nex_b, typ_b = recurse_exp ex_b in
	  let utyp_ab = do_unify typ_a typ_b (eq_not_equal loc)
	  in
	    (Equal(
	       TypeAnnotation(nex_a, utyp_ab), nex_b),
	     Bool)
						 (* we are in derive_formula *)
      | Forall(quant_list, f) ->
	  let _ = dd "Forall" in
	  let nf,typ_f = recurse_form f in
	    (* typecheck via unify, don't use result *)
	  let _utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc f))
	  in
	    (Forall( quant_list, nf), Bool)
      | Exists(quant_list, f) -> 
	  let _ = dd "Exists" in
	  let nf,typ_f = recurse_form f in
	    (* typecheck via unify, don't use result *)
	  let _utyp_f = do_unify typ_f Bool (types_dont_match (get_form_loc f))
	  in
	    (Exists( quant_list, nf), Bool)
      | Formula ex -> 
	  let _ = dd "Formula" in
	  let nex, typ_ex = recurse_exp ex
	  in
	    (Formula nex, typ_ex)
						 (* we are in derive_formula *)
      | Obseq(t,ex_a,ex_b) ->		(* t equals None *)
	  let _ = dd "Obseq" in
	  let nex_a, typ_a = recurse_exp ex_a in
	  let nex_b, typ_b = recurse_exp ex_b in
	  let utyp_ab = do_unify typ_a typ_b (eq_not_equal loc)
	  in
	    (Obseq(t, TypeAnnotation(nex_a, utyp_ab), nex_b),
	     Bool)
					(* not in ccsl_input_formulas *)
      | MetaImplies _
      | Bisim _
      | ConstantPredicate _
	-> assert(false)
  in 
  let nntype = (* normalize_type tv_bindings *) ntype 
  in let _ = 
      if not (is_form_loc nformula) then
	dd ("derive " ^ (string_of_ccsl_type nntype) ^ " for " ^
	    (string_of_ccsl_formula nformula))
  in
    (nformula, nntype)
    


let derive_assertion_formula ccl loc tv_bindings formula =
  let nformula, typ_f = derive_formula ccl loc tv_bindings formula in
    (* typecheck via unify, don't use result *)
  let _utyp_f = unify_types tv_bindings typ_f Bool (types_dont_match loc)
  in
    nformula


let derive_definition_expression ccl loc tv_bindings typ expr =
  let nex, typ_ex = derive_expression ccl loc tv_bindings expr in
  let utyp_ex = unify_types tv_bindings typ_ex typ (types_dont_match loc) in
  let _ = assert(types_must_unify utyp_ex typ)
  in
    nex






(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
