(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 16.November 1999 by Jan
 *
 * Time-stamp: <Wednesday 30 June 10 11:25:59 tews@blau.inf.tu-dresden.de>
 *
 * convert CCSL specifics into HOL for the pretty printer
 *
 * $Id: pre_printing.ml,v 1.16 2010-06-30 09:38:01 tews Exp $
 *
 *)


open Global
open Util
open Error
open Top_variant_types
open Top_variant_types_util
open Types_util
open Names
;;

(* Utility function
 *
 * generate a list of identifiers x1 ... xn,
 * this is needed to convert method application without arguments 
 * into a lambda abstraction
 *)

let stupid_make_n_ids n = 
  let rec doit res = function
    | 0 -> res
    | n -> doit (("x" ^ (string_of_int n)) :: res) (n - 1)
  in 
    doit [] n


let rec ccsl_pre_pretty_list pretty_fun l = List.map pretty_fun l


let member_term ooselection ciface instiface name = 
  if !output_mode = Pvs_mode 
  then
    Term(name, Always, [])
  else
    let (memcl, in_source) = 
      match instiface with
	| InstIface(cl, args, Some loc) -> (cl, true)
					(* no annotiontion from source *)
	| InstIface(cl, args, None) ->  (cl, false)
	| CurrentIface -> (ciface, false)

(* HENDRIK: fix update assertions to generate proper instantiated ifaces *)
	| NoIface -> (* assert false *)
	    (ciface, false)
    in
      if in_source || (ooselection && (memcl#get_name <> ciface#get_name))
      then
	QualifiedTerm(isar_theory_name memcl, Always, [], name)
      else
	Term(name, Always, [])


(* Argument member_fun delivers a algebra (a coalgebra respectively) 
 * for a given member. Argument iface_fun does the same for a class.
 * Then the pre_prettyprinting can do its main 
 * task -- to silently insert coalgebras and algebras whereever needed. 
 * The functional dependence is needed for components with final or 
 * loose semantics and to distinguish the current class.
 * 
 *)

let rec ccsl_pre_pretty_basic_expressions ciface member_fun iface_fun = 
  function
    | TermVar _ as exp	-> BasicExpr(exp)

	(* optional class is ignored, because method is resolved *)
    | Member(instiface, mem) ->
	let rom = resolution_of mem in
	let name_of_member  = rom#get_name 
	in
	  match rom#hosting_class#get_kind with
	    | Spec_sig 
	    | Spec_adt ->
		if rom#get_sort = InfixGroundTerm 
		then
		  Term("( " ^ name_of_member ^ " )", Always, [])
		else
		  member_term false ciface instiface name_of_member
	    | Spec_class ->
		(match member_fun rom with
		   | Some suit -> 
		       (* RecordSelection(name_of_member, suit ) *)
		       Application(
			 member_term false ciface instiface name_of_member,
			 suit)
		   | None -> member_term false ciface instiface name_of_member
		)
	    | Spec_Spec -> assert(false)


and ccsl_pre_pretty_expression ciface member_fun iface_fun exp =
  let recurse_ex = ccsl_pre_pretty_expression ciface member_fun iface_fun in
  let recurse_form = ccsl_pre_pretty_formula ciface member_fun iface_fun in
   (match exp with
      | ExprLoc(ex,_) -> recurse_ex ex
      | BasicExpr bex     -> 
	  ccsl_pre_pretty_basic_expressions ciface member_fun iface_fun bex
      | Term _ 		-> exp
      | TypedTerm(ex,typ) ->
	  let nex = recurse_ex ex
	  in
	      TypedTerm(nex,typ)
      | QualifiedTerm _ 	-> exp
	  
		(* optional class is ignored, because method is resolved *)
      | MethodSelection (ex, instiface, mem) ->
	  let rmem = (resolution_of mem) in
	  let ncoalg = 
	    (* only real methods in MethodSelection 
	     * -> member_fun returns Some thing
	     *)
	    remove_option (member_fun rmem) in
	  let name_of_method  = rmem#get_name in
	  let nex = recurse_ex ex in
	  let mem_term = 
	    Application(
	      member_term true ciface instiface name_of_method,
	      ncoalg)
	  in 
	    (match rmem#get_domain with
	       | Self -> (Application(mem_term, nex))
	       | Product(Self :: typ_list) -> 
		   let ids = stupid_make_n_ids (List.length typ_list) in
		   let terms = List.map (fun s -> Term(s,Never,[])) ids 
		   in
		     Abstraction( 
		       List.combine ids typ_list, 
		       Application(
			 mem_term,
			 Tuple(nex :: terms)))
	       | _ -> assert(false)
	    )
	    
		(* optional class is ignored, because method is resolved *)
      | Application(TypeAnnotation(
		      ExprLoc(MethodSelection(ex, instiface, m),_),
		      _), ExprLoc(arg,_)) 
	  when (resolution_of m)#get_domain <> Self ->
	  let arg_list = match arg with
	    | Tuple al -> al
	    | _ -> [arg] 
	  in
	  let rmem = (resolution_of m) in
	  let ncoalg = 
	    (* only real methods in MethodSelection 
	     * -> member_fun returns Some thing
	     *)
	    remove_option (member_fun rmem) in
	  let name_of_method = rmem#get_name in
	  let nex = recurse_ex ex in
	  let mem_term = 
	    Application(
	      member_term true ciface instiface name_of_method,
	      ncoalg)
	  in 
	  let nargs = ccsl_pre_pretty_list 
			(recurse_ex) arg_list
	  in 
	    Application(
	      mem_term,
	      Tuple(nex :: nargs))
	      
      | Tuple(l)		-> 
 	  let nl = 
	    ccsl_pre_pretty_list (recurse_ex) l
	  in
	      Tuple(nl)
      | Application(TypeAnnotation(
		      ExprLoc(TypeAnnotation(Projection(i,n), typ_proj),_), 
		      _), arg) 
	->
	  Application(Projection(i,n), recurse_ex arg)
	    
      | TypeAnnotation(Projection(i,n), proj_typ) -> 
	  let domtlist = match proj_typ with
	    | Function(Product tl,_) -> tl
	    | _ -> assert(false)
	  in let _ = assert(n = List.length domtlist) in
	  let ids = stupid_make_n_ids n in
	  let terms = List.map (fun s -> Term(s,Never,[])) ids 
	  in
	    Abstraction(
	      List.combine ids domtlist,
	      Application(
		Projection(i,n),
		Tuple(terms)))
					(* proj needs TypeAnnotation *)
      | Projection _ -> assert(false)
      | RecordTuple(l)	->
	  let nl = 
	    ccsl_pre_pretty_list
	      (fun (v,ex) -> 
		 (v, recurse_ex ex))
	      l
	  in
	      RecordTuple(nl)
      | RecordSelection(s,ex) ->
	  let nex = recurse_ex ex
	  in
	      RecordSelection(s, nex)
      | RecordUpdate(ex,l)	->
	  let nex = recurse_ex ex
	  in
	  let nl = 	ccsl_pre_pretty_list
			  (fun (v,ex) -> 
			     (v, recurse_ex ex))
			  l
	  in
	      RecordUpdate(nex,nl)
      | List(l) 		->
	  let nl = ccsl_pre_pretty_list 
		     (recurse_ex) l
	  in
	      List(nl)
		
      | Abstraction(l,ex)	-> 
	  let nex = recurse_ex ex
	  in
	      Abstraction(l,nex)

      | SmartAbstraction(l,ex)	-> 
	  let nex = recurse_ex ex
	  in
	      SmartAbstraction(l,nex)

      | Application(ex1,ex2) ->
	  let nex1 = recurse_ex ex1
	  in
	  let nex2 = recurse_ex ex2
	  in
	      Application(nex1,nex2)
      | InfixApplication(ex1,instiface,memc,ex2) ->
	  let nex1 = recurse_ex ex1
	  in
	  let nex2 = recurse_ex ex2
	  in
	      InfixApplication(nex1,instiface,memc,nex2)
      | SmartApplication(ex,l) ->
	  let nex = recurse_ex ex
	  in
	  let nl = 	ccsl_pre_pretty_list
			  (recurse_ex) l
	  in
	      SmartApplication(nex, nl)
      | FunUpdate(ex,l) ->
	  let nex = recurse_ex ex
	  in
	  let nl = 	
	    ccsl_pre_pretty_list
	      (fun (ex1,ex2) ->
		 (recurse_ex ex1,
		  recurse_ex ex2
		 )) 
	      l
	  in
	      FunUpdate(nex,nl)
      | Let(l,ex)	->
	  let nex = recurse_ex ex
	  in
	  let nl = 	
	    ccsl_pre_pretty_list
	      (fun (na,ty,e) ->
		 (na, ty, recurse_ex e))
	      l
	  in
	      Let(nl, nex)
      | If(fexl, ex) 	->
	  let nex = recurse_ex ex
	  in
	  let nfexl = 	
	    ccsl_pre_pretty_list 
	      (fun (f,e) -> 
		 (recurse_form f,
		  recurse_ex e)) 
	      fexl      
	  in
	      If(nfexl,nex)
      | Case(ex,pattex) 	->
	  let nex = recurse_ex ex
	  in
	  let npattex = 	
	    ccsl_pre_pretty_list
	      (fun (pat,ex1) ->
		 (recurse_ex pat,
		  recurse_ex ex1)) 
	      pattex      
	  in
	      Case(nex, npattex)
      | CCSL_Case(ex, pattex) ->
	  let nex = recurse_ex ex
	  in
	  let npattex = 	
	    ccsl_pre_pretty_list  
	      (fun (const, vlist, ex1) ->
		 (const, vlist, 
		  recurse_ex ex1)) 
	      pattex
	  in
	      CCSL_Case(nex, npattex)

      | Modality(modal,typ, ex, tl) ->
	  let nex = recurse_ex ex in
	  let meth_list = 
	    List.map (fun m -> Term(name_of_method_tag_string m.token_name, 
				    Always,[])) tl
	  in
	    Application(
	      Application(
		Application(
		  Term(name_of_modal modal ciface, Always, []),
		  iface_fun typ
		),
		List(meth_list)
	      ),
	      nex
	    )

      | TypeAnnotation(ex,typ) ->
	  let nex = recurse_ex ex
	  in
	    if nex == ex then
	      exp
	    else
	      TypeAnnotation(nex,typ)
      | Every(t,l) 		->
	  let nl = ccsl_pre_pretty_list
		     (recurse_ex) l
	  in
	      Every(t,nl)
      | RelEvery(t,l)	->
	  let nl = ccsl_pre_pretty_list
		     (recurse_ex) l
	  in
	      RelEvery(t,nl)
      | Map(t,l)            ->
	  let nl = ccsl_pre_pretty_list
		     (recurse_ex) l
	  in
	      Map(t,nl)      
      | Expression(f)	->
	  let nef = recurse_form f in
	      Expression(nef)
      | Comprehension(s,t,f) ->
	  let nef = recurse_form f in
	      Comprehension(s,t,nef)
   )		
and ccsl_pre_pretty_formula ciface member_fun iface_fun form =
  let recurse_ex = ccsl_pre_pretty_expression ciface member_fun iface_fun in
  let recurse_form = ccsl_pre_pretty_formula ciface member_fun iface_fun in
    (match form with
       | FormLoc(f,l) -> recurse_form f
       | True	-> form
       | False 	-> form
       | Not(f)	->
	   let nf = recurse_form f in
	       Not(nf)
       | And(l)	->
	   let nl = ccsl_pre_pretty_list
		      (recurse_form)
		      l
	   in
	       And(nl)
       | Or(l)	->
	   let nl = ccsl_pre_pretty_list
		      (recurse_form)
		      l
	   in
	       Or(nl)
       | Implies(f1,f2) ->
	   let nf1 = recurse_form f1 in
	   let nf2 = recurse_form f2 in
	       Implies(nf1,nf2)
       | Iff(f1,f2)	->
	   let nf1 = recurse_form f1 in
	   let nf2 = recurse_form f2 in
	       Iff(nf1,nf2)
       | Equal(ex1,ex2) ->
	   let nex1 = recurse_ex ex1 in
	   let nex2 = recurse_ex ex2 in
	       Equal(nex1,nex2)
       | Forall(l,f)	->
	   let nf = recurse_form f in
	       Forall(l,nf)
       | Exists(l,f) ->
	   let nf = recurse_form f in
	       Exists(l,nf)
       | ConstantPredicate _ -> form
       | Formula(ex)	->
	   let nex = recurse_ex ex in
	       Formula(nex)
       | MetaImplies(f1,f2) ->
	   let nf1 = recurse_form f1 in
	   let nf2 = recurse_form f2 in
	       MetaImplies(nf1,nf2)
					(* Bisim is not in ccsl_expressions *)
       | Bisim(typ,ex1,ex2)	->
	   assert(false)

       | Obseq(t,ex1,ex2) ->
	   let nex1 = recurse_ex ex1 in
	   let nex2 = recurse_ex ex2 in
	   let obsname,obstyp = match t with
	     | None -> assert(false)
	     | Some (n,t) -> n,t
	   in
	    (match obstyp with
	       | Self -> 
		   Formula(
		     Application(
		       Application(Term(name_of_private_bisim_eq ciface, 
					Always, []),
		      		   iface_fun obstyp),
		       Tuple([nex1;nex2])))

	       | Class(cl,[]) ->
		   Bisim(obstyp, nex1, nex2)

	       | _ ->
		   assert(obsname <> "");
		   Formula(
		     Application(
		       Application(
			 Term(obsname, Always, []),
			 iface_fun Self
		       ),
		       Tuple([nex1;nex2])))
	    )
    )	    

(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

