(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 22.4.04 by Hendrik
 *
 * Time-stamp: <Wednesday 23 June 04 21:54:47 tews@debian>
 *
 * prelude strings and functions
 *
 * $Id: prelude.ml,v 1.3 2004-07-08 22:10:02 tews Exp $
 *
 *)

open Global
open Top_variant_types
open Names
;;

(***********************************************************************
 ***********************************************************************
 *
 * the prelude strings
 *
 *)
  

let builtin_prelude = function
  | Pvs_mode -> 
  "
   Begin EmptySig : GroundSignature
      Importing EmptyTypeDef
      Type EmptyType
   End EmptySig
   
   Begin EmptyFunSig [A : Type]: GroundSignature
      Importing EmptyFun[A]
      Constant
         empty_fun : [EmptyType -> A];
   End EmptyFunSig
   
   Begin list[ X : Type ] : Adt
      Constructor
         null : Carrier;
         cons( car, cdr ) : [X, Carrier] -> Carrier
   End list"
      ^ 
      (if !old_lift 
       then ""
       else
  "

   Begin lift[ X : Type ] : Adt
      Constructor
         bottom : Carrier;
         up( down ) : X -> Carrier
   End lift"
      )
  | Isa_mode -> 
  "
   Begin list[ X : Type ] : Adt
      Constructor
         Nil : Carrier;
         Cons( car, cdr ) : [X, Carrier] -> Carrier
   End list"


let ccsl_prelude = function
  | Pvs_mode -> 
      (if !old_lift 
       then
  "
   Begin Lift[ X : Type ] : Adt
      Constructor
         bot : Carrier;
         up( down ) : X -> Carrier
   End Lift
  "
       else ""
      ) ^
  "
   Begin Coproduct[ X : Type, Y : Type ] : Adt
      Constructor
         in1(out1) : X -> Carrier;
         in2(out2) : Y -> Carrier;
   End Coproduct
   
   Begin Unit : Adt
      Constructor
         unit : Carrier;
   End Unit
  " ^
      (if !pedantic_mode
       then ""
       else
  "
   Begin PowerSig[X : (?,0) Type] : Groundsignature
     Importing PowerDefs2
     Type Power
     Constant
       emptyset : Power[X];
       member : [X, Power[X]] -> bool;
       pred : Power[X] -> [X -> bool]
   End PowerSig
  "
      ) ^
  "
   Begin FPowerSig[X : (?,0) Type] : Groundsignature
     Importing FPowerDefs2
     Type FPower
     Constant
       femptyset : FPower[X];
       fmember : [X, FPower[X]] -> bool;
       fpred : FPower[X] -> [X -> bool]
   End FPowerSig
  "
  | Isa_mode ->
      (if !old_lift 
       then
  "
   Begin Lift[ X : Type ] : Adt
      Constructor
         bot : Carrier;
         up( down ) : X -> Carrier
   End Lift
  "
       else
  "
   Begin lift[ X : Type ] : Adt
      Constructor
         bottom : Carrier;
         up( down ) : X -> Carrier
   End lift
  "
      ) ^
  "
   Begin Coproduct[ X : Type, Y : Type ] : Adt
      Constructor
         in1(out1) : X -> Carrier;
         in2(out2) : Y -> Carrier;
   End Coproduct
   
   Begin Unit : Adt
      Constructor
         unit : Carrier;
   End Unit

   Begin PowerSig[X : (?,0) Type] : Groundsignature
     Type Power
     Constant
       empty : Power[X];
       member : [X, Power[X]] -> bool;
       pred : Power[X] -> [X -> bool]
   End PowerSig
  ";;


let output_prelude () =
  print_endline "PVS prelude:";
  print_endline (builtin_prelude Pvs_mode);
  print_endline (ccsl_prelude Pvs_mode);
  print_endline "Isabelle/HOL prelude:";
  print_endline (builtin_prelude Isa_mode);
  print_endline (ccsl_prelude Isa_mode);
  exit 0


let power_id_rec = ref None
let fpower_id_rec = ref None
let done_set_power_types = ref false


let set_power_types () = 
  assert(Symbol.nesting_size() = 1);
  assert(not !done_set_power_types);
  (try
     (match Symbol.find name_of_power_type with
	| CCSL_GroundTypeSymb(id) -> power_id_rec := Some id
	| _ -> ()
     )
   with
     | Table.Not_defined -> ()
  );
  (try
     (match Symbol.find name_of_fpower_type with
	| CCSL_GroundTypeSymb(id) -> fpower_id_rec := Some id;
	| _ -> ()
     )
   with
     | Table.Not_defined -> ()
  );
  done_set_power_types := true


let is_power_type typ = 
  assert(!done_set_power_types);
  match typ,!power_id_rec with
    | Groundtype(tid,_), Some(pid) -> tid == pid
    | _ -> false


let is_fpower_type typ = 
  assert(!done_set_power_types);
  match typ,!fpower_id_rec with
    | Groundtype(tid,_), Some(pid) -> tid == pid
    | _ -> false

let is_fpower_sig si = 
  assert(!done_set_power_types);
  match !fpower_id_rec with
    | Some id -> (match id.id_origin with
		    | CCSL_GroundTypeDecl fsi -> fsi == si
		    | _ -> assert false
		 )
    | _ -> false

let is_any_power_type typ =
  assert(!done_set_power_types);
  match typ,!power_id_rec,!fpower_id_rec with
    | Groundtype(tid,_), Some(pid), Some(fpid) -> tid == pid || tid == fpid
    | _ -> false




(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
