(*
 * The LOOP Project
 *
 * The LOOP Team, Dresden University and Nijmegen University
 *
 * Copyright (C) 2002
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the
 * parent directories for more details.
 *
 * Created 14.5.99 by Hendrik
 *
 * Time-stamp: <Monday 20 October 03 16:21:23 tews@ithif51.inf.tu-dresden.de>
 *
 * symboltable instanciation for ccsl
 *
 * $Id: symbol.ml,v 1.15 2003-10-20 15:57:05 tews Exp $
 *
 *)

open Util
open Top_variant_types
open Classtypes
open Types_util
;;

(***********************************************************************
 *
 * Types are defined in classtypes.ml
 *
 * 
 * type ccsl_symbol_table_type =
 *     (top_name_space, ccsl_shape_type, ccsl_symbol_type) Table.global
 * 
 * type ccsl_scope_type =
 *     (top_name_space, ccsl_shape_type, ccsl_symbol_type) Table.local
 *)


(***********************************************************************
 *
 * create the global symboltable
 *
 *)

let ccsl_shape_equal s1 s2 = match s1,s2 with
  | No_overloading,No_overloading -> true
	(* Overloading not allowed in ccsl *)
  | Instanciation args1, Instanciation args2 ->
      (List.length args1) = (List.length args2)
      && List.for_all2 eq_ccsl_args args1 args2
  | No_overloading, _ 
  | Instanciation _,_
	 -> assert(false)


let ccsl_gst = ((Table.new_table (ccsl_shape_equal)) : ccsl_symbol_table_type)


(***********************************************************************
 *
 * convenience functions for the symboltable
 * redefine the stuff from table.ml 
 *
 *)


(* see comments in symbol.mli *)

let create name = Table.create ccsl_gst CCSL_ID name No_overloading

let new_local () = Table.new_local ccsl_gst

let create_local local name = 
  Table.local_overload local CCSL_ID name No_overloading

let find = Table.find ccsl_gst CCSL_ID

let find_all = Table.find_all ccsl_gst CCSL_ID

    (*******************************************************************
     *
     * finding entries in local tables
     *)

let find_local block = Table.find_local block CCSL_ID

let delete_local block name = 
  Table.delete_local_overloaded block CCSL_ID name No_overloading


    (*******************************************************************
     *
     * managing scopes
     *)

let start_block () = Table.start_block ccsl_gst

let end_of_defs () = Table.end_of_defs ccsl_gst

let close_block () = Table.close_block ccsl_gst

let reset_gst keep = Table.reset_gst ccsl_gst keep

let open_block = Table.open_block ccsl_gst

let nesting_size () = Table.nesting_size ccsl_gst


    (*******************************************************************
     *
     * miscellany
     *)

let symbol_table_iter = Table.iter ccsl_gst

let symbol_table_dump = Table.dump ccsl_gst

let dump_scope = Table.dump_local



(***********************************************************************
 *
 * create entries in the symboltable
 *
 *)

(* creation function for identifier records:
 * there are several caveats:
 * - it is not a consistent initialization, it only does the things, 
 *   which always have to be initialized in ccsl,
 *   for instance, if record describes a value parameter, then the 
 *   type field is updated afterwards
 *)
let identifier_record token origin =
  { id_token  = token;
    id_type = TypeConstant("Irrelevant field", Never, []);
    id_parameters = [];
    id_origin = origin;
    id_variance = Unset;
    id_sequence = -1;
    id_components = [];
  }


let create_ground_type id_rec =
  create id_rec.id_token.token_name (CCSL_GroundTypeSymb(id_rec))

let create_adt adt =
  create adt#get_name (AdtSymb adt)

let create_class cl = 
  create cl#get_name (ClassSymb cl)

let create_sig si =
  create si#get_name (SigSymb si)

let create_member member = 
  create member#get_name (MemberSymb member)

let create_adt_content adt =
  let do_adt_member const =
    begin
      create_member const;
    end
  in
    begin
      List.iter do_adt_member (adt#get_members);
      create_adt adt
    end

let create_class_content cl =
  let do_member mem = 
    if mem#is_constructor || mem#is_class_method then
      create_member mem
  in
    begin
					(* include constructors and Coreduce *)
      List.iter do_member (cl#get_members);
      create_class cl
    end


let create_sig_content si =
  let do_sig_type typ = create_ground_type typ in
  let do_sig_member const = create_member const
  in
    begin
      List.iter do_sig_type (si#get_all_ground_types);
      List.iter do_sig_member (si#get_members);
      create_sig si
    end


let create_type_parameter id_rec = 
  create id_rec.id_token.token_name (TypevarSymb id_rec)

let create_var id_rec =
  create id_rec.id_token.token_name (VarSymb id_rec)


let create_var_list decls = 
  let token name = {token_name = name;
		    loc = None
		   }
  in
    List.iter (fun (name,typ) ->
		 let id_rec = identifier_record (token name) CCSL_Var in
		   id_rec.id_type <- typ;
		   create_var id_rec
	      )
      decls



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)

