;;;   loop-compile.el --- loop-compile code commands for Emacs

;; Copyright (C) 1998 KUN.
;;
;; Authors:     1998 Wim Janssen KUN (wim@cs.kun.nl)
;; Created:     1998/09/22
;; Version:     0.4
;; Modified:    1999/09/13
;; Keywords:    Unix languages
;;
;; This file is part of LOOP.
;;
;; LOOP stands for Logic of Object-Oriented Programming

;;;   Commentary:

;; This package provides the `loop-compile' extension to ccsl and java.
;; It also offers some options for the loop-compiler.
;;
;; To use loop-compile together with ccsl-mode, all what is necessary
;; is that emacs finds this file via its load-path (which is the case
;; if you followed the installation instructions in ccsl-mode.el).
;; 
;; For java-mode follow these instructions:
;; To setup `loop-compile' put this file in the directory `site-lisp'
;; or put it in some directory and extend the `load-path' by adding the
;; following lines to your `.emacs' file.
;; The directory `~/Emacs/' is used in the sample code.
;;
;;   ;; Extend `load-path' for the `loop-compile' package.
;;   (setq load-path
;;     (cons (concat (expand-file-name "~") "/Emacs/") load-path))
;;
;; Then add hooks for java-mode.
;;
;;   (add-hook 'java-mode-hook 'loop-setup-java)
;;   (autoload 'loop-setup-java "loop-compile")
;;
;; You can preset some user definable variables in your `.emacs' file.
;;
;;   ;; Examples of user definable variables
;;
;;   ;; From `loop-compile.el'.
;;
;;   ;; The file name of the loop-compiler.
;;   ;; Default is "run"
;;   (setq loop-compiler "~/bin/run")
;;   ;;
;;   ;; The output directory used by the loop-compiler.
;;   ;; Default is nil
;;   (setq loop-output-directory "~/output/")
;;
;; End of example code to insert in your .emacs file.


;;;   Code:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; user definable variables                                  ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(defvar loop-compiler "ccslc"
  "*The file name of the loop-compiler.")

(defvar pvs-output-mode nil
  "*The PVS output Mode option of the loop-compiler.")

(defvar isa-output-mode nil
  "*The Isabelle output Mode option of the loop-compiler.")

(defvar loop-class nil
  "*The class used by the loop-compiler.")

(defvar loop-output-directory nil
  "*The output directory used by the loop-compiler.")

(defvar loop-output-file nil
  "*The output-file used by the loop-compiler.")

(defvar loop-jml-mode nil
  "*The jml option of the loop-compiler (java only).")

(defvar loop-verbose nil
  "*The verbose option of the loop-compiler.")

(defvar loop-enable-debug-options nil
  "*Enable the debug options in the `Compile Options' menu.")

(defvar loop-debug-lexer nil
  "*The debug lexer option of the loop-compiler.")

(defvar loop-debug-parser nil
  "*The debug parser option of the loop-compiler.")

(defvar loop-debug-resolution nil
  "*The debug resolution option of the loop-compiler.")

(defvar loop-debug-inheritance nil
  "*The debug inheritance option of the loop-compiler.")

(defvar loop-debug-typecheck nil
  "*The debug typecheck option of the loop-compiler.")

(defvar loop-print-syntax-tree nil
  "*The debug syntax-tree option of the loop-compiler.")

(defvar loop-print-symbol-table nil
  "*The debug symbol-table option of the loop-compiler.")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; no user definable variables beyond this point             ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;   Setup Ccsl
;;;
(defun loop-setup-ccsl ()
  "This function adds `loop-compile' capabilities to `ccsl-mode'.

To setup `loop-compile' capabilities to `ccsl-mode' add the
following lines to your `.emacs' file:

(add-hook 'ccsl-mode-hook 'loop-setup-ccsl)
(autoload 'loop-setup-ccsl \"loop-compile\")"
  ;; Set the language used by the loop-compiler.
  (loop-set-language ccsl-language)
  ;; Construct an initial compile command.
  (loop-set-compile-command)
  ;; Add menu buttons to the appropriate menu.
  (loop-add-menu-buttons ccsl-language)
  ;; Setup a key for fast error searching in PVS buffers.
  (if (not (featurep 'pvs-mode))
      (message "Not running PVS!!")
    (define-key pvs-mode-map "\C-c\C-c" 'loop-pvs-error-to-source))
  ;; Add error recognition expression for the loop-compiler.
  (setq compilation-error-regexp-alist
	(append
	 compilation-error-regexp-alist
	 ;;'(("warning in \\(.*\\): line \\(-?[0-9]+\\)" 1 2))
	 '(("error in \\(.*\\): line \\(-?[0-9]+\\)" 1 2)))))

;;;   Setup Java
;;;
(defun loop-setup-java ()
  "This function adds `loop-compile' capabilities to `java-mode'.

To setup `loop-compile' capabilities to `java-mode' add the
following lines to your `.emacs' file:

(add-hook 'java-mode-hook 'loop-setup-java)
(autoload 'loop-setup-java \"loop-compile\")"
  ;; Set the language used by the loop-compiler.
  (loop-set-language java-language)
  ;; Construct an initial compile command.
  (loop-set-compile-command)
  ;; Add menu buttons to the appropriate menu.
  (loop-add-menu-buttons java-language)
  ;; Setup a key for fast error searching in PVS buffers.
  (if (not (featurep 'pvs-mode))
      (message "Not running PVS!!")
    (define-key pvs-mode-map "\C-c\C-c" 'loop-pvs-error-to-source))
  ;; Add error recognition expression for the loop-compiler.
  (setq compilation-error-regexp-alist
	(append
	 compilation-error-regexp-alist
	 ;;'(("warning in \\(.*\\): line \\(-?[0-9]+\\)" 1 2))
	 '(("error in \\(.*\\): line \\(-?[0-9]+\\)" 1 2)))))


;;;   Set Language
;;;
(defvar loop-language nil
  "The language used by the loop-compiler.

At the moment the Values are `ccsl-language' and `java-language'.
The values are set according to the filename extension:
<file>.beh - ccsl-language and <file>.java - java-language.")

(defvar ccsl-language "ccsl-language"
  "")

(defvar java-language "java-language"
  "")

(defun loop-set-language (language)
  ""
  (setq loop-language language)
  (loop-set-compile-command))

;;;   Set Debug Level
;;;
(defvar loop-debug nil
  "")

(defvar loop-debug-level 0
  "")

(defun loop-reset-debug ()
  (setq loop-debug nil)
  (setq loop-debug-level 0))

(defun loop-set-debug (level)
  (setq loop-debug t)
  (setq loop-debug-level (+ loop-debug-level level)))

;;;   Set Compile Command
;;;
(defun loop-set-compile-command ()
  ""
  (loop-reset-debug)
  (if loop-verbose (loop-set-debug 1))
  (if loop-debug-lexer (loop-set-debug 2))
  (if loop-debug-parser (loop-set-debug 4))
  (if loop-debug-resolution (loop-set-debug 8))
  (if loop-debug-inheritance (loop-set-debug 16))
  (if loop-debug-typecheck (loop-set-debug 32))
  (if loop-print-syntax-tree (loop-set-debug 64))
  (if loop-print-symbol-table (loop-set-debug 128))
  (setq compile-command
	(concat
	 (if loop-compiler
	     (concat loop-compiler " "))
	 (if pvs-output-mode "-pvs ")
	 (if isa-output-mode "-isa ")
	 (if loop-class
	     (concat "-class " loop-class " "))
	 (if loop-output-directory
	     (concat "-d " loop-output-directory " "))
	 (if loop-output-file
	     (concat "-o " loop-output-file " "))
	 (if loop-jml-mode "-jml ")
	 ;;(if loop-verbose "-v ")
	 ;;(if loop-debug-lexer "-D 2 ")
	 ;;(if loop-debug-parser "-D 4 ")
	 ;;(if loop-debug-resolution "-D 8 ")
	 ;;(if loop-debug-inheritance "-D 16 ")
	 ;;(if loop-debug-typecheck "-D 32 ")
	 ;;(if loop-print-syntax-tree "-D 64 ")
	 ;;(if loop-print-symbol-table "-D 128 ")
	 (if loop-debug
	     (concat "-D " loop-debug-level " "))
	 (file-name-nondirectory buffer-file-name))))


;;;   Add Menu Buttons
;;;
;; FIXME
;; I would have liked to put the implementation of `loop-add-menu-buttons'
;; next. But this routine depends heavily on the version of emacs
;; you are running and on the language chosen. The way menu's are
;; implemented in different version of emacs is (IMHO) to blame.
;; Also I don't have enough experience with emacs and lisp to solve
;; this the right way. This of course means I am to blame most.
;; If anybody knows how to do this the right way, please send an
;; email with the solution to wim@cs.kun.nl.
;; FIXME

;;;   Add Menu Buttons
;;;
(defun loop-add-menu-buttons (language)
  "Add Loop menu buttons to the menubar.

This is used by `loop-compile' for automatic reading and checking the
generated pvs files.
This is REALLY ugly, But I don't have enough experience with emacs
and lisp to solve this the right way. If anybody knows how to do
this the right way, please send an email to wim@cs.kun.nl."
  (cond
   ;; Xemacs has feature `menubar'.
   ((featurep 'menubar)
    (cond
     ;; ccsl-mode in xemacs
     ((string= language ccsl-language)
      (define-key ccsl-mode-map [(control c) \c] 'loop-compile)
      (define-key ccsl-mode-map [(control c) \t] 'loop-typecheck)
      (define-key ccsl-mode-map [(control c) \s] 'loop-typecheck-file)
      (define-key ccsl-mode-map [(control c) \x] 'loop-compile-typecheck)
      (add-menu-button '("Loop") ["Compile" loop-compile t] "")
      (add-menu-button '("Loop") ["Typecheck" loop-typecheck t] "")
      (add-menu-button '("Loop") ["Typecheck File" loop-typecheck-file t] "")
      (add-menu-button '("Loop") ["Compile Typecheck" loop-compile-typecheck t] "")
      (add-menu-button '("Loop") loop-compile-options-menu ""))
      ;; (add-menu-button '("CCSL") ["--" nil t] "")
      ;; (add-menu-button '("CCSL") ["Loop Compile" loop-compile t] "")
      ;; (add-menu-button '("CCSL") ["Loop Typecheck" loop-typecheck t] "")
      ;; (add-menu-button '("CCSL") ["Loop Typecheck file" loop-typecheck-file t] "")
      ;; (add-menu-button '("CCSL") ["Loop Compile Typecheck" loop-compile-typecheck t] "")
      ;; (add-menu-button '("CCSL") loop-compile-options-menu ""))
     ;; java-mode in xemacs
     ((string= language java-language)
      (define-key java-mode-map [(control c) \c] 'loop-compile)
      (define-key java-mode-map [(control c) \t] 'loop-typecheck)
      (define-key java-mode-map [(control c) \s] 'loop-typecheck-file)
      (define-key java-mode-map [(control c) \x] 'loop-compile-typecheck)
      (add-menu-button '("Loop") ["Compile" loop-compile t] "")
      (add-menu-button '("Loop") ["Typecheck" loop-typecheck t] "")
      (add-menu-button '("Loop") ["Typecheck File" loop-typecheck-file t] "")
      (add-menu-button '("Loop") ["Compile Typecheck" loop-compile-typecheck t] "")
      (add-menu-button '("Loop") loop-compile-options-menu ""))
      ;; (add-menu-button '("Java Mode Commands") ["--" nil t] "")
      ;; (add-menu-button '("Java Mode Commands") ["Loop Compile" loop-compile t] "")
      ;; (add-menu-button '("Java Mode Commands") ["Loop Typecheck" loop-typecheck t] "")
      ;; (add-menu-button '("Java Mode Commands") ["Loop Typecheck File" loop-typecheck-file t] "")
      ;; (add-menu-button '("Java Mode Commands") ["Loop Compile Typecheck" loop-compile-typecheck t] "")
      ;; (add-menu-button '("Java Mode Commands") loop-compile-options-menu ""))
     (t (message "No language found!!"))))
   ;; gnu-emacs has feature 'menu-bar'.
   ((featurep 'menu-bar)
    (cond
     ;; ccsl-mode in emacs
     ((string= language ccsl-language)
      (define-key ccsl-mode-map "\C-cc" 'loop-compile)
      (define-key ccsl-mode-map "\C-ct" 'loop-typecheck)
      (define-key ccsl-mode-map "\C-cs" 'loop-typecheck-file)
      (define-key ccsl-mode-map "\C-cx" 'loop-compile-typecheck)
      (let ((map (make-sparse-keymap "Loop")))
	(define-key ccsl-mode-map [menu-bar loop] (cons "Loop" map))
	(define-key map [compile-options] (cons "Compile Options" loop-compile-options-menu))
	(define-key map [compile-typecheck] '("Compile Typecheck" . loop-compile-typecheck))
	(define-key map [typecheck-file] '("Typecheck File" . loop-typecheck-file))
	(define-key map [typecheck] '("Typecheck" . loop-typecheck))
	(define-key map [compile] '("Compile" . loop-compile))))
     ;; java-mode in emacs
     ((string= language java-language)
      (define-key java-mode-map "\C-cc" 'loop-compile)
      (define-key java-mode-map "\C-ct" 'loop-typecheck)
      (define-key java-mode-map "\C-cs" 'loop-typecheck-file)
      (define-key java-mode-map "\C-cx" 'loop-compile-typecheck)
      (let ((map (make-sparse-keymap "Loop")))
	(define-key java-mode-map [menu-bar loop] (cons "Loop" map))
	(define-key map [compile-options] (cons "Compile Options" loop-compile-options-menu))
	(define-key map [compile-typecheck] '("Compile Typecheck" . loop-compile-typecheck))
	(define-key map [typecheck-file] '("Typecheck File" . loop-typecheck-file))
	(define-key map [typecheck] '("Typecheck" . loop-typecheck))
	(define-key map [compile] '("Compile" . loop-compile))))
     (t (message "No language found!!"))))
   (t (message "No menubar found!!"))))


;;;   Compile menu button
;;;
(defun loop-compile ()
  ""
  (interactive)
  (loop-set-compile-command)
  (call-interactively 'compile))

;;;   Typecheck menu button
;;;
(defun loop-typecheck ()
  ""
  (interactive)
  (loop-select-all-pvs-files))

;;;   Typecheck File menu button
;;;
(defun loop-typecheck-file ()
  ""
  (interactive)
  (loop-select-one-pvs-file))

;;;   Compile Typecheck menu button
;;;
(defvar loop-current-buffer nil
  "Storage for the current buffer.")

(defun loop-compile-typecheck ()
  ""
  (interactive)
  (loop-compile)
  ;; Because the compile process runs asynchronous, the sequential
  ;; solution doesn't work.
  ;; (if (eq compile-error-list 'nil)
  ;;     (loop-typecheck))
  ;;   (message "There are errors.")
  ;; The check for errors is executed before the compile process has
  ;; terminated. After some searching up and down the source code I
  ;; discovered the variable `compilation-finish-function' that looked
  ;; very promising.
  ;; Trying this solution I discovered that the parse for errors also
  ;; runs asynchronous, that is when the mouse enters the compile window.
  ;; Again searching the source code I decided to run the parse for errors
  ;; in the function bound to the variable `compilation-finish-function'.
  ;; Because this function runs in the compilation buffer saving the
  ;; current buffer was needed to be able to switch back to the current
  ;; buffer when calling `loop-typecheck'.
  (setq loop-current-buffer (current-buffer))
  (setq compilation-finish-function 'loop-continue-typecheck))

(defun loop-continue-typecheck (buffer msg)
  ""
  ;; Unregister the `compilation-finish-function'. This way
  ;; the command "Loop Compile" doesn't start a `loop-typecheck'.
  (setq compilation-finish-function nil)
  (save-excursion
    ;; Force parsing the compilation buffer for errors.
    (compile-reinitialize-errors t)
    (if (eq compilation-error-list 'nil)
	(progn
	  ;; Just to be sure, switch back to the saved buffer.
	  (set-buffer loop-current-buffer)
	  (loop-typecheck))
      ;; Just to be sure, switch back to the saved buffer.
      (set-buffer loop-current-buffer)
      (message "There are errors."))))


;;;   Compile Options menu button
;;;
(defvar loop-compile-options-menu nil
  "loop-compile-options-menu used in ccsl/java-mode buffers.")

(if loop-compile-options-menu
    ()
  (easy-menu-define
   loop-compile-options-menu nil "Compile Options"
   ;; '("Loop Compile Options"
   '("Compile Options"
     :filter file-menu-filter
     ["Select Compiler ..." (loop-select-compiler loop-compiler) t]
     "-"
     ["PVS" (loop-toggle 'pvs-output-mode)
      :style toggle
      :selected pvs-output-mode]
     ["Isabelle" (loop-toggle 'isa-output-mode)
      :style toggle
      :selected isa-output-mode]
     "-"
     ["Select Class ..." (loop-select-class loop-class) t]
     ["Select Output Directory ..." (loop-select-output-directory loop-output-directory) t]
     ["Select Output File ..." (loop-select-output-file loop-output-file) t]
     ["JML (Java Only)" (loop-toggle 'loop-jml-mode)
      :style toggle
      :selected loop-jml-mode]
     ;;
     ;; Debug options in "Loop Compile Options" menu.
     ;;
     "-"
     ["Verbose" (loop-toggle 'loop-verbose)
      :style toggle
      :selected loop-verbose]
     ["Debug Lexer" (loop-toggle 'loop-debug-lexer)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-debug-lexer]
     ["Debug Parser" (loop-toggle 'loop-debug-parser)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-debug-parser]
     ["Debug Resolution" (loop-toggle 'loop-debug-resolution)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-debug-resolution]
     ["Debug Inheritance" (loop-toggle 'loop-debug-inheritance)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-debug-inheritance]
     ["Debug Typecheck" (loop-toggle 'loop-debug-typecheck)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-debug-typecheck]
     ["Print Syntax Tree" (loop-toggle 'loop-print-syntax-tree)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-print-syntax-tree]
     ["Print Symbol Table" (loop-toggle 'loop-print-symbol-table)
      :config loop-enable-debug-options
      :style toggle
      :selected loop-print-symbol-table])))

;;;   Select Compiler
;;;
(defun loop-select-compiler (compiler)
  ""
  (let ((use-dialog-box nil)
	(insert-default-directory nil))
    (setq loop-compiler
	  (read-file-name
	   "Select Compiler: "
	   "" loop-compiler nil loop-compiler)))
  (if (string= loop-compiler "")
      (setq loop-compiler nil))
  (loop-set-compile-command))

;;;   Select Class
;;;
(defun loop-select-class (class)
  ""
  (setq loop-class
	(read-string
	 "Class: "
	 (if (string= (current-word) "")
	     loop-class
	   (current-word))))
  (if (string= loop-class "")
      (setq loop-class nil))
  (loop-set-compile-command))

;;;   Select Output Directory
;;;
(defun loop-select-output-directory (directory)
  ""
  (let ((use-dialog-box nil)
	(insert-default-directory nil))
    (setq loop-output-directory
	  (read-file-name
	   "Select Output Directory: "
	   "" "" nil loop-output-directory)))
  (if (string= loop-output-directory "")
      (setq loop-output-directory nil)
    (unless (string-match "/$" loop-output-directory)
      (setq loop-output-directory (concat loop-output-directory "/"))))
  (loop-set-compile-command))

;;;   Select Output File
;;;
(defun loop-select-output-file (file)
  ""
  (let ((use-dialog-box nil)
	(insert-default-directory nil))
    (setq loop-output-file
	  (read-file-name
	   "Select Output File: "
	   "" "" nil loop-output-file)))
  (if (string= loop-output-file "")
      (setq loop-output-file nil))
  (loop-set-compile-command))

;;;   Toggle
;;;
(defun loop-toggle (flag)
  "Boolean toggle FLAG's value.

FLAG must be a bound symbol.  Nil values change to t,
non-nil values are changed to nil."
  (set-variable flag (not (eval flag)))
  (loop-set-compile-command))


;;;   PVS file and PVS directory
;;;
(defvar loop-pvs-file nil
  "")

(defvar loop-pvs-directory nil
  "")

;;;   Select one PVS file
;;;
(defun loop-select-one-pvs-file ()
  "Select and typecheck one file from the generated PVS files with names
constructed from the contents of the current buffer.

This is used by `loop-compile' to automatically load and typecheck the
PVS files generated from all the class and adt definitions in the current
buffer and prepare for proving."
  (interactive)
  (let ((names (loop-scan-buffer)))
    (setq loop-pvs-file
	  (completing-read
	   (concat "Select PVS File: (default " (loop-nearest-match names) ") ")
	   names nil t))
    ;; Some dirty way to provide the default value.
    (if (string= loop-pvs-file "")
	(setq loop-pvs-file (loop-nearest-match names))
      ()))
  ;; Check if we are in PVS.
  (if (not (featurep 'pvs-mode))
      (message "Not running PVS!!")
    ;; Assume the buffer was `loop-compile'ed using the current
    ;; `loop-output-directory'.
    (if (eq loop-output-directory nil)
	;; Derive PVS context from buffer-file-name.
        (if (string= (pathname-directory (buffer-file-name)) loop-pvs-directory)
	    ;; Context switch already performed, no need to repeat.
	    ()
	  (setq loop-pvs-directory (pathname-directory (buffer-file-name)))
	  (init-change-context (pathname-directory loop-pvs-directory)))
      ;; Change PVS context to the output directory.
      (if (string= loop-output-directory loop-pvs-directory)
	  ;; Context switch already performed, no need to repeat.
	  ()
	;; Change PVS context to the `loop-output-directory'.
	(setq loop-pvs-directory loop-output-directory)
	(init-change-context (pathname-directory loop-pvs-directory))))
    ;; Typecheck the selected PVS file.
    ;; FIXME
    ;; I should check the existence of files named by `(car (car names))'.
    ;; FIXME
    (pvs-typecheck-file (pathname-name loop-pvs-file) nil nil 'typecheck)))

;;;   Select all PVS files
;;;
(defun loop-select-all-pvs-files ()
  "Select and typecheck all files from the generated PVS files with names
constructed from the contents of the current buffer.

This is used by `loop-compile' to automatically load and typecheck the
PVS files generated from all the class and adt definitions in the current
buffer and prepare for proving."
  (interactive)
  (let ((names (loop-scan-buffer)))
    ;; Check if we are in PVS.
    (if (not (featurep 'pvs-mode))
	(message "Not running PVS!!")
      ;; Assume the buffer was `loop-compile'ed using the current
      ;; `loop-output-directory'.
      (if (eq loop-output-directory nil)
	  ;; Derive PVS context from buffer-file-name.
          (if (string= (pathname-directory (buffer-file-name)) loop-pvs-directory)
	      ;; Context switch already performed, no need to repeat.
	      ()
	    (setq loop-pvs-directory (pathname-directory (buffer-file-name)))
	    (init-change-context (pathname-directory loop-pvs-directory)))
	;; Change PVS context to the output directory.
	(if (string= loop-output-directory loop-pvs-directory)
	    ;; Context switch already performed, no need to repeat.
	    ()
	  ;; Change PVS context to the `loop-output-directory'.
	  (setq loop-pvs-directory loop-output-directory)
	  (init-change-context (pathname-directory loop-output-directory))))
      (while names
	;; FIXME
	;; I should check the existence of files named by `(car (car names))'.
	;; FIXME
	(pvs-typecheck-file (pathname-name (car (car names))) nil nil 'typecheck)
	(setq names (cdr names))))))

(defun loop-scan-buffer ()
  "Scans the current buffer and returns an alist of names.

Scan the current buffer for 'class' like names."
  (let
      ((case-fold-search t))
    
  (cond
   ;; Search for prelude, adt-names, class names,
   ;; depending on the language used.
   ((string= loop-language ccsl-language)
    (remove
     nil
     (append
      ;; Start with semantic prelude.
      (list (cons "ccsl_prelude.pvs" 0))
      ;; Continue with generated files.
      (if (not (eq loop-output-file nil))
	  ;; There is one output file generated.
	  ;; The name to be used is derived from `loop-output-file'.
	  (list (cons (concat loop-output-file "_basic.pvs") 1))
	;; There are multiple output files generated.
	;; The names to be used are derived from the buffer content.
	(append
	 (loop-get-names-from-buffer
	  "\\([a-zA-Z][a-zA-Z0-9_]*\\)[ \t]*:[ \t]*adt" 1 "_basic.pvs")
	 (loop-get-names-from-buffer
	  "\\([a-zA-Z][a-zA-Z0-9_]*\\)[ \t]*:[ \t]*classspec" 1 "_basic.pvs")
	 (loop-get-names-from-buffer
	  "\\([a-zA-Z][a-zA-Z0-9_]*\\)[ \t]*\\[.*:[ \t]*adt" 1 "_basic.pvs")
	 (loop-get-names-from-buffer
	  "\\([a-zA-Z][a-zA-Z0-9_]*\\)[ \t]*\\[.*:[ \t]*classspec"
	     1 "_basic.pvs"
	 ))))))
   ((string= loop-language java-language)
    (remove
     nil
     (append
      ;; Start with semantic prelude.
      (list (cons "JavaADT.pvs" 0))
      (list (cons "JavaStatement.pvs" 0))
      (list (cons "JavaExpression.pvs" 0))
      (list (cons "JavaBasicOperations.pvs" 0))
      (list (cons "JavaObjectMemory.pvs" 0))
      (list (cons "JavaException.pvs" 0))
      (list (cons "JavaAdvancedOperations.pvs" 0))
      (list (cons "JavaArray.pvs" 0))
      (list (cons "JavaPrelude.pvs" 0))
      ;; Continue with API classes.
      ;; FIXME more to be added.
      (list (cons "java_lang_Object_basic.pvs" 0))
      ;; Continue with generated files.
      (if (not (eq loop-output-file nil))
	  ;; There is one output file generated.
	  ;; The name to be used is derived from `loop-output-file'.
	  (list (cons (concat loop-output-file "_basic.pvs") 1))
	;; There are multiple output files generated.
	;; The names to be used are derived from the buffer content.
	(append
	 (loop-get-names-from-buffer
	  "class[ \t]*\\([a-zA-Z][a-zA-Z0-9_]*\\)" 1 "_basic.pvs"))))))
   (t
    nil))))

(defun loop-get-names-from-buffer (names-def-regexp group post)
  "Scans the current buffer and return an alist of names.

Uses STRING `names-def-regexp' to scan the current buffer for names.
The match-group INTEGER `group' is added to the list of names found
to which the STRING `post' is appended."
  (let (names name)
    (save-restriction
      (save-excursion
	(widen)
	(goto-char (point-min))
	(while (re-search-forward names-def-regexp nil t)
	  (setq name
		(cons
		 (concat
		  (buffer-substring (match-beginning group) (match-end group))
		  post)
		 (count-lines (point-min) (point))))
	  (setq names (cons name names)))))
    ;; Reverse the `names' to provide the names in the
    ;; order found in the current buffer.
    (setq names (reverse names))))

(defun loop-nearest-match (names)
  "Look up in the current buffer the entry nearest to `point'.

The alist `names' is searched for an entry nearest to point."
  (let ((current-line-nr (count-lines (point-min) (point)))
	(dist (count-lines (point-min) (point-max)))
	(name "")
	(line-nr 0))
    (while names
      (setq line-nr (cdr (car names)))
      (if (or (= line-nr 0)
	      (< dist (abs (- line-nr current-line-nr))))
	  ()
	(setq dist (abs (- line-nr current-line-nr)))
	(setq name (car (car names))))
      (setq names (cdr names)))
    name))


;;;   PVS error to source file
;;;
(defun loop-pvs-error-to-source ()
  "An error occurred during typechecking a PVS file.

Find the buffer containing the source file and jump to the
line on which the error was found."
  (interactive)
  (let ((source-regexp "% source in \\(.*\\): line \\(-?[0-9]+\\)")
	(source-file nil)
	(source-line nil)
	(source-point 0)
	(error-point 0)
	(base-offset 0)
	(line-offset 0))
    (save-excursion
      (move-to-column 0)
      (setq error-point (point))
      (cond
       ;; First try to find a source descriptor seaching backward through
       ;; the buffer.
       ((re-search-backward source-regexp nil t)
	(setq source-file (buffer-substring (match-beginning 1) (match-end 1)))
	(setq source-line (buffer-substring (match-beginning 2) (match-end 2)))
	(move-to-column 0)
	(setq source-point (point))
	;; The search was backward, line offset is positive.
	;; To get a consistent line count it was necessary to move both
	;; `source-point' and `error-point' to the beginning of the line.
	;; See also the documentation of `count-lines'.
	(setq line-offset
	      (+ base-offset (count-lines source-point error-point))))
       ;; Otherwise try to find a source descriptor seaching forward through
       ;; the buffer.
       ((re-search-forward source-regexp nil t)
	(setq source-file (buffer-substring (match-beginning 1) (match-end 1)))
	(setq source-line (buffer-substring (match-beginning 2) (match-end 2)))
	(move-to-column 0)
	(setq source-point (point))
	;; The search was forward, line offset is negative.
	;; To get a consistent line count it was necessary to move both
	;; `source-point' and `error-point' to the beginning of the line.
	;; See also the documentation of `count-lines'.
	(setq line-offset
	      (- base-offset (count-lines source-point error-point))))
       (t
	;; No source descriptor found, do nothing.
	())))
    (if (eq source-file 'nil)
	(message "No source descriptor found in the neighborhood of point!!")
      (if (not (file-exists-p source-file))
	  (message "File does not exist!!")
	(if (not (file-readable-p source-file))
	    (message "File is not readable!!")
	  (find-file-other-window source-file)
	  (widen)
	  (move-to-column 0)
	  (goto-line (+ (string-to-int source-line) line-offset)))))))


;;;   Always evaluated
;;;
(require 'cl)
(require 'compile)

;; Make at least `loop-language' a buffer-local variable to avoid problems
;; when switching between, for example, `ccsl-mode' and `java-mode' buffers.
;; I still have to think about other variables.
(make-variable-buffer-local 'loop-language)

(provide 'loop-compile)

;;;   loop-compile.el ends here
