(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: admissibility.v,v 1.15 2013/04/10 11:17:13 tews Exp $
 *)

(** ** Admissible rules

      This module defines the concept of an admissible rule and
      generlizes it to sets of rules.
*)

Require Export rules.

Section Admissibility.

  Variable V : Type.
  Variable L : modal_operators.

  Variable rules : set (sequent_rule V L).
  Variable hypotheses : set (sequent V L).


  (**************************************************************************)
  (** ***  admissibility, not depth preserving  *)
  (**************************************************************************)

  Definition admissible(r : sequent_rule V L) : Prop :=
    every_nth (provable rules hypotheses) (assumptions r) ->
      provable rules hypotheses (conclusion r).

  Definition admissible_rule_set(rs : set (sequent_rule V L)) : Prop :=
    forall(r : sequent_rule V L), rs r -> admissible r.

  Lemma admissible_prop : 
    forall(rs : set (sequent_rule V L))(s : sequent V L),
      admissible_rule_set rs ->
        (provable rules hypotheses s <->
           provable (union rs rules) hypotheses s).
  Proof.
    intros rs s H.
    split.
      intros H0.
      eapply proof_mono_rules.
        apply subset_union_right.
      trivial.
    intros H0.
    destruct H0.
    clear H0.
    induction x using proof_sequent_ind.
      exists (assume rules hypotheses gamma H0).
      trivial.
    destruct H0.
      apply H.
        trivial.
      trivial.
    assert (H2 := every_nth_exists _ _ H1).
    destruct H2.
    clear H2.
    exists (rule rules hypotheses r H0 x).
    trivial.
  Qed.


  (**************************************************************************)
  (** ***  admissibility, depth preserving  *)
  (**************************************************************************)

  Definition depth_preserving_admissible(r : sequent_rule V L) : Prop :=
    forall(n : nat),
      every_nth (provable_at_depth rules hypotheses n) (assumptions r) ->
        provable_at_depth rules hypotheses n (conclusion r).

  Definition depth_preserving_admissible_rule_set
                                       (rs : set (sequent_rule V L)) : Prop :=
    forall(r : sequent_rule V L), rs r -> 
      depth_preserving_admissible r.

  Lemma admissible_depth_preserving_admissible :
    forall(rs : set (sequent_rule V L)),
      depth_preserving_admissible_rule_set rs -> admissible_rule_set rs.
  Proof.
    unfold depth_preserving_admissible_rule_set, admissible_rule_set in *.
    intros rs H r H0.
    specialize (H _ H0).
    unfold depth_preserving_admissible, admissible in *.
    intros H1.
    assert (H2 := every_nth_exists _ _ H1).
    destruct H2.
    clear H2.
    lapply (H (nat_list_max 
                (dep_map_dep_const (proof_depth (hypotheses := hypotheses))
                   (assumptions r) x))).
      clear. 
      intros H.
      destruct H.
      exists x0.
      trivial.
    clear. 
    intros i i_less.
    unfold provable_at_depth in *.
    exists (dep_nth (assumptions r) x i i_less).
    rewrite nth_dep_map_dep_const_inv.
    apply nat_list_max_lub.
  Qed.


  Lemma delete_depth_admissible_rule :
    forall(r : sequent_rule V L)(n : nat)(s : sequent V L),
      depth_preserving_admissible r ->
      provable_at_depth (add r rules) hypotheses n s ->
        provable_at_depth rules hypotheses n s.
  Proof.
    intros r n s H H0.
    unfold provable_at_depth in H0.
    destruct H0.
    revert n H0.
    induction x.
      intros n H0.
      exists (assume rules hypotheses gamma in_hypotheses).
      trivial.
    intros n H1.
    destruct n.
      eapply proof_depth_0.
      eexact H1.
    lapply (every_nth_exists 
             (fun(s : sequent V L)(p2 : proof rules hypotheses s) => 
                     proof_depth p2 <= n)
             (assumptions r0)).
      intros H2.
      destruct H2.
      unfold add in in_rules.
      destruct in_rules.
        subst r0.
        unfold depth_preserving_admissible in *.
        apply H.
        intros i i_less.
        exists (dep_nth (assumptions r) x i i_less).
        apply le_S.
        apply H2.
      exists (rule rules hypotheses r0 r1 x).
      apply proof_depth_rule_le.
      trivial.
    intros i i_less.
    apply H0.
    eapply proof_depth_rule_le_inv.
    eexact H1.
  Qed.

  Lemma depth_admissible_prop : 
    forall(r : sequent_rule V L)(s : sequent V L)(n : nat),
      depth_preserving_admissible r ->
        (provable_at_depth rules hypotheses n s <->
           provable_at_depth (add r rules) hypotheses n s).
  Proof.
    intros r s n H.
    split.
      intros H0.
      eapply proof_depth_mono.
          apply subset_add.
        apply subset_refl.
      trivial.
    intros H0.
    eapply delete_depth_admissible_rule.
      eexact H.
    trivial.
  Qed.


End Admissibility.

Implicit Arguments admissible [V L].
Implicit Arguments admissible_rule_set [V L].
Implicit Arguments depth_preserving_admissible [V L].
Implicit Arguments depth_preserving_admissible_rule_set [V L].
