(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: build_proof.v,v 1.11 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Generic proof search

      This module defines a generic proof-search function that uses
      oracles for hypothesis and proofs.
*)

Require Export formulas.

Section Build_proof.
  Variable V : Type.
  Variable L : modal_operators.

  Variable rules : sequent_rule V L -> Prop.
  Variable hypotheses : sequent V L -> Prop.


  (***************************************************************************)
  (** *** Built proofs  *)
  (***************************************************************************)

  Definition hypotheses_oracle_type : Type :=
    forall(s : sequent V L), option (hypotheses s).

  Definition rule_oracle_result(s : sequent V L) : Type :=
    option (r # sequent_rule V L /#\ (rules r /\ r.(conclusion) = s)).

  Definition rule_oracle_type : Type :=
    forall(s : sequent V L), rule_oracle_result s.

  Fixpoint build_proof(i : nat)(ho : hypotheses_oracle_type)
                      (ro : rule_oracle_type)(s : sequent V L) : 
                                 (proof rules hypotheses s) + (sequent V L) :=
    match i with 
      | O => inr s
      | S i =>
        match ho s with
          | Some in_hyp =>
            inl (assume rules hypotheses s in_hyp)
          | None =>
            match ro s with
              | None => inr s
              | Some (dep_conj r (conj in_rules concl_prop)) =>
                match dep_list_proj_left r.(assumptions)
                      (dep_map_const_dep (build_proof i ho ro) r.(assumptions))
                with
                  | inr s => inr s
                  | inl subproofs => 
                      inl (eq_rect (conclusion r) 
                             (fun(s : sequent V L) => proof rules hypotheses s)
                             (rule rules hypotheses r in_rules subproofs)
                             s
                             concl_prop)
                end
            end
        end
    end.


  (***************************************************************************)
  (** *** Build proof error property  *)
  (***************************************************************************)

  Definition well_founded_rule(r : sequent_rule V L)
                                      (measure : sequent V L -> nat) : Prop :=
    forall(i : nat)(i_less : i < length (assumptions r)),
      measure (nth (assumptions r) i i_less) < measure (conclusion r).
    

  Definition well_founded_rule_oracle(ro : rule_oracle_type)
                                      (measure : sequent V L -> nat) : Prop :=
    forall(s : sequent V L),
      match ro s with
        | None => True
        | Some (dep_conj r _) => well_founded_rule r measure
      end.

  Lemma build_proof_right_result :
    forall(i : nat)(ho : hypotheses_oracle_type)
          (ro : rule_oracle_type)(measure : sequent V L -> nat)
          (s r : sequent V L),
      well_founded_rule_oracle ro measure -> 
      measure s < i ->
      build_proof i ho ro s = inr r ->
        ro r = None /\ ho r = None.
  Proof.
    induction i.
      intros ho ro measure s r H H0 H1.
      exfalso.
      omega.
    intros ho ro measure s r H H0 H1.
    simpl in H1.
    destruct (ho s) eqn:?.
      discriminate.
    destruct (ro s) eqn:?.
      destruct d.
      destruct a0.
      destruct (dep_list_proj_left (assumptions a)
                  (dep_map_const_dep (build_proof i ho ro) (assumptions a)))
               eqn:?.
        discriminate.
      assert (H2 := dep_list_proj_left_dep_map _ _ _ _ _ _ Heqs0).
      inversion H1; clear H1.
      subst s0.
      decompose [ex and or dep_and] H2; clear H2.
      eapply IHi; eauto.
      clear - H H0 Heqr0 e.
      specialize (H s).
      rewrite Heqr0 in *.
      specialize (H _ a0).
      subst s.
      omega.
    inversion H1.
    subst s.
    auto.
  Qed.


  (***************************************************************************)
  (** *** Rule upward induction  *)
  (***************************************************************************)

  Definition rule_inductive(P : sequent V L -> Prop) : Prop :=
    forall(r : sequent_rule V L), 
      rules r ->
      P (conclusion r) ->
      every_nth P (assumptions r).

  Lemma build_proof_right_property :
    forall(i : nat)(ho : hypotheses_oracle_type)
          (ro : rule_oracle_type)(measure : sequent V L -> nat)
          (P : sequent V L -> Prop)
          (s r : sequent V L),
      well_founded_rule_oracle ro measure -> 
      measure s < i ->
      rule_inductive P ->
      P s ->
      build_proof i ho ro s = inr r ->
        P r.
  Proof.
    induction i.
      intros ho ro measure P s r H H0 H1 H2 H3.
      omega.
    intros ho ro measure P s r H H0 H1 H2 H3.
    simpl in H3.
    destruct (ho s).
      inversion H3.
    destruct (ro s) eqn:?.
      destruct d.
      destruct a0.
      destruct (dep_list_proj_left (assumptions a)
                  (dep_map_const_dep (build_proof i ho ro) (assumptions a)))
               eqn:?.
        inversion H3.
      assert (H4 := dep_list_proj_left_dep_map _ _ _ _ _ _ Heqs0).
      inversion H3; clear H3.
      subst s0.
      decompose [ex and or dep_and] H4; clear H4.
      eapply IHi with (s := (nth (assumptions a) x a0)); eauto.
        clear - H H0 Heqr0.
        specialize (H s).
        rewrite Heqr0 in *.
        specialize (H _ a0).
        subst s.
        omega.
      clear - H1 H2 r0 e.
      apply H1; auto.
      rewrite e.
      trivial.
    inversion H3.
    subst s.
    trivial.
  Qed.
    
  Fixpoint restrict_hypothesis(P : sequent V L -> Prop)
           (ri : rule_inductive P)
           (s : sequent V L)(ps : P s)(p : proof rules hypotheses s)
                                : proof rules (intersection hypotheses P) s :=
    match p 
      in proof _ _ s 
      return P s -> proof rules (intersection hypotheses P) s
    with
      | assume s hyp_s => fun(ps : P s) => 
        assume rules (intersection hypotheses P) s (conj hyp_s ps)
      | rule r r_rules subproofs =>
        fun(ps : P (conclusion r)) =>
          rule rules (intersection hypotheses P) r r_rules
            ((fix map_subproofs(sl : list (sequent V L))
                  (subproofs : dep_list (sequent V L) 
                                        (proof rules hypotheses) sl)
                  (psl : every_nth P sl) 
                       : dep_list (sequent V L) 
                               (proof rules (intersection hypotheses P)) sl :=
              match subproofs
                in dep_list _ _ sl
                return 
                  every_nth P sl -> 
                     dep_list (sequent V L)
                              (proof rules (intersection hypotheses P)) sl
              with
                | dep_nil => fun _ => dep_nil
                | dep_cons s sl p tl => fun(psl : every_nth P (s :: sl)) =>
                  dep_cons s sl 
                    (restrict_hypothesis P ri s (every_nth_head _ _ _ psl) p)
                    (map_subproofs sl tl (every_nth_tail _ _ _ psl))
              end psl)
             (assumptions r) subproofs (ri r r_rules ps))
    end ps.


End Build_proof.


Implicit Arguments build_proof [V L rules hypotheses].
Implicit Arguments build_proof_right_result [V L rules hypotheses].
Implicit Arguments well_founded_rule_oracle [V L rules].
Implicit Arguments rule_inductive [V L].
Implicit Arguments build_proof_right_property [V L rules hypotheses].
Implicit Arguments restrict_hypothesis [V L rules hypotheses].
