(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: complete.v,v 1.41 2013/04/10 11:17:14 tews Exp $
 *)

(** ** Completeness and semantic cut admissibility, 4.13 - 4.15

 *)

Require Export prop_mod weakening cut_properties
               propositional_completeness sound step_semantics
               backward_substitution.

Section Completeness.

  Variable V : Type.
  Variable L : modal_operators.
  Variable T : functor.

  (** Need a decidable equality on propositional constants for 
      the proof construction.
   *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** ***  Towards rank-n completeness, Prop 4.13  *)
  (***************************************************************************)

  (***************************************************************************)
  (** ****  rank-n completeness for [top_mod] sequents
           This is the heart of the proof of 4.13 *)
  (***************************************************************************)

  Definition subst_osr_conclusions_with_ax(nonempty_v : V)
               (LS : lambda_structure L T)(rules : set (sequent_rule V L))
               (osr : one_step_rule_set rules)(tau : lambda_subst V L)
               (n : nat)(rank_tau : rank_subst (S n) tau)
                                                      : set (sequent V L) :=
    union (subst_Ax_n_set tau (2 + n))
          (subst_sequent_set tau
             (valid_subst_n_conclusions nonempty_v rules osr
                (n_step_subst_coval LS tau n rank_tau))).


  (** Part with cut *)
  Lemma top_mod_n_completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (n : nat)(s : sequent V L)(rank : rank_sequent (2 + n) s),
      non_trivial_functor T ->
      one_step_complete (enum_elem enum_V) LS rules osr ->
      (forall(s : sequent V L)(rank : rank_sequent (S n) s),
        step_semantics_valid (enum_elem enum_V) LS s n rank ->
          provable (GRC_n_set rules (S n)) (empty_sequent_set V L) s) ->
      top_modal_sequent s ->
      step_semantics_valid (enum_elem enum_V) LS s (S n) rank ->
        provable (GRC_n_set rules (2 + n)) (empty_sequent_set V L) s .
  Proof.
    intros enum_V LS rules osr n s rank H H0 H1 H2 H3.
    lapply (step_semantics_valid_nonempty _ _ _ _ _ H3).
      intros H4.
      decompose [ex and] (mod_arg_back_subst_sequent_prop 
                              v_eq s n (enum_infinite enum_V) H2 rank).
      rename x into simple_s, x0 into tau.
      apply (subst_coval_modal_step_semantics_valid_for_4_13 _ _ _ _ 
               H4 H6 _ _ _ H8 H5) in H3.
      apply plug_empty_hypothesis_proof with
            (provable_hyp := subst_osr_conclusions_with_ax (enum_elem enum_V)
                 LS rules osr tau n H8).
        clear - v_eq H1.
        intros subst_s H.
        destruct H.
          unfold subst_Ax_n_set in *.
          decompose [ex and] H; clear H.
          rename x into taut, x0 into delta.
          eapply provable_GRC_n_list_reorder.
            eexact H2.
          apply list_weakening_admissible_GRC_n.
                trivial.
              eapply rank_sequent_mono with (n1 := S n).
                omega.
              eapply rank_sequent_subst_Ax_set.
                eexact H8.
              trivial.
            trivial.
          eapply proof_mono_rules.
            apply rank_rules_subset_rank with (n1 := S n).
            omega.
          apply H1 with (rank := rank_sequent_subst_Ax_set _ _ _ H8 H0).
          eapply step_semantics_valid_taut; trivial.
          eexact H0.
        unfold subst_sequent_set in *.
        decompose [ex and] H; clear H.
        rename x into s.
        subst subst_s.
        unfold valid_subst_n_conclusions in *.
        decompose [ex and or dep_and] H2; clear H2.
        rename x into r_subst, a into H3.
        subst s.
        assert (H4 := H3).
        unfold simple_mod_weaken_rule in H4.
        decompose [ex and or dep_and] H4; clear H4.
        rename x into r, x0 into sigma, x1 into delta.
        eapply provable_with_rule.
          clear H1 H0.
          apply R_n_subset_GRC_n.
          apply stratified_one_step_rules
                with (1 := v_eq)(2 := enum_elem enum_V)(npos := lt_0_Sn _).
            trivial.
          exists r, (subst_compose tau sigma), (subst_sequent tau delta).
          repeat split; trivial.
              eapply rank_subst_subst_compose.
                  eexact H8.
                eexact H.
              omega.
            eapply rank_sequent_subst_add.
                  apply rank_simple_modal_sequent.
                  trivial.
                eexact H8.
              apply le_n_S.
              apply le_0_n.
            trivial.
          simpl.
          rewrite subst_sequent_compose.
          rewrite <- subst_sequent_append.
          apply list_reorder_subst_sequent.
          trivial.
        clear - H2 H9 H0 H1 H H6.
        apply every_nth_map.
        intros i i_less.
        eapply proof_mono_rules.
          apply rank_rules_subset_rank with (n1 := S n).
          omega.
        rewrite subst_sequent_compose.
        assert (rank_sequent (S n) (subst_sequent tau (subst_sequent sigma 
                                        (nth (assumptions r) i i_less)))).
          eapply rank_sequent_subst_add.
                eapply rank_sequent_subst_nth_assumptions; eauto.
              eexact H8.
            apply le_n_S.
            apply le_0_n.
          trivial.
        apply H1 with (rank := H4).
        clear H1.
        assert (i < length (assumptions r_subst)).
          rewrite H6.
          rewrite map_length.
          trivial.
        assert (propositional_sequent 
                    (subst_sequent sigma (nth (assumptions r) i i_less))).
          eapply rank_sequent_subst_nth_assumptions; eauto.
        eapply subst_coval_prop_step_semantics_valid with (prop_s := H5).
        specialize (H0 i H1).
        revert H0.
        generalize (simple_mod_weaken_rule_assumptions 
                                rules r_subst osr H3 i H1).
        revert H1.
        rewrite H6.
        intros H1.
        rewrite nth_map.
        erewrite nth_tcc_irr.
        intros p H0.
        eapply prop_seq_val_valid_tcc_irr.
        apply H0.
      subst s.
      unfold subst_osr_conclusions_with_ax in *.
      apply GRC_n_substitution_lemma; trivial.
        apply rank_sequent_set_valid_subst_n_conclusions.
      eapply proof_mono_rules.
        apply GC_n_subset_GRC_n.
      apply H0 in H3.
      trivial.
    destruct (nonempty_terminal_obj_sequence (S n) (empty_set V) H).
    exists x.
    trivial.
  Qed.

  (** Part without cut *)
  Lemma top_mod_n_cut_free_completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (n : nat)(s weak_cont : sequent V L)
          (rank_s : rank_sequent (2 + n) s),
      non_trivial_functor T ->
      one_step_cut_free_complete (enum_elem enum_V) LS rules osr ->
      (forall(s : sequent V L)(rank : rank_sequent (S n) s),
        step_semantics_valid (enum_elem enum_V) LS s n rank ->
          provable (GR_n_set rules (S n)) (empty_sequent_set V L) s) ->
      top_modal_sequent s ->
      step_semantics_valid (enum_elem enum_V) LS s (S n) rank_s ->
      rank_sequent (2 + n) weak_cont ->
        provable (GR_n_set rules (2 + n)) (empty_sequent_set V L) 
                 (s ++ weak_cont).
  Proof.
    intros enum_V LS rules osr n s weak_cont rank_s H H0 H1 H2 H3 H4.
    lapply (step_semantics_valid_nonempty _ _ _ _ _ H3).
      intros H5.
      decompose [ex and] (mod_arg_back_subst_sequent_prop 
                         v_eq s n (enum_infinite enum_V) H2 rank_s).
      rename x into simple_s, x0 into tau.
      apply (subst_coval_modal_step_semantics_valid_for_4_13 _ _ _ _ 
               H5 H7 _ _ _ H9 H6) in H3.
      apply (iff_right (simple_one_step_cut_free_complete v_eq _ _ _ _) H0)
         in H3.
      decompose [ex and or dep_and] H3; clear H3.
      rename x into r, x0 into rename, a into H11, a0 into H12.
      destruct H8 as [r_weakening].
      eapply provable_with_rule.
        clear H H0 H1 H2 H5 H10.
        apply R_n_subset_GR_n.
        apply stratified_one_step_rules 
              with (1 := v_eq)(2 := enum_elem enum_V)(npos := lt_0_Sn _).
          trivial.
        exists r, (subst_compose tau rename), 
               ((subst_sequent tau r_weakening) ++ weak_cont).
        repeat split; trivial.
            simpl.
            eapply rank_subst_subst_compose.
                eexact H9.
              apply rank_renaming.
              trivial.
            rewrite <- plus_n_O.
            trivial.
          apply rank_sequent_append.
            eapply rank_sequent_subst_add.
                  eapply rank_sequent_append_right.
                  eapply rank_sequent_list_reorder.
                    eexact H3.
                  apply rank_simple_modal_sequent.
                  trivial.
                eexact H9.
              apply le_n_S.
              apply le_0_n.
            omega.
          trivial.
        simpl.
        rewrite app_assoc.
        apply list_reorder_append_right.
        rewrite subst_sequent_compose.
        subst s.
        rewrite <- subst_sequent_append.
        apply list_reorder_subst_sequent.
        apply list_reorder_symm.
        trivial.
      clear - H1 H10.
      intros i i_less.
      eapply proof_mono_rules.
        apply rank_rules_subset_rank with (n1 := S n).
        omega.
      assert (rank_sequent (S n) 
                (nth (map (subst_sequent (subst_compose tau rename))
                          (assumptions r))
                     i i_less)).
        apply rank_subst_assumptions.
          apply osr.
          trivial.
        eapply rank_subst_subst_compose.
            eexact H9.
          apply rank_renaming.
          trivial.
        rewrite <- plus_n_O.
        trivial.
      apply H1 with (rank := H).
      clear H1.
      revert H.
      rewrite nth_map.
      rewrite subst_sequent_compose.
      intros H.
      eapply subst_coval_prop_step_semantics_valid with (rank_sigma := H9).
      apply H10.
    destruct (nonempty_terminal_obj_sequence (S n) (empty_set V) H).
    exists x.
    trivial.
  Qed.


  (***************************************************************************)
  (** **** Build proof up to non-decomposable formulas 

         The following instantiates the generic [build_proof] to do a
         propositional proof search. This is used to decompose the
         sequent into sequents of non-decomposable formulas. (The
         statement about the inversion lemma in the paper is
         nonsense.) These non-decomposable formulas appear as
         hypothesis in the proof search, such that [build_proof] is
         guaranteed to succeed.
   *)
  (***************************************************************************)

  (** Hypothesis for the proof search. *)
  Definition simple_4_13_sequent : set (sequent V L) :=
    intersection (prop_or_mod_sequent V L) 
                 (set_inverse simple_tautology).

  Definition mod_prop_hyp_oracle : 
                       hypotheses_oracle_type V L simple_4_13_sequent :=
    fun(s : sequent V L) => 
      match find_trivial v_eq s s 0 as ft
        return find_trivial v_eq s s 0 = ft -> option (simple_4_13_sequent s)
      with
        | None => fun(ftn : find_trivial v_eq s s 0 = None) =>
          match dprop_or_mod_sequent s as res 
            return dprop_or_mod_sequent s = res
                         -> option (simple_4_13_sequent s)
          with
            | true => fun(p : dprop_or_mod_sequent s = true) => 
              Some (conj (iff_left (prop_or_mod_sequent_prop _) p)
                         (find_trivial_none v_eq _ ftn))
            | false => fun _ => None
          end (eq_refl (dprop_or_mod_sequent s))
        | Some _ => fun _ => None
      end (eq_refl (find_trivial v_eq s s 0)).

  Definition prop_Gn_oracle(n : nat) : 
                              rule_oracle_type V L (G_n_set V L n) :=
    fun(s : sequent V L) => 
    match prop_G_oracle v_eq s with
      | None => None
      | Some (dep_conj r (conj in_rules concl_prop)) =>
        let comp_res := Compare_dec.leb (minimal_rule_rank r) n in
        match comp_res as cr 
          return comp_res = cr 
                    -> rule_oracle_result V L (G_n_set V L n) s
        with
          | true => fun(H : comp_res = true) =>
            Some(dep_conj _ _ r 
                   (conj 
                      (rank_rules_minimal_rule_rank (G_set V L) r n 
                          in_rules (leb_complete _ _ H))
                      concl_prop))
          | false => fun _ => None
        end (eq_refl comp_res)
    end.

  Lemma well_founded_Gn_oracle : forall(n : nat),
    well_founded_rule_oracle (prop_Gn_oracle n) sequent_measure.
  Proof.
    unfold well_founded_rule_oracle, prop_Gn_oracle in *.
    intros n s.
    destruct (prop_G_oracle v_eq s) eqn:?.
      destruct d.
      destruct a0.
      generalize (eq_refl (Compare_dec.leb (minimal_rule_rank a) n)).
      destruct (Compare_dec.leb (minimal_rule_rank a) n) in |- * at 2 3.
        intros e0.
        assert (H := well_founded_G_oracle V L v_eq s).
        rewrite Heqr in H.
        trivial.
      trivial.
    trivial.
  Qed.

  (** [build_proof] instantiation with a type that permits failure. *)
  Definition prop_mod_build_proof_maybe(n : nat)(s : sequent V L) : 
              (proof (G_n_set V L n) simple_4_13_sequent s) + (sequent V L) :=
    build_proof (S (sequent_measure s)) 
      mod_prop_hyp_oracle (prop_Gn_oracle n) s.

  (** Proof that the proof search will always succeed. *)
  Lemma prop_mod_build_proof_left : forall(n : nat)(s : sequent V L), 
    rank_sequent n s ->
      is_inl (prop_mod_build_proof_maybe n s).
  Proof.
    intros n s H.
    destruct (prop_mod_build_proof_maybe n s) eqn:?.
      simpl.
      trivial.
    exfalso.
    unfold prop_mod_build_proof_maybe in *.
    assert (H0 := Heqs0).
    apply build_proof_right_property with (P := rank_sequent n) 
                                     (measure := sequent_measure) in Heqs0.
            clear H.
            apply build_proof_right_result 
                        with (measure := sequent_measure) in H0. 
                clear s.
                destruct H0.
                unfold prop_Gn_oracle in *.
                destruct (prop_G_oracle v_eq s0) eqn:?.
                  clear Heqr H0.
                  destruct d.
                  destruct a0.
                  subst s0.
                  revert H.
                  generalize 
                     (eq_refl (Compare_dec.leb (minimal_rule_rank a) n)).
                  destruct (Compare_dec.leb (minimal_rule_rank a) n) 
                           in |-* at 2 3.
                    discriminate.
                  intros e0 H.
                  clear H.
                  apply leb_complete_conv in e0.
                  assert (rule_has_rank n a).
                    unfold rule_has_rank in *.
                    split.
                      apply const_rank_G_set.
                        trivial.
                      trivial.
                    trivial.
                  apply rank_rules_ge_minimal in H.
                  apply le_not_lt in H.
                  contradiction.
                clear H.
                assert (H1 := Heqr).
                apply non_decomposable_is_prop_mod in Heqr.
                apply prop_or_mod_sequent_prop in Heqr.
                apply prop_G_oracle_None in H1.
                destruct H1.
                clear H1.
                revert H0.
                unfold mod_prop_hyp_oracle in *.
                generalize (eq_refl (find_trivial v_eq s0 s0 0)).
                pattern (find_trivial v_eq s0 s0 0) at 2 3.
                rewrite H.
                clear H.
                intros H.
                generalize (eq_refl (dprop_or_mod_sequent s0)).
                pattern (dprop_or_mod_sequent s0) at 2 3.
                rewrite Heqr.
                discriminate.
              apply well_founded_Gn_oracle.
            apply lt_n_Sn.
          apply well_founded_Gn_oracle.
        apply lt_n_Sn.
      apply rank_G_n_inductive.
    trivial.
  Qed.


  (** Valid subset of the hypotheses *)
  Definition valid_simple_4_13_sequent(nonempty_v : V)
                   (LS : lambda_structure L T)(n : nat) : set (sequent V L) :=
    intersection simple_4_13_sequent 
                 (step_semantics_valid_at_rank nonempty_v LS n).

  Lemma valid_simple_4_13_sequent_list_reorder :
    forall(nonempty_v : V)(LS : lambda_structure L T)(n : nat)
          (s1 s2 : sequent V L),
      list_reorder s1 s2 ->
      valid_simple_4_13_sequent nonempty_v LS n s1 ->
        valid_simple_4_13_sequent nonempty_v LS n s2.
  Proof.
    intros nonempty_v LS n s1 s2 H H0.
    unfold valid_simple_4_13_sequent, simple_4_13_sequent in *.
    repeat split.
        eapply prop_or_mod_sequent_list_reorder.
          eexact H.
        apply H0.
      intros H1.
      eapply simple_tautology_reorder in H1.
        apply H0 in H1.
        contradiction.
      apply list_reorder_symm.
      trivial.
    eapply step_semantics_valid_at_rank_list_reorder.      
        trivial.
      eexact H.
    apply H0.
  Qed.


  (** Restrict the proof building in two ways: First, guarantee
      success by type, second restrict hypotheses to valid sequents.
   *)
  Definition prop_mod_build_proof(classic : classical_logic)(nonempty_v : V)
             (LS : lambda_structure L T)
             (n : nat)(s : sequent V L)(rank : rank_sequent (S n) s)
             (valid : step_semantics_valid nonempty_v LS s n rank)
                    : proof (G_n_set V L (S n)) 
                            (valid_simple_4_13_sequent nonempty_v LS n) s :=
    match prop_mod_build_proof_maybe (S n) s 
      as pmaybe 
      return is_inl pmaybe -> 
                 proof (G_n_set V L (S n)) 
                       (valid_simple_4_13_sequent nonempty_v LS n) s
    with
      | inl p => fun _ => 
        restrict_hypothesis (step_semantics_valid_at_rank nonempty_v LS n)
          (step_semantics_valid_G_rule_inductive nonempty_v LS n classic)
          s 
          (dep_conj (rank_sequent (S n) s) 
                    (step_semantics_valid nonempty_v LS s n)
             rank valid)
          p
      | inr _ => fun(H : False) => False_rect _ H
    end (prop_mod_build_proof_left (S n) s rank).


  (***************************************************************************)
  (** **** Split propositional part and retain a valid modal sequent *)
  (***************************************************************************)

  Lemma split_prop_mod :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (prop_s mod_s : sequent V L)
          (n : nat)(rank : rank_sequent (2 + n) (mod_s ++ prop_s)),
      classical_logic ->
      non_trivial_functor T ->
      prop_sequent prop_s ->
      top_modal_sequent mod_s ->
      ~ (simple_tautology prop_s) ->
      step_semantics_valid nonempty_v LS (mod_s ++ prop_s) (S n) rank ->
        step_semantics_valid nonempty_v LS mod_s (S n)
          (rank_sequent_append_left _ _ _ rank).
  Proof.
    intros nonempty_v LS prop_s mod_s n rank classic H H0 H1 H2 H3.
    assert (H4 := prop_sequent_is_propositional _ H0).
    unfold step_semantics_valid in *.
    apply top_modal_is_prop_modal_sequent in H1.
    eapply step_mod_sequent_semantics 
              with (rank := rank_sequent_append_left _ _ _ rank) in H1.
    destruct H1 as [mod_sem_s].
    lapply (step_prop_sequent_semantics nonempty_v LS prop_s _
                (rank_sequent_append_right _ _ _ rank));
           trivial.
    intros H5.
    destruct H5 as [prop_sem_s].
    eapply set_equal_is_full.
      apply set_equal_symm.
      eexact H1.
    apply (build_counter_model_correct v_eq _ H4 (all_true_model V)) 
            in H2; trivial.
    eapply inv_proj_full 
              with (pi_2 := @terminal_obj_sequence_pi_2 V T (S n))
              (b := (build_counter_model V L v_eq prop_s (all_true_model V))).
          clear. 
          destruct n.
            apply feq_reflexive.
          apply feq_reflexive.
        clear. 
        apply feq_reflexive.
      eapply set_equal_is_full.
        eapply set_equal_union.
          eexact H1.
        eexact H5.
      eapply set_equal_is_full.
        apply set_equal_symm.
        apply union_double_neg_intersection.
        trivial.
      intros x.
      apply step_semantics_sequent_append with (rank := rank).
      trivial.
    clear - H H2 H5.
    assert (H6 := nonempty_terminal_obj_sequence (S n) 
                 (build_counter_model V L v_eq prop_s (all_true_model V)) H).
    destruct H6 as [x H6].
    specialize (H5 x).
    unfold inv_img in *.
    rewrite H6 in H5.
    rewrite <- H5.
    unfold step_semantics_sequent in *.
    rewrite one_step_semantics_propositional 
        with (prop_f := propositional_or_formula nonempty_v _ H4).
    rewrite H6.
    rewrite prop_model_sequent_interpretation in H2.
    apply H2.
  Qed.


  (***************************************************************************)
  (** **** Rank-n completeness, Prop 4.13 *)
  (***************************************************************************)

  (** 4.13, part with cut *)
  Lemma n_completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (n : nat)(s : sequent V L)(rank : rank_sequent (S n) s),
      classical_logic ->
      non_trivial_functor T ->
      one_step_complete (enum_elem enum_V) LS rules osr ->
      step_semantics_valid (enum_elem enum_V) LS s n rank ->
        provable (GRC_n_set rules (S n)) (empty_sequent_set V L) s.
  Proof.
    induction n.
      intros s rank classic H H0 H1.
      eapply proof_mono_rules.
        apply prop_G_subset_GRC.
        trivial.
      eapply propositional_complete_G with (1 := v_eq)(prop_s := rank); 
             trivial.
      eapply one_step_semantics_valid_propositional.
        eexact H.
      eexact H1.
    intros s rank classic H H0 H1.
    apply plug_empty_hypothesis_proof with
            (provable_hyp := valid_simple_4_13_sequent 
                                               (enum_elem enum_V) LS (S n)).
      clear - classic IHn H H0.
      intros s H1.
      lapply (prop_or_mod_partition s).
        intros H2.
        decompose [ex and] H2; clear H2.
        rename x into prop_s, x0 into mod_s.
        eapply provable_GRC_n_list_reorder.
          apply list_reorder_symm.
          eexact H6.
        apply valid_simple_4_13_sequent_list_reorder with (1 := H6) in H1;
              trivial.
        destruct H1.
        destruct H2 as [rank H2].
        destruct H1.
        apply split_prop_mod in H2; trivial.
          eapply list_weakening_admissible_GRC_n; eauto.
	      eapply rank_sequent_append_left.
	      eexact rank.
	    eapply rank_sequent_append_right. 	 
	    eexact rank.
          eapply top_mod_n_completeness; eauto.
        intros H7.
        eapply simple_tautology_append_left in H7.
        apply H5 in H7.
        trivial.
      apply H1.
    clear IHn.
    eapply proof_mono_rules.
      apply G_n_subset_GRC_n.
    exists (prop_mod_build_proof classic 
                 (enum_elem enum_V) LS (S n) s rank H1).
    trivial.
  Qed.

  (** 4.13, cut-free part *)
  Lemma n_cut_free_completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (n : nat)(s : sequent V L)(rank : rank_sequent (S n) s),
      classical_logic ->
      non_trivial_functor T ->
      one_step_cut_free_complete (enum_elem enum_V) LS rules osr ->
      step_semantics_valid (enum_elem enum_V) LS s n rank ->
        provable (GR_n_set rules (S n)) (empty_sequent_set V L) s.
  Proof.
    induction n.
      intros s rank classic H H0 H1.
      eapply proof_mono_rules.
        apply prop_G_subset_GR.
        trivial.
      eapply propositional_complete_G with (1 := v_eq)(prop_s := rank); 
             trivial.
      eapply one_step_semantics_valid_propositional.
        eexact H.
      eexact H1.
    intros s rank classic H H0 H1.
    apply plug_empty_hypothesis_proof 
        with (provable_hyp := valid_simple_4_13_sequent 
                                           (enum_elem enum_V) LS (S n)).
      clear - v_eq classic IHn H H0.
      intros s H1.
      lapply (prop_or_mod_partition s).
        intros H2.
        decompose [ex and] H2; clear H2.
        rename x into prop_s, x0 into mod_s.
        eapply provable_GR_n_list_reorder.
          apply list_reorder_symm.
          eexact H6.
        apply valid_simple_4_13_sequent_list_reorder with (1 := H6) in H1;
              trivial.
        destruct H1.
        destruct H2 as [rank H2].
        destruct H1.
        apply split_prop_mod in H2; trivial.
          eapply top_mod_n_cut_free_completeness; eauto.
          apply rank_sequent_mono with (n1 := 1).
            omega.
          apply rank_prop_sequent.
          trivial.
        intros H7.
        eapply simple_tautology_append_left in H7.
        apply H5 in H7.
        trivial.
      apply H1.
    clear IHn.
    eapply proof_mono_rules.
      apply G_n_subset_GR_n.
    exists (prop_mod_build_proof classic 
                 (enum_elem enum_V) LS (S n) s rank H1).
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Completeness, Corollary 4.14 *)
  (***************************************************************************)

  (** 4.14, part with cut *)
  Lemma completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (s : sequent V L),
      classical_logic ->
      non_trivial_functor T ->
      one_step_complete (enum_elem enum_V) LS rules osr ->
      valid_all_models (enum_elem enum_V) LS s ->
        provable (GRC_set rules) (empty_sequent_set V L) s.
  Proof.
    intros enum_V LS rules osr s classic H H0 H1.
    assert (exists(n : nat), rank_sequent (S n) s).
      clear. 
      destruct s.
        exists 0.
        apply rank_sequent_empty.
      exists (pred (minimal_sequent_rank (l :: s))).
      apply rank_sequent_succ_minimal_nonempty_sequent_rank.
      discriminate.
    destruct H2 as [n].
    eapply proof_mono_rules.
      apply subset_rank_rules with (n := S n).
    apply (n_completeness enum_V LS _ osr _ _ H2); trivial.
    apply step_semantics_validity.
      trivial.
    trivial.
  Qed.

  (** 4.14, cut-free part *)
  Lemma cut_free_completeness : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)
          (s : sequent V L),
      classical_logic ->
      non_trivial_functor T ->
      one_step_cut_free_complete (enum_elem enum_V) LS rules osr ->
      valid_all_models (enum_elem enum_V) LS s ->
        provable (GR_set rules) (empty_sequent_set V L) s.
  Proof.
    intros enum_V LS rules osr s classic H H0 H1.
    assert (exists(n : nat), rank_sequent (S n) s).
      clear. 
      destruct s.
        exists 0.
        apply rank_sequent_empty.
      exists (pred (minimal_sequent_rank (l :: s))).
      apply rank_sequent_succ_minimal_nonempty_sequent_rank.
      discriminate.
    destruct H2 as [n].
    eapply proof_mono_rules.
      apply subset_rank_rules with (n := S n).
    apply (n_cut_free_completeness enum_V LS _ osr _ _ H2); trivial.
    apply step_semantics_validity.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Towards admissibility of cut and contraction, 4.15
         This is the semantic cut-elimination theorem
   *)
  (***************************************************************************)

  Lemma semantic_admissible_rules : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (osr rules : set (sequent_rule V L))
          (osr_prop : one_step_rule_set osr),
      classical_logic ->
      non_trivial_functor T ->
      one_step_sound (enum_elem enum_V) LS osr osr_prop ->
      one_step_cut_free_complete (enum_elem enum_V) LS osr osr_prop ->
      downward_correct_rule_set (enum_elem enum_V) LS rules ->
        admissible_rule_set (GR_set osr) (empty_sequent_set V L) rules.
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros enum_V LS osr rules osr_prop classic H H0 H1 H2 r H3 H4.
    apply (cut_free_completeness enum_V LS _ osr_prop); trivial.
    intros m.
    apply (H2 _ H3).
    intros i i_less.
    apply (sound_GR _ _ _ osr_prop); trivial.
  Qed.


  (** **** 4.15, cut part *)
  Theorem semantic_admissible_cut : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))
          (osr_prop : one_step_rule_set rules),
      classical_logic ->
      non_trivial_functor T ->
      one_step_sound (enum_elem enum_V) LS rules osr_prop ->
      one_step_cut_free_complete (enum_elem enum_V) LS rules osr_prop ->
        admissible_rule_set (GR_set rules) (empty_sequent_set V L)
          is_cut_rule.
  Proof.
    intros enum_V LS osr osr_prop classic H H0 H1.
    apply (semantic_admissible_rules enum_V LS _ _ osr_prop); trivial.
    intros r H2.
    apply downward_correct_rule_strengthen.
    apply strong_downward_correct_cut; trivial.
  Qed.

  (** **** 4.15, contraction part *)
  Theorem semantic_admissible_contraction : 
    forall(enum_V : enumerator V)(LS : lambda_structure L T)
          (osr : set (sequent_rule V L))
          (osr_prop : one_step_rule_set osr),
      classical_logic ->
      non_trivial_functor T ->
      one_step_sound (enum_elem enum_V) LS osr osr_prop ->
      one_step_cut_free_complete (enum_elem enum_V) LS osr osr_prop ->
        admissible_rule_set (GR_set osr) (empty_sequent_set V L)
          is_contraction_rule.
  Proof.
    intros enum_V LS osr osr_prop classic H H0 H1.
    apply (semantic_admissible_rules enum_V LS _ _ osr_prop); trivial.
    intros r H2.
    apply downward_correct_contraction.
      trivial.
    trivial.
  Qed.

End Completeness.
