(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: formulas.v,v 1.84 2013/04/10 11:17:14 tews Exp $
 *)


(** * Logic, basic definitions *)

(** ** Formulas, Sequents, Proofs

      This is the key module of the formalization. It defines the
      types for formulas, sequents and proofs. It also defines various
      subsets of sequents, a measure function on formulas, the modal
      rank and the functions for extracting the propositional
      variables. 
*)


Require Export dep_lists reorder.

Record modal_operators : Type := {
  operator : Type;
  arity : operator -> nat
}.

Section Formulas.
  Variable V : Type.
  Variable L : modal_operators.

  Unset Elimination Schemes.

  (***************************************************************************)
  (** ***  Lambda Formulas, page 5  *)
  (***************************************************************************)

  Inductive lambda_formula : Type :=
    | lf_prop : V -> lambda_formula
    | lf_neg : lambda_formula -> lambda_formula
    | lf_and : lambda_formula -> lambda_formula -> lambda_formula
    | lf_modal : forall(op : operator L),
        counted_list lambda_formula (arity L op) -> lambda_formula.


  Set Elimination Schemes.

  (** The recursion and induction schemes from Coq are not useful,
      because of the use of [counted_list]. I don't like the usual
      workaround, namely unrolling [counted_list] and [lambda_formula]
      into two mutually recursive types, because it is less modular
      and, generally, makes a mess. Therefore I have to define my own
      recursion and induction schemes.
   *)
  Fixpoint lambda_formula_rect(T : lambda_formula -> Type)
             (prop_fun : forall(v : V), T (lf_prop v))
             (neg_fun : forall(f : lambda_formula), T f -> T (lf_neg f))
             (and_fun : forall(f1 f2 : lambda_formula), 
                          T f1 -> T f2 -> T (lf_and f1 f2))
             (modal_fun : forall(op : operator L)
                            (args : counted_list lambda_formula (arity L op)),
                  (dep_list lambda_formula T (list_of_counted_list args))
                     -> T (lf_modal op args))
             (f : lambda_formula) : T f :=
    let rec_fn : forall(f : lambda_formula), T f :=
          lambda_formula_rect T prop_fun neg_fun and_fun modal_fun
    in
      match f with
        | lf_prop v => prop_fun v
        | lf_neg f => neg_fun f (rec_fn f)
        | lf_and f1 f2 => 
            and_fun f1 f2 (rec_fn f1) (rec_fn f2)
        | lf_modal op args =>
            modal_fun op args
              ((fix map_args(len : nat)(args : counted_list lambda_formula len)
                     : dep_list lambda_formula T (list_of_counted_list args) :=
                  match args with
                    | counted_nil => dep_nil
                    | counted_cons n f rargs => 
                        dep_cons f (list_of_counted_list rargs)
                          (rec_fn f) (map_args n rargs)
                  end
               ) (arity L op) args)
      end.


  (** Induction over lambda_formula's *)
  Lemma lambda_formula_ind : 
    forall(P : lambda_formula -> Prop),
      (forall(v : V), P (lf_prop v)) ->
      (forall(f : lambda_formula), P f -> P (lf_neg f)) ->
      (forall(f1 f2 : lambda_formula), P f1 -> P f2 -> P (lf_and f1 f2)) ->
      (forall(op : operator L)
             (args : counted_list lambda_formula (arity L op)),
         every_nth P (list_of_counted_list args) -> P (lf_modal op args)) 
      ->
        forall(f : lambda_formula), P f.
  Proof.
    intros P H H0 H1 H2 f.
    apply lambda_formula_rect.
          trivial.
        trivial.
      trivial.
    intros op args X.
    apply H2.
    apply every_nth_of_dep_list.
    exists X.
    trivial.
  Qed.


  (** Define non-dependent recursion over lambda_formula's. 
      As [lambda_formula_rect], this contains a nested, inlined,
      mutually recursive map over [counted_list], because, otherwise,
      Coq won't accept the definition. This is fixed with the next
      lemma, which references [counted_map].

      Instead of a separate definition, one could probably derive this
      from [lambda_formula_rect], but for this one needs to transform
      a [dep_list] into a [counted_list].
 *)
  Fixpoint lambda_formula_rec(A : Type)
             (prop_fun : V -> A)
             (neg_fun : A -> A)
             (and_fun : A -> A -> A)
             (modal_fun : forall(op : operator L), 
                             counted_list A (arity L op) -> A)
           (f : lambda_formula) : A :=
    let rec_fn : forall(f : lambda_formula), A :=
          lambda_formula_rec A prop_fun neg_fun and_fun modal_fun
    in
      match f with
        | lf_prop v => prop_fun v
        | lf_neg f => neg_fun (rec_fn f)
        | lf_and f1 f2 => 
            and_fun (rec_fn f1) (rec_fn f2)
        | lf_modal op args =>
            modal_fun op 
              ((fix map_args(len : nat)(args : counted_list lambda_formula len)
                          : counted_list A len :=
                  match args with
                    | counted_nil => counted_nil
                    | counted_cons n f rargs => 
                        counted_cons (rec_fn f) (map_args n rargs)
                  end
               ) (arity L op) args)
      end.

  Lemma lambda_formula_rec_char :
    forall(A : Type)(pf : V -> A)(nf : A -> A)(af : A -> A -> A)
          (mf : forall(op : operator L), counted_list A (arity L op) -> A)
          (f : lambda_formula),
      lambda_formula_rec A pf nf af mf f =
        match f with
          | lf_prop v => pf v
          | lf_neg f => nf (lambda_formula_rec A pf nf af mf f)
          | lf_and f1 f2 => 
            af (lambda_formula_rec A pf nf af mf f1) 
               (lambda_formula_rec A pf nf af mf f2)
          | lf_modal op args =>
            mf op (counted_map (lambda_formula_rec A pf nf af mf) args)
        end.
  Proof.
    induction f.
          trivial.
        trivial.
      trivial.
    simpl.
    f_equal.
    induction args.
      simpl.
      trivial.
    simpl.
    f_equal.
    apply IHargs.
    clear IHargs.
    eapply every_nth_tail.
    eexact H.
  Qed.


  (***************************************************************************)
  (** ***  Utility functions for formulas  *)
  (***************************************************************************)

  (** Recognizer predicates: these are boolean decidable recognizers. 
      They can be used in functions such as list_search. 
      For specifications there are some Prop valued recognizers below.
   *)

  Definition is_prop (l : lambda_formula) : bool :=
    match l with 
      | lf_prop _ => true
      | _ => false
    end.

  Definition is_neg (l : lambda_formula) : bool :=
    match l with
      | lf_neg _ => true
      | _ => false
    end.

  Definition is_and (l : lambda_formula) : bool :=
    match l with
      | lf_and _ _ => true
      | _ => false
    end.

  Definition is_modal (l : lambda_formula) : bool :=
    match l with
      | lf_modal _ _ => true
      | _ => false
    end.


  (** Accessor functions. These accessor functions work with the decidable
      recognizers from above.
   *)
  Definition get_prop_var(f : lambda_formula)(is_prop_f : is_prop f = true) : V.
  Proof.
    refine (
      match f as f0 return is_prop f0 = true -> V with
        | lf_prop v => fun _ => v
        | _ => fun (not_prop : _ ) => _
      end is_prop_f
    );
      simpl in *; discriminate.
  Defined.

  Definition get_neg_form(f : lambda_formula)(is_neg_f : is_neg f = true) 
                                                             : lambda_formula.
  Proof.
    refine (
      match f as f0 return is_neg f0 = true -> lambda_formula with
        | lf_neg f => fun _ => f
        | _ => fun (not_neg : _ ) => _
      end is_neg_f
    );
      simpl in *; discriminate.
  Defined.

  Definition get_and_forms(f : lambda_formula)(is_and_f : is_and f = true) 
                                             : lambda_formula * lambda_formula.
  Proof.
    refine (
      match f as f0 
              return is_and f0 = true -> lambda_formula * lambda_formula 
      with
        | lf_and fl fr => fun _ => (fl, fr)
        | _ => fun (not_and : _ ) => _
      end is_and_f
    );
      simpl in *; discriminate.
  Defined.

  Definition get_modal_args(f : lambda_formula)(is_modal_f : is_modal f = true) 
                                         : (operator L) * (list lambda_formula).
  Proof.
    refine (
      match f as f0 
              return is_modal f0 = true -> (operator L) * (list lambda_formula) 
      with
        | lf_modal op args => fun _ => (op, list_of_counted_list args)
        | _ => fun (not_neg : _ ) => _
      end is_modal_f
    );
      simpl in *; discriminate.
  Defined.


  (** standard propositional connectives, page 5 *)
  Definition lambda_or(A B : lambda_formula) : lambda_formula :=
    lf_neg(lf_and (lf_neg A) (lf_neg B)).

  Definition lambda_false(nonempty_v : V) : lambda_formula :=
    lf_and (lf_prop nonempty_v) (lf_neg (lf_prop nonempty_v)).


  (***************************************************************************)
  (** ***  Inversion lemmas for lf_modal  *)
  (***************************************************************************)

  Lemma lf_modal_inversion_op :
    forall(op1 op2 : operator L)
          (args1 : counted_list lambda_formula (arity L op1))
          (args2 : counted_list lambda_formula (arity L op2)),
      lf_modal op1 args1 = lf_modal op2 args2 ->
        op1 = op2.
  Proof.
    intros op1 op2 args1 args2 H.
    inversion H.
    trivial.
  Qed.

  Lemma lf_modal_inversion_args :
    forall(op : operator L)
          (args1 args2 : counted_list lambda_formula (arity L op)),
      lf_modal op args1 = lf_modal op args2 ->
        args1 = args2.
  Proof.
    intros op args1 args2 H.
    assert (is_modal (lf_modal op args1) = true).
      trivial.
    assert (is_modal (lf_modal op args2) = true).
      trivial.
    assert (get_modal_args (lf_modal op args1) H0 =
            get_modal_args (lf_modal op args2) H1).
      generalize H0.
      rewrite H.
      intros H2.
      simpl.
      trivial.
    simpl in H2.
    inversion H2; clear H2.
    apply counted_list_equal.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Sequents, Rules  *)
  (***************************************************************************)

  Definition sequent : Type := list lambda_formula.

  Definition empty_sequent_set : set sequent := empty_set (sequent).


  Record sequent_rule : Type := {
    assumptions: list sequent;
    conclusion: sequent
  }.


  (***************************************************************************)
  (** ***  simple propositional sequents  *)
  (***************************************************************************)

  (* see also is_prop for a decidable recognizer above *)
  Definition prop_form(f : lambda_formula) : Prop :=
    match f with
      | lf_prop _ => True
      | _ => False
    end.

  Definition prop_form_acc(f : lambda_formula)(prop_f : prop_form f) : V.
  Proof.
    refine (
      match f as f0 return prop_form f0 -> V with
        | lf_prop v => fun _ => v
        | _ => fun (not_prop : _ ) => _
      end prop_f
    );
      simpl in *; contradiction.
  Defined.

  Lemma prop_form_acc_tcc_irr :
    forall(f : lambda_formula)(prop_f_1 prop_f_2 : prop_form f),
      prop_form_acc f prop_f_1 = prop_form_acc f prop_f_2.
  Proof.
    intros f prop_f_1 prop_f_2.
    destruct f.
          trivial.
        contradiction.
      contradiction.
    contradiction.
  Qed.

  Definition neg_form(f : lambda_formula) : Prop :=
    match f with
      | lf_neg _ => True
      | _ => False
    end.

  Lemma neg_form_is_neg : 
    forall(f : lambda_formula), is_neg f = true <-> neg_form f. 
  Proof.
    intros f.
    split.
      intros H.
      destruct f; try discriminate.
      simpl.
      trivial.
    intros H.
    destruct f; try contradiction.
    trivial.
  Qed.

  (* see also dneg_form for the decidable version *)
  Definition neg_form_maybe(F : set lambda_formula)(f : lambda_formula) 
                                                                     : Prop :=
    match f with
      | lf_neg f => F f
      | _ => F f
    end.

  Definition prop_sequent(s : sequent) : Prop :=
    every_nth (neg_form_maybe prop_form) s.

  Lemma prop_sequent_cons : 
    forall(s : sequent)(f : lambda_formula),
      neg_form_maybe prop_form f ->
      prop_sequent s ->
        prop_sequent (f :: s).
  Proof.
    intros s f H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma prop_sequent_tail : 
    forall(s : sequent)(f : lambda_formula),
      prop_sequent (f :: s) ->
        prop_sequent s.
  Proof.
    intros s f H.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma prop_sequent_head : 
    forall(s : sequent)(f : lambda_formula),
      prop_sequent (f :: s) ->
        neg_form_maybe prop_form f.
  Proof.
    intros s f H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma prop_sequent_list_reorder :
    forall(s1 s2 : sequent),
      list_reorder s1 s2 ->
      prop_sequent s1 ->
        prop_sequent s2.
  Proof.
    apply every_nth_list_reorder.
  Qed.


  (* Prop( S ), page 5 *)
  Fixpoint neg_and_over(fs : set lambda_formula)(f : lambda_formula) : Prop :=
    match f with
      | lf_neg f => neg_and_over fs f
      | lf_and f1 f2 => neg_and_over fs f1 /\ neg_and_over fs f2
      | _ => fs f
    end.


  (**************************************************************************)
  (** ***  Formula measure for different purposes  *)
  (**************************************************************************)

  Definition formula_measure(f : lambda_formula) : nat :=
    lambda_formula_rec nat
      (fun _ => 0)
      (fun m => 1 + m)
      (fun m1 m2 => 2 + m1 + m2)
      (fun _ cl => 1 + nat_list_sum (list_of_counted_list cl))
    f.

  Lemma formula_measure_char :
    forall(f : lambda_formula),
      formula_measure f =
        match f with
          | lf_prop _ => 0
          | lf_neg f => 1 + (formula_measure f)
          | lf_and f1 f2 => 2 + (formula_measure f1) + (formula_measure f2)
          | lf_modal _ args => 
              1 + nat_list_sum (map formula_measure 
                                    (list_of_counted_list args))
        end.
  Proof.
    intros f.
    unfold formula_measure in *.
    rewrite lambda_formula_rec_char.
    destruct f; trivial.
    f_equal.
    rewrite list_of_counted_list_map.
    trivial.
  Qed.

  Definition sequent_measure(s : sequent) : nat :=
    nat_list_sum (map formula_measure s).

  Lemma sequent_measure_cons :
    forall(s : sequent)(f : lambda_formula),
      sequent_measure (f :: s) = 
        (formula_measure f) + (sequent_measure s).
  Proof.
    intros s f.
    unfold sequent_measure in *.
    simpl.
    trivial.
  Qed.
  
  Lemma sequent_measure_append :
    forall(sl sr : sequent),
      sequent_measure (sl ++ sr) =
        sequent_measure sl + sequent_measure sr.
  Proof.
    intros sl sr.
    unfold sequent_measure in *.
    rewrite map_app.
    apply nat_list_sum_append.
  Qed.


  Lemma sequent_measure_context_lt :
   forall(os ns: sequent)(n : nat)(n_less : n < length os),
     sequent_measure ns
           < formula_measure (nth os n n_less) 
     ->
       sequent_measure (firstn n os ++ ns ++ skipn (1 + n) os)
         < sequent_measure os.
  Proof.
    intros os ns n n_less H.
    rewrite (list_split_at_n _ _ n_less) at 3.
    rewrite sequent_measure_append.
    rewrite sequent_measure_append.
    rewrite sequent_measure_append.
    rewrite sequent_measure_cons.
    omega.
  Qed.

  Lemma sequent_measure_simple_context_lt :
   forall(s : sequent)(n : nat)(n_less : n < length s)(f : lambda_formula),
     formula_measure f 
           < formula_measure (nth s n n_less) 
     ->
       sequent_measure (firstn n s ++ f :: skipn (1 + n) s)
         < sequent_measure s.
  Proof.
    intros s n n_less f H.
    assert (f :: skipn (1 + n) s = [f] ++ skipn (1 + n) s).
      trivial.
    rewrite H0.
    eapply sequent_measure_context_lt.
    unfold sequent_measure in *.
    simpl.
    rewrite plus_0_r.
    eexact H.
  Qed.


  (***************************************************************************)
  (** ***  Proofs  *)
  (***************************************************************************)

  (** Proofs, page 9
      
      In the paper they use multi-sets for sequents with transparent 
      reorderings in the written representation of a multiset. For the
      formalization of proofs I have to ensure that sequents of 
      rules are identical, where proof rules are plugged together. 
      - The multiset coq library uses functions A -> nat, therefore one
        cannot use intentional equality on those multisets
      - In a simple list representation I would have to identify lists up to 
        reorderings.
      - This reordering could be built into the rules, complicating all rules.
      - Assuming a total order on V and L I could implement a total order on 
        the formulas and use ordered lists for sequents only. Then intentional
        equality would be sequent equality. However, reordering calls would
        be needed in the rules.
      - I therefore push the whole problem now to the conceptual
        level: I use simple unordered lists and standard equality on
        them in proofs. Rules and hypothesis must be order invariant.
  *)

  Unset Elimination Schemes.

  Inductive proof(rules : set sequent_rule)(hypotheses : set sequent)
                                                           : sequent -> Type :=
    | assume : forall(gamma : sequent), 
        hypotheses gamma -> proof rules hypotheses gamma
    | rule : forall(r : sequent_rule), rules r -> 
        dep_list sequent (proof rules hypotheses) (assumptions r) ->
          proof rules hypotheses (conclusion r).

  Set Elimination Schemes.

  (***************************************************************************
   ***************************************************************************
   ***************************************************************************)


  (***************************************************************************)
  (** ***  Recursion for proofs  *)
  (***************************************************************************)

  Fixpoint proof_rect(rules : set sequent_rule)
             (hypotheses : set sequent)
             (T : sequent -> Type)
             (assume_fn : forall(gamma : sequent), hypotheses gamma -> T gamma)
             (rule_fn :
                 forall(r : sequent_rule)(in_rules : rules r),
                   (dep_list sequent T (assumptions r)) ->
                     T (conclusion r))
             (s : sequent)(p : proof rules hypotheses s)
           : T s :=
    match p with
      | assume g g_hyp => assume_fn g g_hyp
      | rule r r_rules subproofs =>
          rule_fn r r_rules
            ((fix map_subproofs(sl : list sequent)
                     (subproofs : dep_list sequent (proof rules hypotheses) sl)
                            : dep_list sequent T sl :=
               match subproofs with
                 | dep_nil => dep_nil
                 | dep_cons s sl p tl =>
                     dep_cons s sl 
                       (proof_rect rules hypotheses T assume_fn rule_fn s p)
                       (map_subproofs sl tl)
               end)
              (assumptions r) subproofs)
    end.

  Lemma proof_rect_rule :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (T : sequent -> Type)
          (assume_fn : forall(g : sequent), hypotheses g -> T g)
          (rule_fn : forall(r : sequent_rule)(in_rules : rules r),
             (dep_list sequent T (assumptions r)) -> T (conclusion r))
          (r : sequent_rule)(r_rules : rules r)
          (subproofs : dep_list sequent (proof rules hypotheses) 
                         (assumptions r)),
      proof_rect rules hypotheses T assume_fn rule_fn (conclusion r) 
          (rule rules hypotheses r r_rules subproofs)
      = rule_fn r r_rules 
          (dep_map_dep_dep 
             (proof_rect rules hypotheses T assume_fn rule_fn)
             (assumptions r) subproofs).
  Proof.
    intros rules hypotheses T assume_fn rule_fn r r_rules subproofs.
    simpl.
    assert 
      (((fix map_subproofs (sl : list sequent)
                         (subproofs0 : dep_list sequent
                                         (proof rules hypotheses) sl)
                         {struct subproofs0} : dep_list sequent T sl :=
           match
             subproofs0 in (dep_list _ _ l) return (dep_list sequent T l)
           with
           | dep_nil => dep_nil
           | dep_cons s sl0 p tl =>
               dep_cons s sl0
                 (proof_rect rules hypotheses T assume_fn rule_fn s p)
                 (map_subproofs sl0 tl)
           end) (assumptions r) subproofs)
       = (dep_map_dep_dep (proof_rect rules hypotheses T assume_fn rule_fn)
           (assumptions r) subproofs)).
      revert subproofs.
      generalize (proof_rect rules hypotheses T assume_fn rule_fn).
      generalize (assumptions r).
      clear. 
      induction subproofs.
        simpl.
        trivial.
      rewrite IHsubproofs.
      simpl.
      trivial.
    rewrite H.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Proof depth *)
  (***************************************************************************)

  Definition proof_depth{rules : sequent_rule -> Prop}
                        {hypotheses : sequent -> Prop}
                        {s : sequent}
                        (p : proof rules hypotheses s) : nat :=
    proof_rect rules hypotheses (fun _ => nat)
      (fun _ _ => 1)
      (fun r _ l => S (nat_list_max (list_of_dep_list (assumptions r) l)))
      s p.

  Lemma proof_depth_assume :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (s : sequent)(in_hyp : hypotheses s),
      proof_depth (assume rules hypotheses s in_hyp) = 1.
  Proof.
    intros rules hypotheses s in_hypo.
    unfold proof_depth in *.
    trivial.
  Qed.

  Lemma proof_depth_rule :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (r : sequent_rule)(r_rules : rules r)
          (subproofs : dep_list sequent (proof rules hypotheses) 
                         (assumptions r)),
     proof_depth (rule rules hypotheses r r_rules subproofs) =
       1 + (nat_list_max (dep_map_dep_const 
                            (@proof_depth rules hypotheses)
                            (assumptions r)
                            subproofs)).
  Proof.
    intros rules hypotheses r r_rules subproofs.
    unfold proof_depth at 1.
    rewrite proof_rect_rule.
    assert ((proof_rect rules hypotheses (fun _ : sequent => nat)
                 (fun (gamma : sequent) (_ : hypotheses gamma) => 1)
                 (fun (r0 : sequent_rule) (_ : rules r0)
                    (l : dep_list sequent (fun _ : sequent => nat)
                           (assumptions r0)) =>
                  S (nat_list_max (list_of_dep_list (assumptions r0) l))))
            = @proof_depth rules hypotheses).
      trivial.
    rewrite H.
    rewrite list_of_dep_list_dep_map_dep_dep.
    trivial.
  Qed.

  Lemma proof_depth_rule_le :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (r : sequent_rule)(r_rules : rules r)(n : nat)
          (subproofs : dep_list sequent (proof rules hypotheses) 
                         (assumptions r)),
      every_dep_nth
        (fun(s : sequent)(p : proof rules hypotheses s) => proof_depth p <= n)
        (assumptions r)
        subproofs 
      -> proof_depth (rule rules hypotheses r r_rules subproofs) <= S n.
  Proof.
    intros rules hypotheses r r_rules n subproofs H.
    rewrite proof_depth_rule.
    apply le_n_S.
    apply nat_list_max_le.
    intros i i_less.
    erewrite nth_dep_map_dep_const.
    apply H.
  Qed.

  Lemma proof_depth_rule_le_inv :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (r : sequent_rule)(r_rules : rules r)(n : nat)
          (subproofs : dep_list sequent (proof rules hypotheses) 
                         (assumptions r)),
      proof_depth (rule rules hypotheses r r_rules subproofs) <= S n ->
        every_dep_nth
          (fun(s : sequent)(p : proof rules hypotheses s) => 
              proof_depth p <= n)
          (assumptions r)
          subproofs. 
  Proof.
    intros rules hypotheses r r_rules n subproofs H.
    rewrite proof_depth_rule in H.
    apply le_S_n in H.
    intros i i_less.
    rewrite nth_dep_map_dep_const_inv.
    apply nat_list_max_le_inv.
    trivial.
  Qed.

  Lemma proof_depth_0 : 
    forall(X : Type)
          (rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (s : sequent)(p : proof rules hypotheses s),
      proof_depth p <= 0 -> X.
  Proof.
    intros X rules hypotheses s p H.
    destruct p.
      rewrite proof_depth_assume in H.
      omega.
    rewrite proof_depth_rule in H.
    omega.
  Qed.


  Lemma proof_depth_rule_1 : 
    forall(rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (r : sequent_rule)(in_rules : rules r)
          (subproofs : 
             dep_list sequent (proof rules hypotheses) (assumptions r)),
      proof_depth (rule rules hypotheses r in_rules subproofs) <= 1
        -> assumptions r = [].
  Proof.
    intros rules hypotheses r in_rules subproofs H.
    apply proof_depth_rule_le_inv in H.
    destruct subproofs.
      trivial.
    assert (H0 := every_dep_nth_head _ _ _ _ _ H).
    simpl in H0.
    eapply proof_depth_0.
    eexact H0.
  Qed.


  (***************************************************************************)
  (** *** Induction on proofs *)
  (***************************************************************************)

  Lemma proof_ind :
    forall(rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (P : forall(s : sequent), proof rules hypotheses s -> Prop),
      (forall(gamma : sequent)(in_hypotheses : hypotheses gamma),
                  P gamma (assume rules hypotheses gamma in_hypotheses)) ->
      (forall(r : sequent_rule)(in_rules : rules r)
             (pl : dep_list sequent (proof rules hypotheses) (assumptions r)),
          every_dep_nth P (assumptions r) pl ->
                 P (conclusion r) (rule rules hypotheses r in_rules pl)) ->
      forall(s : sequent)(p : proof rules hypotheses s), P s p.
  Proof.
    intros rules hypotheses P H H0.
    assert (forall (n : nat)(s : sequent)(p : proof rules hypotheses s), 
              proof_depth p <= n -> P s p).
      induction n.
        intros s p H1.
        clear - H1.
        unfold proof_depth in *.
        destruct p.
          simpl in H1.
          omega.
        simpl in H1.
        omega.
      intros s p H1.
      destruct n.
        destruct p.
          apply H.
        assert (H2 := proof_depth_rule_1 _ _ _ _ _ H1).
        apply H0.
        clear H1.
        destruct d.
          apply every_dep_nth_empty.
        discriminate.
      destruct p.
        apply H.
      apply H0; clear H0.
      apply proof_depth_rule_le_inv in H1.
      intros i i_less.
      apply IHn.
      apply H1.
    intros s p.
    apply (H1 (proof_depth p)).
    trivial.
  Qed.

  (** Another, simpler induction scheme. Often, this simple induction
      suffices. 
   *)
  Lemma proof_sequent_ind : 
    forall(rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (P : sequent -> Prop),
      (forall(gamma : sequent), hypotheses gamma -> P gamma) ->
      (forall(r : sequent_rule), 
                rules r -> every_nth P (assumptions r) -> P (conclusion r))
        -> forall(s : sequent), proof rules hypotheses s -> P s.
  Proof.
    intros rules hypotheses P H H0.
    apply proof_ind.
      trivial.
    intros r in_rules pl H1.
    apply H0.
      trivial.
    intros i i_less.
    apply H1.
  Qed.


  (***************************************************************************)
  (** *** Provability *)
  (***************************************************************************)

  Definition provable(rules : set sequent_rule)(hypotheses : set sequent)
                     (s : sequent) : Prop :=
    exists _ : proof rules hypotheses s, True.

  Lemma provable_with_assumption :
    forall(rules : set sequent_rule)(hypotheses : set sequent)(s : sequent),
      hypotheses s ->
        provable rules hypotheses s.
  Proof.
    intros rules hypotheses s H.
    split.
      apply assume.
      trivial.
    trivial.
  Qed.

  Lemma provable_with_rule :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (assum : list sequent)(s : sequent),
      rules {| assumptions := assum; conclusion := s |} ->
      every_nth (provable rules hypotheses) assum ->
        provable rules hypotheses s.
  Proof.
    intros rules hypotheses assum s H H0.
    apply every_nth_exists in H0.
    destruct H0 as [sub_proofs].
    clear H0.
    exists (rule rules hypotheses {| assumptions := assum; conclusion := s |} 
                 H sub_proofs).
    trivial.
  Qed.


  Definition provable_at_depth(rules : set sequent_rule)
                              (hypotheses : set sequent)
                              (d : nat)
                              (s : sequent) : Prop :=
    exists p : proof rules hypotheses s, proof_depth p <= d.


  Lemma provable_at_proof_depth :
    forall(rules : set sequent_rule)(hypotheses : set sequent)(s : sequent)
          (p : proof rules hypotheses s),
      provable_at_depth rules hypotheses (proof_depth p) s.
  Proof.
    intros rules hypotheses s p.
    unfold provable_at_depth in *.
    exists p.
    trivial.
  Qed.

  Lemma provable_at_depth_provable :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (n : nat)(s : sequent),
      provable_at_depth rules hypotheses n s ->
        provable rules hypotheses s.
  Proof.
    unfold provable_at_depth, provable in *.
    intros rules hypotheses n s H.
    destruct H.
    exists x.
    trivial.
  Qed.

  Lemma provable_at_depth_0 : 
    forall(P : Prop)
          (rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (s : sequent),
      provable_at_depth rules hypotheses 0 s -> P.
  Proof.
    intros P rules hypotheses s H.
    destruct H.
    eapply proof_depth_0.
    eexact H.
  Qed.

  Lemma provable_at_depth_with_rule :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (assum : list sequent)(d : nat)(s : sequent),
      rules {| assumptions := assum; conclusion := s |} ->
      every_nth (provable_at_depth rules hypotheses d) assum ->
        provable_at_depth rules hypotheses (S d) s.
  Proof.
    intros rules hypotheses assum d s H H0.
    apply every_nth_exists in H0.
    destruct H0 as [sub_proofs].
    exists (rule rules hypotheses {| assumptions := assum; conclusion := s |} 
                 H sub_proofs).
    apply proof_depth_rule_le with (rules := rules) (r_rules := H).
    trivial.
  Qed.

  Lemma provable_at_depth_destruct :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (d : nat)(s : sequent),
      provable_at_depth rules hypotheses (S d) s ->
        hypotheses s \/
        (provable rules hypotheses s /\
         exists(r : sequent_rule), rules r /\
           s = conclusion r /\
           every_nth (provable_at_depth rules hypotheses d) (assumptions r)).
  Proof.
    intros rules hypotheses d s H.
    destruct H.
    destruct x.
      left.
      trivial.
    right.
    split.
      exists (rule rules hypotheses r r0 d0).
      trivial.
    exists r.
    repeat split; trivial.
    apply proof_depth_rule_le_inv in H.
    intros i i_less.
    specialize (H i i_less).
    simpl in *.
    exists (dep_nth (assumptions r) d0 i i_less).
    trivial.
  Qed.

  (** Induction on the proof depth, formulated with provability *)
  Lemma proof_depth_sequent_ind :
    forall(rules : sequent_rule -> Prop)(hypotheses : sequent -> Prop)
          (P : nat -> sequent -> Prop),
      (forall(n : nat),
         (forall(s : sequent),
            provable_at_depth rules hypotheses n s -> P n s) ->
         forall(s : sequent),
            provable_at_depth rules hypotheses (S n) s -> P (S n) s)
      -> forall(s : sequent)(p : proof rules hypotheses s), 
           P (proof_depth p) s.
  Proof.
    intros rules hypotheses P H s p.
    assert (forall(n : nat)(s : sequent),
              provable_at_depth rules hypotheses n s -> P n s).
      clear s p.
      induction n.
        intros s H0.
        eapply provable_at_depth_0.
        eexact H0.
      apply H.
      trivial.
    apply (H0 (proof_depth p) _).
    apply provable_at_proof_depth.
  Qed.


  (***************************************************************************)
  (** *** Interpretation of sequents, page 9 *)
  (***************************************************************************)

  (** The paper uses \hat for this. There are two functions for the
      finite disjunction of a sequent. The first works on the empty
      sequent too, but needs a propositional variable to construct
      false for it. This cannot be used for the TX semantics, because
      the result is not a [prop_modal_prop] formula. The second
      function avoids this by working only on non-empty sequents.
   *)
  Fixpoint or_formula_of_sequent_iter(res : lambda_formula)
                                  (l : sequent) : lambda_formula :=
    match l with
      | [] => res
      | f :: r => or_formula_of_sequent_iter (lambda_or res f) r
    end.

  Lemma or_formula_of_sequent_iter_append :
    forall(s1 s2 : sequent)(f : lambda_formula),
      or_formula_of_sequent_iter f (s1 ++ s2) =
        or_formula_of_sequent_iter (or_formula_of_sequent_iter f s1) s2.
  Proof.
    induction s1.
      simpl.
      trivial.
    rename a into f'.
    intros s2 f.
    simpl.
    apply IHs1.
  Qed.

  Lemma or_formula_of_sequent_iter_rev :
    forall(f g : lambda_formula)(s : sequent),
      or_formula_of_sequent_iter f (rev (g :: s)) =
        lambda_or (or_formula_of_sequent_iter f (rev s)) g.
  Proof.
    intros f g s.
    simpl.
    apply or_formula_of_sequent_iter_append.
  Qed.

  Definition or_formula_of_sequent(l : sequent)(nonempty_v : V) 
                                                           : lambda_formula :=
    match l with
      | [] => lambda_false nonempty_v
      | f :: r => or_formula_of_sequent_iter f r
    end.

  Definition or_formula_of_ne_sequent(l : sequent)(nonempty_l : l <> []) 
                                                           : lambda_formula :=
    match l return l <> [] -> lambda_formula with
      | [] => fun(H : [] <> []) => False_rect lambda_formula (H (eq_refl []))
      | f :: r => fun _ => or_formula_of_sequent_iter f r
    end nonempty_l.

  Lemma or_formula_of_ne_sequent_tcc_irr :
    forall(l : sequent)(nonempty_l_1 nonempty_l_2 : l <> []),
      or_formula_of_ne_sequent l nonempty_l_1 = 
        or_formula_of_ne_sequent l nonempty_l_2.
  Proof.
    intros l nonempty_l_1 nonempty_l_2.
    destruct l.
      exfalso.
      apply nonempty_l_1.
      trivial.
    trivial.
  Qed.

  Lemma or_formula_of_sequent_nonempty :
    forall(s : sequent)(nonempty_v : V)(nonempty_s : s <> []),
      or_formula_of_sequent s nonempty_v = 
        or_formula_of_ne_sequent s nonempty_s.
  Proof.
    intros s nonempty_v nonempty_s.
    destruct s.
      exfalso.
      apply nonempty_s.
      trivial.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Proof monotonicity (wrt. the rules and hypothesis)  *)
  (***************************************************************************)

  Lemma proof_depth_mono : 
    forall(rules1 rules2 : set sequent_rule)(hyp1 hyp2 : set sequent)
          (n : nat)(s : sequent),
      subset rules1 rules2 -> 
      subset hyp1 hyp2 ->
      provable_at_depth rules1 hyp1 n s -> 
        provable_at_depth rules2 hyp2 n s.
  Proof.
    intros rules1 rules2 hyp1 hyp2 n s H H0 H1.
    unfold provable_at_depth in *.
    decompose [ex] H1; clear H1.
    revert n H2.
    induction x.
      intros n H2.
      exists (assume rules2 hyp2 gamma (H0 gamma in_hypotheses)).
      rewrite proof_depth_assume in *.
      trivial.
    intros n H2.
    destruct n.
      eapply proof_depth_0.
      eexact H2.
    lapply (every_nth_exists 
             (fun(s : sequent)(p2 : proof rules2 hyp2 s) => proof_depth p2 <= n)
             (assumptions r)).
      intros H3.
      destruct H3.
      exists (rule rules2 hyp2 r (H r in_rules) x).
      apply proof_depth_rule_le.
      trivial.
    intros i i_less.
    apply H1.
    eapply proof_depth_rule_le_inv.
    eexact H2.
  Qed.

  Lemma proof_mono : 
    forall(rules1 rules2 : set sequent_rule)(hyp1 hyp2 : set sequent)
          (s : sequent),
      subset rules1 rules2 -> 
      subset hyp1 hyp2 ->
      provable rules1 hyp1 s -> 
        provable rules2 hyp2 s.
  Proof.
    intros rules1 rules2 hyp1 hyp2 s H H0 H1.
    destruct H1.
    eapply provable_at_depth_provable.
    eapply proof_depth_mono.
        eexact H.
      eexact H0.
    apply provable_at_proof_depth with (p := x).
  Qed.

  Lemma proof_mono_hyp : 
    forall(rules : set sequent_rule)(hyp1 hyp2 : set sequent)(s : sequent),
      subset hyp1 hyp2 ->
      provable rules hyp1 s -> 
        provable rules hyp2 s.
  Proof.
    intros rules hyp1 hyp2 s H H0.
    eapply proof_mono.
        apply subset_refl.
      eexact H.
    trivial.
  Qed.

  Lemma proof_mono_rules : 
    forall(rules1 rules2 : set sequent_rule)(hyp : set sequent)(s : sequent),
      subset rules1 rules2 -> 
      provable rules1 hyp s -> 
        provable rules2 hyp s.
  Proof.
    intros rules1 rules2 hyp s H H0.
    eapply proof_mono.
        eexact H.
      apply subset_refl.
    trivial.
  Qed.

  Lemma proof_set_equal : 
    forall(rules1 rules2 : set sequent_rule)(hyp1 hyp2 : set sequent)
          (s : sequent),
      set_equal rules1 rules2 -> 
      set_equal hyp1 hyp2 ->
      provable rules1 hyp1 s -> 
        provable rules2 hyp2 s.
  Proof.
    intros rules1 rules2 hyp1 hyp2 s H H0 H1.
    eapply proof_mono.
        apply set_equal_implies_subset.
        eexact H.
      apply set_equal_implies_subset.
      eexact H0.
    trivial.
  Qed.

  Lemma proof_set_equal_rules : 
    forall(rules1 rules2 : set sequent_rule)(hyp : set sequent)
          (s : sequent),
      set_equal rules1 rules2 -> 
      provable rules1 hyp s -> 
        provable rules2 hyp s.
  Proof.
    intros rules1 rules2 hyp s H H0.
    eapply proof_set_equal.
        eexact H.
      apply set_equal_refl.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Provability is closed under reordering  *)
  (***************************************************************************)

  Definition sequent_multiset(ss : set sequent) : Prop :=
    forall(s r : sequent), ss s -> list_reorder s r -> ss r.

  Lemma sequent_multiset_empty : sequent_multiset empty_sequent_set.
  Proof.
    unfold sequent_multiset, empty_sequent_set in *.
    intros s r H H0.
    contradiction.
  Qed.

  Definition reordered_rule(or : sequent_rule)(s : sequent)(rr : sequent_rule) 
                                                                      : Prop :=
    s = conclusion rr /\
    length (assumptions or) = length (assumptions rr) /\
    forall(n : nat)(n_less_rs : n < length (assumptions or))
          (n_less_rr : n < length (assumptions rr)),
      list_reorder (nth (assumptions or) n n_less_rs)
                   (nth (assumptions rr) n n_less_rr).

  Definition rule_multiset(rs : set sequent_rule) : Prop :=
    forall(or : sequent_rule)(s : sequent),
      rs or ->
      list_reorder (conclusion or) s ->
      exists(rr : sequent_rule),
        reordered_rule or s rr /\
        rs rr.

  Lemma set_equal_multiset : forall(rs1 rs2 : set sequent_rule),
    set_equal rs1 rs2 ->
    rule_multiset rs1 ->
      rule_multiset rs2.
  Proof.
    intros rs1 rs2 H H0.
    unfold rule_multiset in *.
    intros or s H1 H2.
    apply H in H1.
    specialize (H0 or s H1 H2).
    decompose [ex and] H0; clear H0.
    exists x.
    split.
      trivial.
    apply H.
    trivial.
  Qed.

  Lemma multiset_union : forall(rs1 rs2 : set sequent_rule),
    rule_multiset rs1 ->
    rule_multiset rs2 ->
      rule_multiset (union rs1 rs2).
  Proof.
    intros rs1 rs2 H H0.
    unfold rule_multiset in *.
    intros or s H1 H2.
    destruct H1.
      clear H0.
      specialize (H or s H1 H2).
      decompose [ex and] H; clear H.
      exists x.
      split.
        trivial.
      left.
      trivial.
    clear H.
    specialize (H0 or s H1 H2).
    decompose [ex and] H0; clear H0.
    exists x.
    split.
      trivial.
    right.
    trivial.
  Qed.

  Lemma multiset_depth_provability : 
    forall(rules : set sequent_rule)(hypothesis : set sequent)
          (d : nat)(s r : sequent),
      rule_multiset rules ->
      sequent_multiset hypothesis ->
      list_reorder s r ->
      provable_at_depth rules hypothesis d s ->
        provable_at_depth rules hypothesis d r.
  Proof.
    intros rules hypothesis d s r H H0 H1 H2.
    destruct H2.
    revert d r H2 H1.
    induction x.
      intros d r H2 H1.
      rewrite proof_depth_assume in H2.
      specialize (H0 _ _ in_hypotheses H1).
      exists (assume rules hypothesis r H0).
      rewrite proof_depth_assume.
      trivial.
    intros d s H2 H3.
    destruct d.
      eapply proof_depth_0.
      eexact H2.
    specialize (H r s in_rules H3).
    unfold reordered_rule in *.
    decompose [ex and or dep_and] H; clear H.
    subst s.
    lapply (every_nth_exists 
              (fun(s : sequent)(p : proof rules hypothesis s) => 
                  proof_depth p <= d)
              (assumptions x)).
      clear pl H1 H2.
      intros H.
      destruct H.
      exists (rule rules hypothesis x H6 x0).
      apply proof_depth_rule_le.
      trivial.
    intros i i_less.
    assert (i < length (assumptions r)).
      rewrite H5.
      trivial.
    specialize (H1 i H d (nth (assumptions x) i i_less)).
    apply H1.
      eapply proof_depth_rule_le_inv.
      eexact H2.
    apply H8.
  Qed.

  Lemma multiset_provability : 
    forall(rules : set sequent_rule)(hypothesis : set sequent)(s r : sequent),
      rule_multiset rules ->
      sequent_multiset hypothesis ->
      list_reorder s r ->
      provable rules hypothesis s ->
        provable rules hypothesis r.
  Proof.
    intros rules hypothesis s r H H0 H1 H2.
    destruct H2.
    eapply provable_at_depth_provable.
    eapply multiset_depth_provability; eauto.
    apply provable_at_proof_depth with (p := x).
  Qed.



  (**************************************************************************)
  (** *** Make sequent multisets out of lists *)
  (**************************************************************************)

  Definition reordered_sequent_list_set(l : list sequent) : set sequent := 
    fun(s1 : sequent) => exists(s2 : sequent), list_reorder s1 s2 /\ In s2 l.

  Lemma sequent_multiset_reordered_sequent_list_set : 
    forall(l : list sequent), sequent_multiset (reordered_sequent_list_set l).
  Proof.
    unfold sequent_multiset, reordered_sequent_list_set in *.
    intros l s1 s2 H H0.
    destruct H as [s3].
    destruct H.
    exists s3.
    split; trivial.
    eapply list_reorder_trans.
      apply list_reorder_symm.
      eexact H0.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Plug proofs for hypothesis  *)
  (***************************************************************************)

  Lemma plug_hypothesis_proof :
    forall(rules : set sequent_rule)(provable_hyp hyp : set sequent)
          (s : sequent),
      (forall(h : sequent), provable_hyp h -> provable rules hyp h) ->
      provable rules (union provable_hyp hyp) s ->
        provable rules hyp s.
  Proof.
    intros rules provable_hyp hyp s H H0.
    destruct H0.
    clear H0.
    induction x using proof_sequent_ind.
      destruct H0.
        apply H.
        trivial.
      exists (assume rules hyp gamma H0).
      trivial.
    apply every_nth_exists in H1.
    destruct H1.
    clear H1.
    exists (rule rules hyp r H0 x).
    trivial.
  Qed.

  Lemma plug_empty_hypothesis_proof :
    forall(rules : set sequent_rule)(provable_hyp : set sequent)
          (s : sequent),
      (forall(h : sequent), provable_hyp h -> 
             provable rules empty_sequent_set h) ->
      provable rules provable_hyp s ->
        provable rules empty_sequent_set s.
  Proof.
    intros rules provable_hyp s H H0.
    eapply plug_hypothesis_proof.
      eexact H.
    eapply proof_mono_hyp.
      apply subset_union_left.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Provability with different rules *)
  (***************************************************************************)

  Lemma change_rules_hyp_provability :
    forall(rules1 rules2 : set sequent_rule)(hyp1 hyp2 : set sequent),
      sequent_multiset hyp2 ->
      rule_multiset rules2 ->
      (forall(r : sequent_rule), rules1 r ->
         provable rules2 
                  (union (reordered_sequent_list_set (assumptions r)) hyp2)
            (conclusion r)) ->
      (forall(h : sequent), hyp1 h -> provable rules2 hyp2 h) ->
      forall(s : sequent),
        provable rules1 hyp1 s ->
          provable rules2 hyp2 s.
  Proof.
    intros rules1 rules2 hyp1 hyp2 H H0 H1 H2 s H3.
    destruct H3 as [p].
    clear H3.
    induction p using proof_sequent_ind.
      auto.
    apply plug_hypothesis_proof 
             with (provable_hyp := reordered_sequent_list_set (assumptions r)).
      clear H1 H2.
      intros h H1.
      unfold reordered_sequent_list_set in *.
      decompose [ex and] H1; clear H1.
      rename x into h'.
      apply multiset_provability with (s := h'); trivial.
        apply list_reorder_symm.
        trivial.
      eapply every_nth_In.
        eexact H4.
      trivial.
    apply H1.
    trivial.
  Qed.

  Lemma change_rules_provability :
    forall(rules1 rules2 : set sequent_rule),
      rule_multiset rules2 ->
      (forall(r : sequent_rule), rules1 r ->
         provable rules2 (reordered_sequent_list_set (assumptions r)) 
             (conclusion r)) ->
      forall(s : sequent),
        provable rules1 empty_sequent_set s ->
          provable rules2 empty_sequent_set s.
  Proof.
    intros rules1 rules2 H H0 s H1.
    eapply change_rules_hyp_provability with (hyp1 := empty_sequent_set).
            apply sequent_multiset_empty.
          trivial.
        intros r H2.
        eapply proof_set_equal.
            apply set_equal_refl.
          apply set_equal_symm.
          apply set_equal_union_empty_right.
          apply set_equal_refl.
        apply H0.
        eexact H2.
      intros h H2.
      contradiction.
    trivial.
  Qed.


  (***************************************************************************)
  (** ***  Modal rank, page 5  *)
  (***************************************************************************)

  (** In the paper propositional formulas have rank 0, but the paper uses
      rank -1 in exceptional cases, eg F_n on page 6 and in 
      Lemma 3.7, page 13. There, F_{-1} is the empty set.
     
      I therefore use rank 1 for propositional formulas. No formula has 
      rank 0, but the empty sequent has rank 0. Every formula with a modality
      has at least rank 2, even if the modality has arity 0.
   *)
  Definition modal_rank : lambda_formula -> nat :=
    lambda_formula_rec nat (fun _ => 1) id max 
      (fun _ (subranks : counted_list nat _) => 
         1 + (nat_list_max (1 :: (list_of_counted_list subranks)))).

  Lemma modal_rank_char :
    forall(f : lambda_formula),
      modal_rank f =
        match f with
          | lf_prop _ => 1
          | lf_neg f => modal_rank f
          | lf_and f1 f2 => max (modal_rank f1) (modal_rank f2)
          | lf_modal op args =>
            1 + (nat_list_max 
                    (1 :: (map modal_rank (list_of_counted_list args))))
        end.
  Proof.
    intros f.
    unfold modal_rank at 1.
    rewrite lambda_formula_rec_char.
    simpl.
    destruct f.
          trivial.
        trivial.
      trivial.
    rewrite list_of_counted_list_map.
    trivial.
  Qed.

  Definition minimal_sequent_rank(s : sequent) : nat :=
    nat_list_max (map modal_rank s).

  (* 
   * Lemma minimal_sequent_rank_reorder : forall(s1 s2 : sequent),
   *   list_reorder s1 s2 -> 
   *     minimal_sequent_rank s1 = minimal_sequent_rank s2.
   * Proof.
   *   intros s1 s2 H.
   *   unfold minimal_sequent_rank in *.
   *   apply nat_list_max_reorder.
   *   apply list_reorder_map.
   *   trivial.
   * Qed.
   *)

  Lemma modal_rank_ge_1 : forall(f : lambda_formula),
    0 < modal_rank f.
  Proof.
    induction f.
          rewrite modal_rank_char.
          omega.
        rewrite modal_rank_char.
        trivial.
      rewrite modal_rank_char.
      assert (H0 := Max.le_max_l (modal_rank f1) (modal_rank f2)).
      assert (H1 := Max.le_max_r (modal_rank f1) (modal_rank f2)).      
      omega.
    rewrite modal_rank_char.
    omega.
  Qed.

  Lemma minimal_sequent_rank_gt_0 : forall(s : sequent),
    s <> [] -> 0 < minimal_sequent_rank s.
  Proof.
    clear. 
    intros s H.
    destruct s.
      exfalso.
      auto.
    unfold minimal_sequent_rank in *.
    simpl.
    assert (H0 := modal_rank_ge_1 l).
    assert (H1 := Max.le_max_l (modal_rank l) 
                               (nat_list_max (map modal_rank s))).
    omega.
  Qed.


  (** This is used for F_n(\Lambda), page 6 *)
  Definition rank_formula(n : nat) : set lambda_formula :=
    fun(f : lambda_formula) => modal_rank f <= n.

  Lemma rank_formula_zero : 
    set_equal (rank_formula 0) (empty_set lambda_formula).
  Proof.
    intros f.
    split.
      intros H.
      exfalso.
      unfold rank_formula in *.
      assert (H0 := modal_rank_ge_1 f).
      omega.
    intros H.
    contradiction.
  Qed.

  Lemma rank_formula_zero_TCC : 
    forall(A : Type)(f : lambda_formula), rank_formula 0 f -> A.
  Proof.
    intros A f H.
    apply rank_formula_zero in H.
    contradiction.
  Qed.

  Lemma rank_formula_ge : 
    forall(f : lambda_formula)(n1 n2 : nat),
      n1 <= n2 ->
      rank_formula n1 f ->
        rank_formula n2 f.
  Proof.
    intros f n1 n2 H H0.
    unfold rank_formula in *.
    omega.
  Qed.

  Lemma rank_formula_lf_prop : forall(v : V)(n : nat),
    1 <= n -> rank_formula n (lf_prop v).
  Proof.
    intros v n H.
    unfold rank_formula in *.
    rewrite modal_rank_char.
    trivial.
  Qed.

  Lemma rank_formula_lf_neg : forall(n : nat)(f : lambda_formula),
    rank_formula n (lf_neg f) <-> rank_formula n f.
  Proof.
    intros n f.
    unfold rank_formula in *.
    rewrite modal_rank_char.
    trivial.
    tauto.
  Qed.

  Lemma rank_formula_lf_neg_TCC : forall(n : nat)(f : lambda_formula),
    rank_formula n (lf_neg f) -> rank_formula n f.
  Proof.
    intros n f.
    apply rank_formula_lf_neg.
  Qed.

  Lemma rank_formula_lf_and : forall(n : nat)(f1 f2 : lambda_formula),
    rank_formula n f1 -> rank_formula n f2 -> rank_formula n (lf_and f1 f2).
  Proof.
    intros n f1 f2 H H0.
    unfold rank_formula in *.
    rewrite modal_rank_char.
    apply Max.max_lub.
      trivial.
    trivial.
  Qed.

  Lemma rank_formula_and_left : forall(n : nat)(f1 f2 : lambda_formula),
    rank_formula n (lf_and f1 f2) -> rank_formula n f1.
  Proof.
    intros n f1 f2 H.
    unfold rank_formula in *.
    rewrite modal_rank_char in H.
    apply (Max.max_lub_l _ _ _ H).
  Qed.

  Lemma rank_formula_and_right : forall(n : nat)(f1 f2 : lambda_formula),
    rank_formula n (lf_and f1 f2) -> rank_formula n f2.
  Proof.
    intros n f1 f2 H.
    unfold rank_formula in *.
    rewrite modal_rank_char in H.
    eapply Max.max_lub_r.
    eexact H.
  Qed.

  Lemma rank_formula_false : forall(nonempty_v : V)(n : nat),
    1 <= n -> rank_formula n (lambda_false nonempty_v).
  Proof.
    intros nonempty_v n H.
    unfold lambda_false in *.
    apply rank_formula_lf_and.
      apply rank_formula_lf_prop.
      trivial.
    apply rank_formula_lf_neg.
    apply rank_formula_lf_prop.
    trivial.
  Qed.

  Lemma rank_formula_neg_form_maybe :
    forall(fs : set lambda_formula)(n : nat)(f : lambda_formula),
      (forall(f : lambda_formula), fs f -> rank_formula n f) ->
      neg_form_maybe fs f ->
        rank_formula n f.
  Proof.
    intros fs n f H H0.
    destruct f.
          apply H.
          apply H0.
        apply rank_formula_lf_neg.
        apply H.
        apply H0.
      apply H.
      apply H0.
    apply H.
    apply H0.
  Qed.

  Lemma rank_formula_neg_and_over :
    forall(fs : set lambda_formula)(n : nat)(f : lambda_formula),
      (forall(f : lambda_formula), fs f -> rank_formula n f) ->
      neg_and_over fs f ->
        rank_formula n f.
  Proof.
    induction f.          
          intros H H0.
          apply H.
          apply H0.
        intros H H0.
        apply rank_formula_lf_neg.
        apply IHf.
          trivial.
        apply H0.
      intros H H0.
      apply rank_formula_lf_and.
        apply IHf1.
          trivial.
        apply H0.
      apply IHf2.
        trivial.
      apply H0.
    intros H0 H1.
    apply H0.
    apply H1.
  Qed.

  (** Unusual argument order because this is used to instantiate 
      form_prop_or in some_neg_form.v.
   *)
  Lemma rank_formula_lambda_or :
    forall(n : nat)(f1 f2 : lambda_formula),
      rank_formula n f1 -> 
      rank_formula n f2 -> 
        rank_formula n (lambda_or f1 f2).
  Proof.
    intros f1 f2 n H H0.
    unfold lambda_or in *.
    apply rank_formula_lf_neg.
    apply rank_formula_lf_and.
      apply rank_formula_lf_neg.
      trivial.
    apply rank_formula_lf_neg.
    trivial.
  Qed.

  Lemma rank_formula_lambda_or_rev :
    forall(f1 f2 : lambda_formula)(n : nat),
      rank_formula n (lambda_or f1 f2) ->
      rank_formula n f1 /\ rank_formula n f2.
  Proof.
    intros f1 f2 n H.
    unfold lambda_or in *.
    rewrite rank_formula_lf_neg in H.
    split.
      apply rank_formula_and_left in H.
      rewrite rank_formula_lf_neg in H.
      trivial.
    apply rank_formula_and_right in H.
    rewrite rank_formula_lf_neg in H.
    trivial.
  Qed.

  Lemma rank_formula_prop_form : forall(f : lambda_formula),
    prop_form f -> rank_formula 1 f.
  Proof.
    intros f H.
    destruct f.
          unfold rank_formula in *.
          trivial.
        contradiction.
      contradiction.
    contradiction.
  Qed.

  Lemma rank_formula_modal :
    forall(op : operator L)
          (args : counted_list lambda_formula (arity L op))
          (n : nat),
      1 < n ->
      every_nth (rank_formula (pred n)) (list_of_counted_list args) ->
        rank_formula n (lf_modal op args).
  Proof.
    intros op args n H H0.
    unfold rank_formula in *.
    rewrite modal_rank_char.
    destruct n.
      exfalso.
      omega.
    apply le_n_S.
    destruct n.
      exfalso.
      omega.
    apply nat_list_max_le.
    unfold every_nth in *.
    intros n0 n_less.
    destruct n0.
      simpl.
      omega.
    simpl in *.
    rewrite nth_map.
    apply H0.
  Qed.


  (***************************************************************************)
  (** *** Rank of sequents *)
  (***************************************************************************)

  Definition rank_sequent(n : nat) : set sequent :=
    fun(s : sequent) => every_nth (rank_formula n) s.

  Lemma rank_sequent_mono : forall(n1 n2 : nat),
    n1 <= n2 -> subset (rank_sequent n1) (rank_sequent n2).
  Proof.
    unfold subset, rank_sequent, every_nth in *.
    intros n1 n2 H a H0 n n_less.
    eapply rank_formula_ge.
      eexact H.
    apply H0.
  Qed.

  Lemma rank_sequent_list_reorder :
    forall(s1 s2 : sequent)(n : nat), list_reorder s1 s2 ->
      ((rank_sequent n s1) <-> (rank_sequent n s2)).
  Proof.
    intros s1 s2 n H.
    apply every_nth_list_reorder.
    trivial.
  Qed.

  Lemma rank_sequent_empty : forall(n : nat), rank_sequent n [].
  Proof.
    clear. 
    intros n.
    apply every_nth_empty.
  Qed.

  Lemma rank_sequent_cons : forall(n : nat)(f : lambda_formula)(s : sequent),
    rank_formula n f -> rank_sequent n s -> rank_sequent n (f :: s).
  Proof.
    clear. 
    intros n f s H H0.
    apply every_nth_cons.
      trivial.
    trivial.
  Qed.

  Lemma rank_sequent_head : forall(n : nat)(f : lambda_formula)(s : sequent),
    rank_sequent n (f :: s) -> rank_formula n f.
  Proof.
    clear. 
    intros n f s H.
    eapply every_nth_head.
    eexact H.
  Qed.

  Lemma rank_sequent_tail : forall(n : nat)(f : lambda_formula)(s : sequent),
    rank_sequent n (f :: s) -> rank_sequent n s.
  Proof.
    clear. 
    intros n f s H.
    eapply every_nth_tail.
    eexact H.
  Qed.

  Lemma rank_sequent_append :
    forall(s1 s2 : sequent)(n : nat),
      rank_sequent n s1 -> rank_sequent n s2 -> rank_sequent n (s1 ++ s2).
  Proof.
    intros s1 s2 n.
    apply every_nth_append.
  Qed.

  Lemma rank_sequent_append_left :
    forall(s1 s2 : sequent)(n : nat),
      rank_sequent n (s1 ++ s2) -> rank_sequent n s1.
  Proof.
    intros s1 s2 n.
    apply every_nth_append_left.
  Qed.

  Lemma rank_sequent_append_right :
    forall(s1 s2 : sequent)(n : nat),
      rank_sequent n (s1 ++ s2) -> rank_sequent n s2.
  Proof.
    intros s1 s2 n.
    apply every_nth_append_right.
  Qed.

  Lemma rank_sequent_different_head : 
    forall(n : nat)(f : lambda_formula)(s1 s2 : sequent),
      rank_sequent n (f :: s2) ->
      (rank_formula n f -> rank_sequent n s1) ->
        rank_sequent n (s1 ++ s2).
  Proof.
    clear. 
    intros n f s1 s2 H H0.
    apply rank_sequent_append.
      apply H0.
      eapply rank_sequent_head.
      eexact H.
    eapply rank_sequent_tail.
    eexact H.
  Qed.

  Lemma rank_formula_or_formula_iter : 
    forall(s : sequent)(f : lambda_formula)(n : nat),
      rank_sequent n s ->
      rank_formula n f ->
        rank_formula n (or_formula_of_sequent_iter f s).
  Proof.
    induction s.
      intros f n H H0.
      trivial.
    intros f n H H0.
    simpl.
    apply IHs.
      eapply rank_sequent_tail.
      eexact H.
    apply rank_formula_lambda_or.
      trivial.
    eapply rank_sequent_head.
    eexact H.
  Qed.

  Lemma rank_formula_or_formula_iter_rev : 
    forall(s : sequent)(f : lambda_formula)(n : nat),
      rank_formula n (or_formula_of_sequent_iter f s) ->
        rank_formula n f /\ rank_sequent n s.
  Proof.
    induction s.
      intros f n H.
      split.
        simpl in *.
        trivial.
      apply rank_sequent_empty.
    intros f n H.
    simpl in H.
    apply IHs in H.
    destruct H.
    apply rank_formula_lambda_or_rev in H.
    destruct H.
    split.
      trivial.
    apply rank_sequent_cons.
      trivial.
    trivial.
  Qed.

  Lemma rank_formula_or_formula_of_sequent : 
    forall(s : sequent)(n : nat)(nonempty_v : V),
      1 <= n ->                 (* needed for s = [] *)
      rank_sequent n s ->
        rank_formula n (or_formula_of_sequent s nonempty_v).
  Proof.
    destruct s.
      intros n nonempty_v H H0.
      simpl.
      apply rank_formula_false.
      trivial.
    intros n nonempty_v H H0.
    simpl.
    apply rank_formula_or_formula_iter.
      eapply rank_sequent_tail.
      eexact H0.
    eapply rank_sequent_head.
    eexact H0.
  Qed.

  (** The unusual argument order simplifies the proof of 
      state_seq_step_semantics_correct 
   *)
  Lemma rank_formula_succ_or_formula_of_sequent : 
    forall(n : nat)(nonempty_v : V)(s : sequent),
      rank_sequent (S n) s ->
        rank_formula (S n) (or_formula_of_sequent s nonempty_v).
  Proof.
    intros s n nonempty_v H.
    apply rank_formula_or_formula_of_sequent.
      apply le_n_S.
      apply le_0_n.
    trivial.
  Qed.

  Lemma rank_formula_or_formula_of_ne_sequent : 
    forall(s : sequent)(n : nat)(nonempty_s : s <> []),
      rank_sequent n s ->
        rank_formula n (or_formula_of_ne_sequent s nonempty_s).
  Proof.
    intros s n nonempty_s H.
    destruct s.
      exfalso.
      auto.
    apply rank_formula_or_formula_iter.
      apply rank_sequent_tail in H.
      trivial.
    apply rank_sequent_head in H.
    trivial.
  Qed.

  Lemma rank_prop_sequent : forall(s : sequent),
    prop_sequent s -> rank_sequent 1 s.
  Proof.
    intros s H i i_less.
    specialize (H i i_less).
    unfold rank_formula in *.
    destruct (nth s i i_less).
          apply rank_formula_prop_form.
          trivial.
        destruct l.
              rewrite modal_rank_char.
              apply rank_formula_prop_form.
              trivial.
            contradiction.
          contradiction.
        contradiction.
      contradiction.
    contradiction.
  Qed.

  Lemma rank_sequent_minimal_sequent_rank : forall(s : sequent),
    rank_sequent (minimal_sequent_rank s) s.
  Proof.
    clear. 
    intros s.
    unfold rank_sequent, minimal_sequent_rank, rank_formula in *.
    intros n n_less.
    assert (n < length (map modal_rank s)).
      rewrite map_length.
      trivial.
    erewrite nth_tcc_irr.
    erewrite <- nth_map. 
    instantiate (1 := H).
    apply nat_list_max_lub.
  Qed.

  Lemma rank_sequent_succ_minimal_nonempty_sequent_rank : 
    forall(s : sequent),
      s <> [] -> rank_sequent (S (pred (minimal_sequent_rank s))) s.
  Proof.
    intros s H.
    assert (H0 := minimal_sequent_rank_gt_0 _ H).
    assert (S (pred (minimal_sequent_rank s)) = minimal_sequent_rank s).
      omega.
    rewrite H1.
    apply rank_sequent_minimal_sequent_rank.
  Qed.

  Lemma rank_sequent_le_minimal : forall(s : sequent)(n : nat),
    minimal_sequent_rank s <= n ->
      rank_sequent n s.
  Proof.
    clear. 
    intros s n H.
    unfold minimal_sequent_rank, rank_sequent in *.
    unfold rank_formula in *.
    rewrite <- (every_nth_map _ _ (fun(r : nat) => r <= n)).
    apply nat_list_max_le_inv.
    trivial.
  Qed.

  Lemma rank_sequent_ge_minimal : forall(s : sequent)(n : nat),
    rank_sequent n s -> minimal_sequent_rank s <= n.
  Proof.
    intros s n H.
    unfold minimal_sequent_rank, rank_sequent, rank_formula in *.
    rewrite <- (every_nth_map _ _ (fun(r : nat) => r <= n)) in H.
    apply nat_list_max_le.
    trivial.
  Qed.

  Lemma rank_formula_modal_ge_2 :
    forall(op : operator L)
          (args : counted_list lambda_formula (arity L op))
          (n : nat),
      rank_formula n (lf_modal op args) ->
        1 < n.
  Proof.
    intros op args n H.
    unfold rank_formula in *.
    rewrite modal_rank_char in *.
    unfold nat_list_max in *.
    fold nat_list_max in *.
    assert (H0 := Max.le_max_l 1 
              (nat_list_max (map modal_rank (list_of_counted_list args)))).
    omega.
  Qed.

  Lemma rank_formula_modal_1_TCC :
    forall(A : Type)(op : operator L)
          (args : counted_list lambda_formula (arity L op)),
      rank_formula 1 (lf_modal op args) -> A.
  Proof.
    intros A op args H.
    assert (H0 := rank_formula_modal_ge_2 _ _ _ H).
    omega.
  Qed.

  Lemma rank_formula_modal_args_TCC :
    forall(op : operator L)
          (args : counted_list lambda_formula (arity L op))
          (n : nat),
      rank_formula n (lf_modal op args) ->
        every_nth (rank_formula (pred n)) (list_of_counted_list args).
  Proof.
    intros op args n H.
    destruct n.
      exfalso.
      assert (H0 := rank_formula_modal_ge_2 _ _ _ H).
      omega.
    unfold rank_formula in *.
    rewrite modal_rank_char in H.
    assert (H0 := le_S_n _ _ H).
    clear H.
    unfold nat_list_max in H0.
    fold nat_list_max in H0.
    rewrite <- every_nth_map with (P := fun i => i <= pred (S n)).
    apply nat_list_max_le_inv.
    eapply Max.max_lub_r.
    eexact H0.
  Qed.


  (***************************************************************************)
  (** *** Ranked sequent sets *)
  (***************************************************************************)

  Definition rank_sequent_set(n : nat)(ss : set sequent) : Prop :=
    forall(s : sequent), ss s -> rank_sequent n s.

  Lemma rank_sequent_set_empty : forall(n : nat),
    rank_sequent_set n (empty_sequent_set).
  Proof.
    intros n s H.
    contradiction.
  Qed.

  Lemma rank_sequent_set_mono : forall(n1 n2 : nat)(ss : set sequent),
    n1 <= n2 ->
    rank_sequent_set n1 ss ->
      rank_sequent_set n2 ss.
    intros n1 n2 ss H H0 s H1.
    eapply rank_sequent_mono.
      eexact H.
    apply H0.
    trivial.
  Qed.

  Lemma rank_sequent_set_sequent_list_set : 
    forall(n : nat)(sl : list sequent),
      every_nth (rank_sequent n) sl -> 
        rank_sequent_set n (reordered_sequent_list_set sl). 
  Proof.
    unfold rank_sequent_set, reordered_sequent_list_set in *.
    intros n sl H s H0.
    decompose [ex and] H0; clear H0.
    rename x into s'.
    eapply rank_sequent_list_reorder.
      eexact H2.
    eapply every_nth_In in H.
      eexact H.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Ranked rules *)
  (***************************************************************************)

  (** This is used for S_n, page 10 *)
  Definition rule_has_rank(n : nat)(r : sequent_rule) : Prop :=
    every_nth (rank_sequent n) (assumptions r) /\
    rank_sequent n (conclusion r).

  Definition rank_rules(n : nat)(rules : set sequent_rule) : set sequent_rule :=
    fun(r : sequent_rule) => 
      rules r /\ rule_has_rank n r.

  Lemma subset_rank_rules :
    forall(n : nat)(rules : set sequent_rule),
      subset (rank_rules n rules) rules.
  Proof.
    unfold subset, rank_rules in *.
    intros n rules a H.
    tauto.
  Qed.

  Lemma rank_rules_mono :
    forall(n : nat)(rules1 rules2 : set sequent_rule),
      subset rules1 rules2 -> 
        subset (rank_rules n rules1) (rank_rules n rules2).
  Proof.
    unfold subset in *.
    intros n rules1 rules2 H a H0.
    unfold rank_rules in *.
    decompose [and] H0; clear H0.
    split.
      apply H.
      trivial.
    tauto.
  Qed.

  Lemma rank_rules_set_eq : 
    forall(n : nat)(rules1 rules2 : set sequent_rule),
      set_equal rules1 rules2 ->
        set_equal (rank_rules n rules1) (rank_rules n rules2).
  Proof.
    intros n rules1 rules2 H r.
    apply set_equal_subset_char in H.
    destruct H.
    split.
      apply rank_rules_mono.
      trivial.
    apply rank_rules_mono.
    trivial.
  Qed.


  Lemma rank_rules_subset_rank :
    forall(n1 n2 : nat)(rules : set sequent_rule),
      n1 <= n2 ->
        subset (rank_rules n1 rules) (rank_rules n2 rules).
  Proof.
    unfold subset in *.
    intros n1 n2 rules H a H0.
    unfold rank_rules in *.
    decompose [and] H0; clear H0.
    repeat split.
        trivial.
      eapply every_nth_mono.
        apply rank_sequent_mono.
        eexact H.
      apply H2.
    eapply rank_sequent_mono.
      eexact H.
    apply H2.
  Qed.

  Lemma multiset_rank_rules : forall(n : nat)(rules : set sequent_rule),
    rule_multiset rules -> rule_multiset (rank_rules n rules).
  Proof.
    unfold rule_multiset, rank_rules, rule_has_rank in *.
    intros n rules H or s H0 H1.
    decompose [and] H0; clear H0.
    specialize (H or s H2 H1).
    decompose [ex and or dep_and] H; clear H.
    unfold reordered_rule in *.
    decompose [and] H3; clear H3.
    exists x.
    repeat split; auto.
      intros i i_less.
      assert (i < length (assumptions or)).
        rewrite H7.
        trivial.
      rewrite <- rank_sequent_list_reorder 
                 with (s1 := nth (assumptions or) i H0).
        apply H4.
      apply H8.
    subst s.
    rewrite <- rank_sequent_list_reorder.
      eexact H5.
    trivial.
  Qed.

  Lemma provable_rank_rules_hyp_has_rank_n :
    forall(rules : set sequent_rule)(hyp : set sequent)
          (n : nat)(s : sequent),
      rank_sequent_set n hyp ->
      provable (rank_rules n rules) hyp s ->
        rank_sequent n s.
  Proof.
    intros rules hyp n s H H0.
    destruct H0.
    clear H0.
    destruct x.
      apply H.
      trivial.
    clear d.
    destruct r0.
    apply H1.
  Qed.

  Lemma provable_rank_rules_has_rank_n :
    forall(rules : set sequent_rule)(n : nat)(s : sequent),
      provable (rank_rules n rules) empty_sequent_set s ->
        rank_sequent n s.
  Proof.
    intros rules n s H.
    eapply provable_rank_rules_hyp_has_rank_n.
      apply rank_sequent_set_empty.
    eexact H.
  Qed.


  Lemma rank_rules_distribute_union :
    forall(n : nat)(rules1 rules2 : set sequent_rule),
      set_equal (rank_rules n (union rules1 rules2))
                (union (rank_rules n rules1) (rank_rules n rules2)).
  Proof.
    intros n rules1 rules2.
    split.
      intros r.
      unfold rank_rules, union in *.
      tauto.
    intros r.
    unfold rank_rules, union in *.
    tauto.
  Qed.


  (***************************************************************************)
  (** *** Minimal rank of a rule *)
  (***************************************************************************)

  Definition minimal_rule_rank(r : sequent_rule) : nat :=
    nat_list_max ((minimal_sequent_rank (conclusion r)) :: 
                  (map minimal_sequent_rank (assumptions r))).

  Lemma minimal_rule_rank_gt_0 : forall(r : sequent_rule),
    conclusion r <> [] ->
      0 < minimal_rule_rank r. 
  Proof.
    clear. 
    intros r H.
    unfold minimal_rule_rank in *.
    simpl.
    assert (H0 := minimal_sequent_rank_gt_0 _ H).
    assert (H1 := Max.le_max_l (minimal_sequent_rank (conclusion r))
                  (nat_list_max (map minimal_sequent_rank (assumptions r)))).
    omega.
  Qed.

  Lemma minimal_rule_rank_assumptions : forall(r : sequent_rule),
    every_nth (rank_sequent (minimal_rule_rank r)) (assumptions r).
  Proof.
    clear. 
    intros r n n_less.
    unfold minimal_rule_rank in *.
    simpl.
    eapply rank_sequent_mono.
      apply Max.le_max_r.
    apply rank_sequent_le_minimal.
    assert (n < length (map minimal_sequent_rank (assumptions r))).
      rewrite map_length.
      trivial.
    erewrite nth_tcc_irr.
    erewrite <- nth_map.
    instantiate (1 := H).
    apply nat_list_max_lub.
  Qed.

  Lemma minimal_rule_rank_conclusion : forall(r : sequent_rule),
      rank_sequent (minimal_rule_rank r) (conclusion r).
  Proof.
    clear. 
    intros r.
    unfold minimal_rule_rank in *.
    simpl.
    eapply rank_sequent_mono.
      apply Max.le_max_l.
    apply rank_sequent_minimal_sequent_rank.
  Qed.

  Lemma rank_rules_minimal_rule_rank : 
    forall(rules : set (sequent_rule))(r : sequent_rule)(n : nat),
      rules r ->
      minimal_rule_rank r <= n ->
        rank_rules n rules r.
  Proof.
    intros rules r n H H0.
    eapply rank_rules_subset_rank.
      eexact H0.
    repeat split.
        trivial.
      apply minimal_rule_rank_assumptions.
    apply minimal_rule_rank_conclusion.
  Qed.

  Lemma rank_rules_ge_minimal : forall(r : sequent_rule)(n : nat),
    rule_has_rank n r -> minimal_rule_rank r <= n.
  Proof.
    intros r n H.
    unfold rule_has_rank, minimal_rule_rank in *.
    destruct H.
    apply nat_list_max_le.
    apply every_nth_cons.
      apply rank_sequent_ge_minimal.
      trivial.
    apply every_nth_map.
    intros i i_less.
    apply rank_sequent_ge_minimal.
    apply H.
  Qed.


  (***************************************************************************)
  (** ***  Minimal rank of proofs  *)
  (***************************************************************************)

  Definition minimal_proof_rank{rules : sequent_rule -> Prop}
                               {hypotheses : sequent -> Prop}
                               {s : sequent}
                               (p : proof rules hypotheses s) : nat :=
    proof_rect rules hypotheses (fun _ => nat)
      (fun(s : sequent) _ => minimal_sequent_rank s)
      (fun(r : sequent_rule) _ 
          (l : dep_list sequent (fun _ => nat) (assumptions r)) =>
          max (minimal_rule_rank r)
              (nat_list_max (list_of_dep_list (assumptions r) l)))
      s p.

  Lemma minimal_proof_rank_char :
    forall(rules : set sequent_rule)(hypotheses : set sequent)
          (s : sequent)(p : proof rules hypotheses s),
      minimal_proof_rank p = match p with
        | assume g _ => minimal_sequent_rank g
        | rule r _ subp =>
           max (minimal_rule_rank r)
               (nat_list_max 
                   (dep_map_dep_const (@minimal_proof_rank rules hypotheses)
                                      (assumptions r) subp))
      end.
  Proof.
    clear. 
    intros rules hypotheses s p.
    destruct p.
      trivial.
    unfold minimal_proof_rank in *.
    rewrite proof_rect_rule.
    f_equal.
    f_equal.
    apply list_of_dep_list_dep_map_dep_dep.
  Qed.


  (***************************************************************************)
  (** ***  Extract the list of propositional variables  *)
  (***************************************************************************)

  Definition prop_var_formula : lambda_formula -> list V :=
    lambda_formula_rec (list V)
      (fun(v : V) => [v])
      id
      (@app V)
      (fun(op : operator L)(pl : counted_list (list V) (arity L op)) =>
         flatten (list_of_counted_list pl)).

  (* Have a separate definition for the arguments of modal operators
   * to improve readability in proofs.
   *)
  Definition prop_var_modal_args(n : nat)
                           (args : counted_list lambda_formula n) : list V :=
    flatten ((map prop_var_formula) (list_of_counted_list args)).

  Lemma prop_var_formula_char : forall(f : lambda_formula),
    prop_var_formula f = match f with
      | lf_prop v => [v]
      | lf_neg f => prop_var_formula f
      | lf_and f1 f2 => (prop_var_formula f1) ++ (prop_var_formula f2)
      | lf_modal op args => prop_var_modal_args (arity L op) args
    end.
  Proof.
    intros f.
    unfold prop_var_formula at 1.
    rewrite lambda_formula_rec_char.
    destruct f.
          trivial.
        trivial.
      trivial.
    rewrite list_of_counted_list_map.
    trivial.
  Qed.

  Lemma prop_var_modal_args_cons :
    forall(n : nat)(form : lambda_formula)
          (args : counted_list lambda_formula n),
      prop_var_modal_args (S n) (counted_cons form args) =
        (prop_var_formula form) ++ (prop_var_modal_args n args).
  Proof.
    intros n form args.
    trivial.
  Qed.

  Definition prop_var_sequent(s : sequent) : list V :=
    flatten (map prop_var_formula s).

  Lemma prop_var_sequent_cons :
    forall(f : lambda_formula)(s : sequent),
      prop_var_sequent (f :: s) = (prop_var_formula f) ++ prop_var_sequent s.
  Proof.
    intros f s.
    trivial.
  Qed.

  Lemma prop_var_sequent_append : forall(s1 s2 : sequent),
    prop_var_sequent (s1 ++ s2) = 
      (prop_var_sequent s1) ++ (prop_var_sequent s2).
  Proof.
    induction s1.
      intros s2.
      simpl.
      trivial.
    intros s2.
    simpl.
    rewrite prop_var_sequent_cons.
    rewrite prop_var_sequent_cons.
    rewrite IHs1.
    apply app_assoc.
  Qed.

  Lemma prop_var_sequent_list_reorder : forall(s1 s2 : sequent),
    list_reorder s1 s2 ->
      list_reorder (prop_var_sequent s1) (prop_var_sequent s2).
  Proof.
    intros s1 s2 H.
    unfold prop_var_sequent in *.
    induction H.
      simpl.
      apply list_reorder_nil.
    simpl.
    rewrite map_app.
    simpl.
    rewrite flatten_append.
    simpl.
    apply (@list_reorder_insert_list V []).
      simpl.
      rewrite <- flatten_append.
      rewrite <- map_app.
      rewrite firstn_skipn.
      trivial.
    apply list_reorder_refl.
  Qed.

  Lemma In_prop_var_sequent : 
    forall(v : V)(s : sequent),
      In v (prop_var_sequent s) ->
        exists(f : lambda_formula), 
          In v (prop_var_formula f) /\ In f s.
  Proof.
    intros v s H.
    apply In_flatten in H.
    decompose [ex and or dep_and] H; clear H.
    rename x into n, a into n_less, b into H.
    rewrite nth_map in H.
    eexists.
    split.
      eexact H.
    eapply In_nth_rev.
    trivial.
  Qed.

  Lemma incl_prop_var_formula_sequent :
    forall(f : lambda_formula)(s : sequent),
      In f s ->
        incl (prop_var_formula f) (prop_var_sequent s).
  Proof.
    induction s.
      intros H.
      contradiction.
    intros H.
    rewrite prop_var_sequent_cons.
    destruct H.
      clear IHs.
      subst a.
      apply incl_appl.
      apply incl_refl.
    apply incl_appr.
    auto.
  Qed.

  Lemma prop_var_sequent_cutout_nth : forall(n : nat)(s : sequent),
    every_nth (fun f => length (prop_var_formula f) = 1) s ->
      prop_var_sequent (cutout_nth s n) = cutout_nth (prop_var_sequent s) n.
  Proof.
    induction n.
      intros s H.
      destruct s.
        trivial.
      rename l into f.
      rewrite cutout_nth_cons_0.
      rewrite prop_var_sequent_cons.
      apply every_nth_head in H.
      destruct (prop_var_formula f).
        discriminate.
      destruct l.
        simpl.
        rewrite cutout_nth_cons_0.
        trivial.
      discriminate.
    intros s H.
    destruct s.
      trivial.
    rename l into f.
    rewrite cutout_nth_cons_succ.
    rewrite prop_var_sequent_cons.
    rewrite prop_var_sequent_cons.
    rewrite IHn; clear IHn.
      apply every_nth_head in H.
      destruct (prop_var_formula f).
        discriminate.
      destruct l.
        simpl.
        rewrite cutout_nth_cons_succ.
        trivial.
      discriminate.
    apply every_nth_tail in H.
    trivial.
  Qed.

End Formulas.

Implicit Arguments lf_prop [[V] [L]].
Implicit Arguments lf_neg [[V] [L]].
Implicit Arguments lf_and [[V] [L]].
Implicit Arguments lf_modal [[V] [L]].
Implicit Arguments lambda_formula_rect [V L].
Implicit Arguments lambda_formula_rec [V L A].
Implicit Arguments is_prop [[V] [L]].
Implicit Arguments is_neg [[V] [L]].
Implicit Arguments is_and [[V] [L]].
Implicit Arguments is_modal [[V] [L]].
Implicit Arguments get_prop_var [V L].
Implicit Arguments get_neg_form [V L].
Implicit Arguments get_and_forms [V L].
Implicit Arguments get_modal_args [V L].
Implicit Arguments lambda_or [V L].
Implicit Arguments lambda_false [V L].
Implicit Arguments lf_modal_inversion_op [V L].
Implicit Arguments assumptions [V L].
Implicit Arguments conclusion [V L].
Implicit Arguments prop_form [[V] [L]].
Implicit Arguments prop_form_acc [V L].
Implicit Arguments neg_form [[V] [L]].
Implicit Arguments neg_form_maybe [V L].
Implicit Arguments prop_sequent [[V] [L]].
Implicit Arguments prop_sequent_head [V L].
Implicit Arguments prop_sequent_tail [V L].
Implicit Arguments neg_and_over [V L].
Implicit Arguments formula_measure [V L].
Implicit Arguments formula_measure_char [V L].
Implicit Arguments sequent_measure [[V] [L]].
Implicit Arguments proof [V L].
Implicit Arguments assume [V L].
Implicit Arguments rule [V L].
Implicit Arguments proof_depth [V L rules hypotheses s].
Implicit Arguments proof_depth_rule [V L].
Implicit Arguments provable [V L].
Implicit Arguments provable_at_depth [V L].
Implicit Arguments proof_depth_sequent_ind [V L].
Implicit Arguments or_formula_of_sequent_iter [V L].
Implicit Arguments or_formula_of_sequent [V L].
Implicit Arguments or_formula_of_ne_sequent [V L].
Implicit Arguments sequent_multiset [V L].
Implicit Arguments reordered_rule [V L].
Implicit Arguments rule_multiset [V L].
Implicit Arguments reordered_sequent_list_set [V L].
Implicit Arguments modal_rank [V L].
Implicit Arguments modal_rank_char [V L].
Implicit Arguments minimal_sequent_rank [V L].
(* Implicit Arguments minimal_sequent_rank_reorder [V L]. *)
Implicit Arguments modal_rank_ge_1 [V L].
Implicit Arguments rank_formula [V L].
Implicit Arguments rank_formula_ge [V L].
Implicit Arguments rank_formula_lf_neg_TCC [V L].
Implicit Arguments rank_formula_modal_ge_2 [V L].
Implicit Arguments rank_formula_modal_1_TCC [V L].
Implicit Arguments rank_formula_modal_args_TCC [V L].
Implicit Arguments rank_formula_lf_neg [V L].
Implicit Arguments rank_formula_and_left [V L].
Implicit Arguments rank_formula_and_right [V L].
Implicit Arguments rank_formula_lambda_or [V L].
Implicit Arguments rank_sequent [V L].
Implicit Arguments rank_sequent_head [V L].
Implicit Arguments rank_sequent_cons [V L].
Implicit Arguments rank_sequent_list_reorder [V L].
Implicit Arguments rank_sequent_head [V L].
Implicit Arguments rank_sequent_tail [V L].
Implicit Arguments rank_sequent_append [V L].
Implicit Arguments rank_sequent_append_left [V L].
Implicit Arguments rank_sequent_append_right [V L].
Implicit Arguments rank_formula_or_formula_iter_rev [V L].
Implicit Arguments rank_formula_succ_or_formula_of_sequent [V L].
Implicit Arguments rank_sequent_succ_minimal_nonempty_sequent_rank [V L].
Implicit Arguments rank_sequent_set [V L].
Implicit Arguments rule_has_rank [V L].
Implicit Arguments rank_rules [V L].
Implicit Arguments provable_rank_rules_hyp_has_rank_n [V L].
Implicit Arguments rank_rules_distribute_union [V L].
Implicit Arguments minimal_rule_rank [V L].
Implicit Arguments rank_rules_minimal_rule_rank [V L].
Implicit Arguments minimal_proof_rank [V L rules hypotheses s].
Implicit Arguments prop_var_formula [V L].
Implicit Arguments prop_var_formula_char [V L].
Implicit Arguments prop_var_modal_args [V L].
Implicit Arguments prop_var_sequent [[V] [L]].
