(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: generic_cut.v,v 1.9 2013/04/10 12:06:16 tews Exp $
 *)


(** * Results on propositional logic *)

(** ** High-level misc results

      The name of this module is a bad choice. This module contains
      some utility results that need to go into a separate file
      because of module dependencies. There is:
      - Derive general cut elimination from cut elimination at head
        position. This is needed in the propositional cut-elimination
        proof.  
      - Prove cut elimination in the weakening contest of a one-step
        rule. This is needed in module 
        #<A HREF="mixed_cut.html"><spanclass="inlinecode">mixed_cut</span></A>#
        and module
        #<A HREF="osr_cut.html"><spanclass="inlinecode">osr_cut</span></A>#. 
      - Construct a proof for finite disjunction.
      - Sequent axiom
*)

Require Export weakening.

Section Generic_cut.

  Variable V : Type.
  Variable L : modal_operators.


  (***************************************************************************)
  (** *** Cut elimination relative to cut elimination at head position *)
  (***************************************************************************)

  Lemma cut_admissibile_from_head_elim : 
    forall(rules : set (sequent_rule V L))(n : nat),
      rule_multiset rules ->
      (forall(f : lambda_formula V L)(q r : sequent V L),
             provable (rank_rules n rules) (empty_sequent_set V L) 
                      (f :: q) ->
             provable (rank_rules n rules) (empty_sequent_set V L) 
                      ((lf_neg f) :: r) ->
               provable (rank_rules n rules) (empty_sequent_set V L) (q ++ r))
      ->
        admissible_rule_set (rank_rules n rules) (empty_sequent_set V L)
                            (bounded_cut_rules V L n).
  Proof.
    unfold admissible_rule_set, admissible in *.
    intros rules n H H0 r H1 H2.
    unfold bounded_cut_rules, rank_rules, is_cut_rule in H1.
    decompose [ex and] H1; clear H1.
    rename x into gl, x0 into gr, x1 into dl, x2 into dr, x3 into f.
    rewrite H3 in *; clear H3.
    assert (rule_multiset (rank_rules n rules)).
      apply multiset_rank_rules.
      trivial.
    eapply multiset_provability.
          trivial.
        apply sequent_multiset_empty.
      apply list_reorder_symm.
      eexact H6.
    rewrite app_assoc.
    apply H0 with (f := f).
      eapply multiset_provability.
            trivial.
          apply sequent_multiset_empty.
        apply list_reorder_symm.
        apply list_reorder_cons_parts.
        apply list_reorder_refl.
      apply every_nth_head in H2.
      trivial.
    eapply multiset_provability.
          trivial.
        apply sequent_multiset_empty.
      apply list_reorder_symm.
      apply list_reorder_cons_parts.
      apply list_reorder_refl.
    apply every_nth_tail in H2.
    apply every_nth_head in H2.
    trivial.
  Qed.

  Lemma provable_GRC_n_GR_n_from_head_elim : 
    forall(rules : set (sequent_rule V L))(n : nat)(s : sequent V L),
      one_step_rule_set rules ->
      (forall(f : lambda_formula V L)(q r : sequent V L),
         provable (GR_n_set rules n) (empty_sequent_set V L)
                  (f :: q) ->
         provable (GR_n_set rules n) (empty_sequent_set V L)
                  ((lf_neg f) :: r) ->
           provable (GR_n_set rules n) (empty_sequent_set V L) (q ++ r)) ->
      provable (GRC_n_set rules n) (empty_sequent_set V L) s ->
        provable (GR_n_set rules n) (empty_sequent_set V L) s.
  Proof.
    intros rules n s H H0 H1.
    rewrite admissible_prop with (rs := bounded_cut_rules V L n).
      eapply proof_set_equal_rules.
        apply GRC_n_as_GR_C_union.
      trivial.
    eapply cut_admissibile_from_head_elim; trivial.
    apply GR_multiset.
  Qed.


  (***************************************************************************)
  (** *** Cut elimination in the context of one-step rules *)
  (***************************************************************************)

  (** The following is case a) 1 and b) 1 in 5.6.3 cut elimination:
      The cut formula is in the weakening context of a one-step rule, 
      so one can easily add the other rule into the context.
   *)
  Lemma cut_elimination_osr_context :
    forall(rules : set (sequent_rule V L))(n : nat)(ssn_pos : 0 < 2 + n)
          (f : lambda_formula V L)(q r : sequent V L)
          (rb : sequent_rule V L)(sigma : lambda_subst V L)
          (delta delta_tl : sequent V L),
      one_step_rule_set rules ->
      rules rb ->
      rank_subst (S n) sigma ->
      rank_sequent (2 + n) delta ->
      list_reorder (f :: q)
                   ((subst_sequent sigma (conclusion rb)) ++ delta) ->
      list_reorder (f :: delta_tl) delta ->
      every_nth
        (provable (GR_n_set rules (S n)) (empty_sequent_set V L))
        (map (subst_sequent sigma) (assumptions rb)) ->
      rank_sequent (2 + n) r ->
        provable_subst_n_conclusions rules (2 + n) ssn_pos (q ++ r).
  Proof.
    clear. 
    intros rules n ssn_pos f q r rb sigma delta delta_tl H H0 H1 H2 H3 H4 H5 H6.
    exists {| assumptions := map (subst_sequent sigma) (assumptions rb);
              conclusion := q ++ r |}.
    split; auto.
    exists rb, sigma, (delta_tl ++ r).
    split; trivial.
    split; trivial.
    split.
      apply rank_sequent_append.
        eapply rank_sequent_tail.
        eapply rank_sequent_list_reorder.
          eexact H4.
        trivial.
      trivial.
    split; trivial.
    simpl.
    rewrite app_assoc.
    apply list_reorder_append_right.
    eapply list_reorder_tail.
    eapply list_reorder_trans.
      eexact H3.
    apply list_reorder_append_left.
    apply list_reorder_symm.
    trivial.
  Qed.


  (***************************************************************************)
  (** *** Proof finite disjunction *)
  (***************************************************************************)

  Lemma provable_or_formula_of_ne_sequent :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (n : nat)(s : sequent V L)(nonempty_s : s <> []),
      rank_sequent_set n hyp ->
      subset (G_n_set V L n) (rank_rules n rules) ->
      provable (rank_rules n rules) hyp s ->
        provable (rank_rules n rules) hyp 
                 [or_formula_of_ne_sequent s nonempty_s].
  Proof.
    intros rules hyp n s nonempty_s H H0 H1.
    destruct s as [|f].
      exfalso.
      auto.
    simpl.
    clear nonempty_s.
    revert s f H1.
    induction s.
      trivial.
    rename a into f'.
    intros f H1.
    simpl.
    apply IHs.
    unfold lambda_or in *.
    assert (rank_sequent n (lf_neg (lf_and (lf_neg f) (lf_neg f')) :: s)).
      apply provable_rank_rules_hyp_has_rank_n in H1; trivial.
      apply rank_sequent_cons.
        assert (H2 := rank_sequent_tail _ _ _ H1).
        apply rank_sequent_head in H1.
        apply rank_sequent_head in H2.
        apply rank_formula_lf_neg.
        apply rank_formula_lf_and; trivial.
      apply rank_sequent_tail in H1.
      apply rank_sequent_tail in H1.
      trivial.
    apply provable_with_neg_and with (sl := []); trivial.
    apply const_rank_neg_and_rule_context with (sl := []) in H2.
    apply provable_with_neg_neg; trivial.
    apply const_rank_neg_neg_rule_context in H2.
    apply provable_with_neg_neg with (sl := [f]); trivial.
  Qed.


  (***************************************************************************)
  (** *** Proof ~(\/ G), G *)
  (***************************************************************************)

  Lemma provable_sequent_axiom_ind :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (f : lambda_formula V L)(s1 s2 : sequent V L)(n : nat),
      rule_multiset rules ->
      sequent_multiset hyp ->
      subset (G_n_set V L n) (rank_rules n rules) ->
      rank_formula n f ->
      rank_sequent n (rev s1) ->
      rank_sequent n s2 ->
      incl (f :: s1) s2 ->
      (forall(f : lambda_formula V L)(s : sequent V L),
         rank_formula n f ->
         rank_sequent n s ->
           provable (rank_rules n rules) hyp (f :: lf_neg f :: s))
      ->
        provable (rank_rules n rules) hyp 
                 ((lf_neg (or_formula_of_sequent_iter f (rev s1))) :: s2).
  Proof.
    intros rules hyp f s1 s2 n H H0 H1 H2 H3 H4 H5 H6.
    apply multiset_rank_rules with (n := n) in H.
    induction s1.
      simpl.
      lapply (H5 f); clear H5.
        intros H5.
        apply list_reorder_In_split in H5.
        destruct H5 as [s2l].
        destruct H5 as [s2r].
        eapply multiset_provability; trivial.
          apply list_reorder_cons_head.
          apply list_reorder_symm.
          eexact H5.
        eapply multiset_provability; trivial.
          apply list_reorder_swap_head.
        apply H6; trivial.
        eapply rank_sequent_tail.
        eapply rank_sequent_list_reorder.
          apply list_reorder_symm.
          eexact H5.
        trivial.
      left.
      trivial.
    rename a into g.
    rewrite or_formula_of_sequent_iter_rev.
    simpl in H3.
    assert (H7 := rank_sequent_append_left _ _ _ H3).
    apply rank_sequent_append_right in H3.
    apply rank_sequent_head in H3.
    unfold lambda_or in *.
    assert (rank_sequent n (lf_neg (lf_neg (lf_and (lf_neg 
                (or_formula_of_sequent_iter f (rev s1))) (lf_neg g))) :: s2)).
      apply rank_sequent_cons; trivial.
      apply rank_formula_lf_neg.
      apply rank_formula_lf_neg.
      apply rank_formula_lf_and; trivial.
      apply rank_formula_lf_neg.
      apply rank_formula_or_formula_iter; trivial.
    apply provable_with_neg_neg with (sl := []); trivial.
    apply const_rank_neg_neg_rule_context with (sl := []) in H8.
    apply provable_with_and; trivial.
      apply IHs1; trivial.
      clear - H5.
      intros f' H.
      apply H5.
      destruct H.
        left.
        trivial.
      right.
      right.
      trivial.
    clear IHs1 H8.
    simpl.
    lapply (H5 g); clear H5.
      intros H5.
      apply list_reorder_In_split in H5.
      destruct H5 as [s2l].
      destruct H5 as [s2r].
      eapply multiset_provability; trivial.
        apply list_reorder_cons_head.
        apply list_reorder_symm.
        eexact H5.
      eapply multiset_provability; trivial.
        apply list_reorder_swap_head.
      apply H6; trivial.
      eapply rank_sequent_tail.
      eapply rank_sequent_list_reorder.
        apply list_reorder_symm.
        eexact H5.
      trivial.
    right.
    left.
    trivial.
  Qed.

  Lemma provable_sequent_axiom :
    forall(rules : set (sequent_rule V L))(hyp : set (sequent V L))
          (s1 s2 : sequent V L)(nonempty_s1 : s1 <> [])(n : nat),
      rule_multiset rules ->
      sequent_multiset hyp ->
      subset (G_n_set V L n) (rank_rules n rules) ->
      rank_sequent n s1 ->
      rank_sequent n s2 ->
      incl s1 s2 ->
      (forall(f : lambda_formula V L)(s : sequent V L),
         rank_formula n f ->
         rank_sequent n s ->
           provable (rank_rules n rules) hyp (f :: lf_neg f :: s))
      ->
        provable (rank_rules n rules) hyp 
                 ((lf_neg (or_formula_of_ne_sequent s1 nonempty_s1)) :: s2).
  Proof.
    intros rules hyp s1 s2 nonempty_s1 n H H0 H1 H2 H3 H4 H5.
    destruct s1 as [|f].
      exfalso.
      auto.
    simpl.
    rewrite <- (rev_involutive s1).
    apply provable_sequent_axiom_ind; trivial.
        apply rank_sequent_head in H2.
        trivial.
      rewrite rev_involutive.
      apply rank_sequent_tail in H2.
      trivial.
    apply incl_cons.
      apply H4.
      left.
      trivial.
    apply incl_left_tail in H4.
    eapply incl_tran.
      apply incl_rev.
    trivial.
  Qed.

End Generic_cut.
