(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: image.v,v 1.11 2013/04/10 11:17:15 tews Exp $
 *)

(** ** Direct and inverse image for functions *)

Require Export sets functions.

Section Image.

  Variable A B : Type.

  Definition direct_img(f : A -> B)(P : set A) : set B :=
    fun(b : B) => exists(a : A), f a = b /\ P a.
  
  Definition inv_img(f : A -> B)(P : set B) : set A :=
    fun(a : A) => P (f a).


  (***************************************************************************)
  (** ***  inverse image  *)
  (***************************************************************************)

  Lemma set_equal_inv_img :
    forall(f g : A -> B)(P Q : set B),
      f ≡ g -> set_equal P Q -> set_equal (inv_img f P) (inv_img g Q).
  Proof.
    intros f g P Q H H0 b.
    unfold inv_img in *.
    rewrite H.
    apply H0.
  Qed.

  Lemma set_equal_inv_img_pred : forall(f : A -> B)(P Q : set B),
    set_equal P Q -> set_equal (inv_img f P) (inv_img f Q).
  Proof.
    intros f P Q H.
    apply set_equal_inv_img.
      apply feq_reflexive.
    trivial.
  Qed.

  Lemma set_equal_inv_img_feq : forall(f g : A -> B)(P : set B),
    f ≡ g -> set_equal (inv_img f P) (inv_img g P).
  Proof.
    intros f g P H.
    apply set_equal_inv_img.
      trivial.
    apply set_equal_refl.
  Qed.


  (***************************************************************************)
  (** ***  direct image  *)
  (***************************************************************************)

  Lemma feq_direct_img :
    forall(f g : A -> B)(P Q : set A),
      f ≡ g -> set_equal P Q -> set_equal (direct_img f P) (direct_img g Q).
  Proof.
    intros f g P Q H H0 b.
    unfold direct_img in *.
    apply set_equal_subset_char in H0.
    destruct H0.
    split.
      intros H2.
      decompose [ex and or] H2; clear H2.
      exists x.
      split.
        rewrite <- H.
        trivial.
      auto.
    intros H2.
    decompose [ex and or] H2; clear H2.
    exists x.
    split.
      rewrite H.
      trivial.
    auto.
  Qed.


  (***************************************************************************)
  (** ***  fibredness  *)
  (***************************************************************************)

  Lemma fibred_set_inverse : forall(f : A -> B)(P : set B),
    set_inverse (inv_img f P) = inv_img f (set_inverse P).
  Proof.
    intros f P.
    unfold set_inverse, inv_img in *.
    trivial.
  Qed.

  Lemma fibred_intersection : forall(f : A -> B)(P Q : set B),
    intersection (inv_img f P) (inv_img f Q) =
      inv_img f (intersection P Q).
  Proof.
    intros f P Q.
    unfold inv_img, intersection in *.
    trivial.
  Qed.

End Image.


Implicit Arguments direct_img [A B].
Implicit Arguments inv_img [A B].


  (***************************************************************************)
  (** *** Other results *)
  (***************************************************************************)

Lemma inv_img_id : forall{A : Type}(P : set A),
  inv_img id P = P.
Proof.
  intros A P.
  trivial.
Qed.

Lemma inv_img_compose : 
  forall(A B C : Type)(f : A -> B)(g : B -> C)(P : set C),
    inv_img f (inv_img g P) = inv_img (g ∘ f) P.
Proof.
  intros A B C f g P.
  trivial.
Qed.

(** This following lemma is the core of the "elementary boolean algebra"
    reasoning in Lemma 4.13 on page 23.
    Actually, what is really needed there is the same result with 
    assumption 
    [[
       is_full_set (set_inverse (intersection 
                                  (set_inverse (inv_img pi_1 P))
                                  (set_inverse (inv_img pi_2 Q))))
    ]]
    However, then this lemma is only valid in classical logic, 
    see classic_axiom_neg_disjunction_other_case in classic.v
    I leave the classical reasoning to the uses of this lemma.
 *)
Lemma inv_proj_full :
  forall(A B : Type)(pi_1 : A * B -> A)(pi_2 : A * B -> B)
        (P : set A)(Q : set B)(b : B),
    pi_1 ≡ (@fst A B) ->
    pi_2 ≡ (@snd A B) ->
    is_full_set (union (inv_img pi_1 P) (inv_img pi_2 Q)) ->
    not (Q b) ->
      is_full_set (inv_img pi_1 P).
Proof.
  intros A B pi_1 pi_2 P Q b H H0 H1 H2 ab.
  destruct ab.
  specialize (H1 (a, b)).
  unfold union, inv_img in *.
  rewrite H in *.
  rewrite H0 in *.
  simpl in *.
  destruct H1.
    trivial.
  contradiction.
Qed.

Lemma direct_image_left_adjoint :
  forall(A B : Type)(f : A -> B)(P : set A)(Q : set B),
    subset P (inv_img f Q) <-> subset (direct_img f P) Q.
Proof.
  intros A B f P Q.
  unfold inv_img, direct_img in *.
  split.
    intros H b H0.
    destruct H0.
    destruct H0.
    apply H in H1.
    subst b.
    trivial.
  intros H a H0.
  apply H.
  exists a.
  auto.
Qed.

