(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: list_support.v,v 1.6 2013/04/10 11:17:15 tews Exp $
 *)

(** ** Support of lists

      The support of a list is the list of its elements each occuring
      exactly once. This is needed for sequent support.
*)

Require Export list_set list_multiset.


Section List_support.

  Variable A : Type.

  Variable aeq : eq_type A.


  Fixpoint list_support(l : list A) : list A :=
    match l with 
      | [] => []
      | a :: l => 
        let sl := list_support l 
        in
          if member aeq a sl then sl else a :: sl
    end.

  Lemma list_support_correct_no_dup :
    forall(l : list A), NoDup (list_support l).
  Proof.
    induction l.
      simpl.
      apply NoDup_nil.
    simpl.
    destruct (member aeq a (list_support l)) eqn:H.
      eapply IHl.
    apply NoDup_cons.
      intros H0.
      apply member_In_false with (a_eq := aeq) in H.
      contradiction.
    trivial.
  Qed.

  Lemma list_support_correct_content :
    forall(l : list A), incl l (list_support l).
  Proof.
    induction l.
      intros a H.
      contradiction.
    intros a0 H.
    destruct H.
      subst a0.
      simpl. 
      destruct (member aeq a (list_support l)) eqn:H.
        apply member_In in H; trivial.
      left.
      trivial.
    simpl.
    destruct (member aeq a (list_support l)).
      apply IHl.
      trivial.
    apply in_cons.
    apply IHl.
    trivial.
  Qed.

  Lemma list_support_correct_subset :
    forall(l : list A), multi_subset (list_support l) l.
  Proof.
    unfold multi_subset in *.
    induction l.
      exists [].
      simpl.
      apply list_reorder_nil.
    destruct IHl as [dups].
    simpl.
    destruct (member aeq a (list_support l)).
      exists (a :: dups).
      apply list_reorder_symm.
      apply list_reorder_cons_parts.
      apply list_reorder_symm.
      trivial.
    exists dups.
    simpl.
    apply list_reorder_cons_head.
    trivial.
  Qed.


  Lemma list_support_incl :
    forall(l : list A), incl (list_support l) l.
  Proof.
    intros l a H.
    assert (H0 := list_support_correct_subset l).
    unfold multi_subset in *.
    destruct H0 as [ldup].
    eapply list_reorder_In.
      eexact H0.
    apply in_or_app.
    auto.
  Qed.

  Lemma every_nth_list_support : 
    forall(P : A -> Prop)(l : list A),
      every_nth P l ->
        every_nth P (list_support l).
  Proof.
    intros P l H.
    apply every_nth_In_rev.
    intros a H0.
    eapply every_nth_In.
      eexact H.
    apply list_support_incl.
    trivial.
  Qed.

  Lemma list_support_destruct :
    forall(l1 : list A), exists(l2 : list A),
      list_reorder l1 (l2 ++ list_support l1) /\
      incl l2 (list_support l1).
  Proof.
    intros l1.
    assert (H := list_support_correct_subset l1).
    unfold multi_subset in *.
    destruct H as [l2].
    exists l2.
    split.
      apply list_reorder_symm.
      eapply list_reorder_trans.
        apply list_reorder_append_swap.
      trivial.
    eapply incl_tran.
    eapply incl_lappr.
      apply incl_list_reorder.
      eexact H.
    intros a H0.
    apply list_support_correct_content.
    trivial.
  Qed.

  Lemma list_support_of_no_dup : forall(l : list A),
    NoDup l ->
      list_support l = l.
  Proof.
    induction l.
      trivial.
    intros H.
    specialize (IHl (NoDup_tail _ _ H)).
    apply NoDup_head in H.
    apply (contrapositive (list_support_incl _ _)) in H.
    simpl.
    apply member_In_false with (a_eq := aeq) in H.
    rewrite H.
    rewrite IHl.
    trivial.
  Qed.

  Lemma list_support_head_contract :
    forall(a : A)(l : list A),
      list_support (a :: a :: l) = list_support (a :: l).
  Proof.
    intros a l.
    remember (a :: l) as al.
    simpl.
    destruct (member aeq a (list_support al)) eqn:?.
      trivial.
    exfalso.
    apply member_In_false in Heqb; trivial.
    apply Heqb; clear Heqb.
    subst al.
    simpl.
    destruct (member aeq a (list_support l)) eqn:?.
      apply member_In in Heqb; trivial.
    left.
    trivial.
  Qed.


  Lemma multi_subset_list_support :
    forall(l1 l2 : list A),
      incl l1 l2 ->
        multi_subset (list_support l1) (list_support l2).
  Proof.
    induction l1.
      intros l2 H.
      exists (list_support l2).
      apply list_reorder_refl.
    intros l2 H.
    simpl.
    destruct (member aeq a (list_support l1)) eqn:H0.
      apply IHl1.
      apply incl_left_tail in H.
      trivial.
    lapply (IHl1 l2); clear IHl1.
      intros H1.
      lapply (H a); clear H.
        intros H.
        apply list_support_correct_content in H.
        destruct H1 as [l2wol1].
        apply list_reorder_In with (l2 := list_support l1 ++ l2wol1) in H.
          apply in_app_or in H.
          destruct H.
            apply member_In_false in H0.
            contradiction.
          apply in_split in H.
          destruct H as [l_l2wol1].
          destruct H as [r_l2wol1].
          subst l2wol1.
          exists (l_l2wol1 ++ r_l2wol1).
          simpl in *.
          rewrite app_assoc.
          eapply list_reorder_trans.
            apply list_reorder_move_append.
          rewrite app_assoc_reverse.
          trivial.
        apply list_reorder_symm.
        trivial.
      left.
      trivial.
    apply incl_left_tail in H.
    trivial.
  Qed.

  Lemma list_support_same_set :
    forall(l1 l2 : list A),
      (forall(a : A), In a l1 <-> In a l2) ->
        list_reorder (list_support l1) (list_support l2).
  Proof.
    intros l1 l2 H.
    apply multi_subset_antisymm.
      apply multi_subset_list_support.
      intros a.
      apply H.
    apply multi_subset_list_support.
    intros a.
    apply H.
  Qed.

  Lemma list_support_reorder :
    forall(l1 l2 : list A),
      list_reorder l1 l2 ->
        list_reorder (list_support l1) (list_support l2).
  Proof.
    intros l1 l2 H.
    apply list_support_same_set.
    intros a.
    split.
      intros H0.
      eapply list_reorder_In.
        eexact H.
      trivial.
    intros H0.
    eapply list_reorder_In.
      apply list_reorder_symm.
      eexact H.
    trivial.
  Qed.

  Lemma multi_subset_right_list_support :
    forall(l1 l2 : list A),
      NoDup l1 ->
      incl l1 l2 ->
        multi_subset l1 (list_support l2).
  Proof.
    intros l1 l2 H H0.
    apply list_support_of_no_dup in H.
    rewrite <- H.
    apply multi_subset_list_support.
    trivial.
  Qed.

End List_support.

Implicit Arguments list_support [A].

Lemma list_support_map :
  forall{A B : Type}{aeq : eq_type A}{beq : eq_type B}
        {f : A -> B}(l : list A),
    (forall(a1 a2 : A), In a1 l -> In a2 l -> f a1 = f a2 -> a1 = a2) ->
      list_support beq (map f l) = map f (list_support aeq l).
Proof.
  induction l.
    intros H.
    trivial.
  intros H.
  lapply IHl; clear IHl.
    intros H0.
    simpl.
    destruct (member aeq a (list_support aeq l)) eqn:H1.
      rewrite member_In in H1.
      apply in_map with (f := f) in H1.
      rewrite <- member_In in H1.
      rewrite <- H0 in H1.
      rewrite H1.
      trivial.
    rewrite member_In_false in H1.
    assert (~ In (f a) (map f (list_support aeq l))).
      intros H2.
      apply in_map_reverse in H2.
        contradiction.
      clear - H.
      intros a1 H0 H1.
      apply H.
          right.
          eapply list_support_incl.
          eexact H0.
        left.
        trivial.
      trivial.
    clear H1.
    rewrite <- H0 in H2.
    rewrite <- member_In_false in H2.
    rewrite H2.
    simpl.
    rewrite H0.
    trivial.
  intros a1 a2 H0 H1 H2.
  apply H.
      right.
      trivial.
    right.
    trivial.
  trivial.
Qed.
