(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: one_step_conditions.v,v 1.28 2013/04/10 12:06:16 tews Exp $
 *)


(** ** One-step conditions, 4.1 - 4.3 

      This module defines one-step soundness and completeness (4.1)
      and proves the simplification for cut-free one-step completeness (4.3.)
 *)

Require Export rule_sets semantics renaming list_multiset.

Section One_step_conditions.

  Variable V : Type.
  Variable L : modal_operators.
  Variable T : functor.

  (** Need a decidable equality for limit_subst, which is needed in 4.3 *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** ***  One-step Soundness, Def 4.1, page 16  *)
  (***************************************************************************)

  Definition one_step_sound(nonempty_v : V)(LS : lambda_structure L T)
                           (rules : set (sequent_rule V L))
                           (osr : one_step_rule_set rules) : Prop :=
    forall(X : Type)(r : sequent_rule V L)(rules_r : rules r)
          (coval : X -> set V), 
      (forall(n : nat)(n_less : n < length (assumptions r)),
         prop_seq_val_valid nonempty_v coval
           (nth (assumptions r) n n_less)
           (one_step_rule_propositional_assumption r 
               (osr r rules_r) n n_less))
        ->
          mod_seq_val_valid LS coval (conclusion r) 
            (one_step_rule_nonempty_conclusion r (osr r rules_r))
            (one_step_rule_prop_modal_prop_conclusion r (osr r rules_r)).


  (***************************************************************************)
  (** *** Towards one-step Completeness, Def 4.1  *)
  (***************************************************************************)

  (** Define first the rule set appearing in the hypothesis.
      This is very similar to 
      #<A HREF="weakening.html##rank_weaken_subst_rule"><spanclass="inlinecode">rank_weaken_subst_rule</span></A>#
       in module
      #<A HREF="weakening.html"><spanclass="inlinecode">weakening</span></A>#.
   *)
  Definition simple_mod_weaken_rule(rules : set (sequent_rule V L))
                                                   : set (sequent_rule V L) :=
    fun(r_subst : sequent_rule V L) =>
      exists(r_base : sequent_rule V L)
            (sigma : lambda_subst V L)
            (delta : sequent V L),
        rules r_base /\
        rank_subst 1 sigma /\          (* sigma : V -> Prop(V) *)
        simple_modal_sequent delta /\  (* delta : sequents(Lambda(V)) *)
        assumptions r_subst = map (subst_sequent sigma) (assumptions r_base) /\
        list_reorder (conclusion r_subst) 
                     ((subst_sequent sigma (conclusion r_base)) ++ delta).

  Lemma simple_mod_weaken_rule_assumptions :
    forall(rules : set (sequent_rule V L))(r : sequent_rule V L),
      one_step_rule_set rules ->
      simple_mod_weaken_rule rules r ->
        every_nth propositional_sequent (assumptions r).
  Proof.
    intros rules r H H0.
    unfold simple_mod_weaken_rule in *.
    decompose [ex and or dep_and] H0; clear H0.
    rename x into r_base, x0 into sigma, x1 into delta.
    rewrite H4.
    clear H3 H4 H6.
    intros i i_less.
    rewrite nth_map.
    unfold propositional_sequent in *.
    apply (rank_sequent_subst_add _ _ 1 1); trivial.
    apply rank_prop_sequent.
    specialize (H _ H2).
    unfold one_step_rule in *.
    destruct H.
    apply H.
  Qed.

  (** The set of assumptions *)
  Definition valid_subst_n_conclusions
                     {X : Type}(nonempty_v : V)
                     (rules : set (sequent_rule V L))
                     (osr : one_step_rule_set rules)
                     (coval : X -> set V) : set (sequent V L) :=
    fun(s : sequent V L) =>
      exists(r : sequent_rule V L),
        swr # simple_mod_weaken_rule rules r /#\
        s = conclusion r /\
        forall(i : nat)(i_less : i < length (assumptions r)),
          prop_seq_val_valid nonempty_v coval 
            (nth (assumptions r) i i_less)
            (simple_mod_weaken_rule_assumptions _ _ osr swr _ _).

  Lemma multiset_valid_subst_n_conclusions :
    forall(X : Type)(nonempty_v : V)(rules : set (sequent_rule V L))
          (osr : one_step_rule_set rules)(coval : X -> set V),
      sequent_multiset 
        (valid_subst_n_conclusions nonempty_v rules osr coval).
  Proof.
    clear. 
    unfold sequent_multiset in *.
    intros X nonempty_v rules osr coval s r H H0.
    unfold valid_subst_n_conclusions in *.
    decompose [ex and or dep_and] H; clear H.
    assert (simple_mod_weaken_rule rules 
              {| assumptions := assumptions x; conclusion := r |}).
      subst s.
      clear - H0 a.
      unfold simple_mod_weaken_rule in *.
      decompose [ex and or dep_and] a; clear a.
      exists x0, x1, x2.
      repeat split; auto.
      simpl.
      eapply list_reorder_trans.
        apply list_reorder_symm.
        eexact H0.
      trivial.
    exists {| assumptions := assumptions x; conclusion := r |}.
    simpl.
    constructor 1 with (a := H).
    split.
      trivial.
    intros i i_less.
    erewrite prop_seq_val_valid_tcc_irr.
    apply H2.
  Qed.

  Lemma rank_sequent_set_valid_subst_n_conclusions :
    forall(X : Type)(nonempty_v : V)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules)
          (coval : X -> set V),
      rank_sequent_set 2
        (valid_subst_n_conclusions nonempty_v rules osr coval).
  Proof.
    clear. 
    intros X nonempty_v rules osr coval s H.
    unfold valid_subst_n_conclusions in *.
    decompose [ex and or dep_and] H; clear H.
    clear H1.
    subst s.
    unfold simple_mod_weaken_rule in *.
    decompose [ex and or dep_and] a; clear a.
    eapply rank_sequent_list_reorder.
      eexact H4.
    apply rank_sequent_append.
      eapply rank_sequent_subst_add.
            apply rank_simple_modal_sequent.
            apply one_step_rule_simple_modal_conclusion.
            apply osr.
            trivial.
          eexact H.
        trivial.
      trivial.
    apply rank_simple_modal_sequent.
    trivial.
  Qed.

  (** Generic one-step completeness *)
  Definition one_step_generic_complete
             (nonempty_v : V)(LS : lambda_structure L T)
             (prop_rules : set (sequent_rule V L))
             (rules : set (sequent_rule V L))
             (osr : one_step_rule_set rules) : Prop :=
    forall(X : Type)(coval : X -> set V)
          (gamma : sequent V L)(gamma_nonempty : gamma <> [])
          (gamma_simple : simple_modal_sequent gamma),
      mod_seq_val_valid LS coval gamma gamma_nonempty
             (simple_modal_sequent_is_prop_modal_prop gamma gamma_simple) ->
        provable prop_rules 
          (valid_subst_n_conclusions nonempty_v rules osr coval)
          gamma.

  (** **** One-step completeness, Def 4.1 *)
  Definition one_step_complete(nonempty_v : V)(LS : lambda_structure L T)
             (rules : set (sequent_rule V L))
             (osr : one_step_rule_set rules) : Prop :=
    one_step_generic_complete nonempty_v LS (GC_n_set V L 2) rules osr.

  (** **** One-step cut-free completeness, Def 4.1 *)
  Definition one_step_cut_free_complete
             (nonempty_v : V)
             (LS : lambda_structure L T)
             (rules : set (sequent_rule V L))
             (osr : one_step_rule_set rules) : Prop :=
    one_step_generic_complete nonempty_v LS (G_n_set V L 2) rules osr.


  (***************************************************************************)
  (** ***  Simple characterization of one-step cut-free completeness
          Lemma 4.3, page 17 *)
  (***************************************************************************)

  Lemma propositional_renamed_one_step_assumptions :
    forall(rules : set (sequent_rule V L))(r : sequent_rule V L)
          (sigma : lambda_subst V L)
          (i : nat)(i_less : i < length (assumptions r)),
      one_step_rule_set rules ->
      rules r ->
      renaming sigma ->
        propositional_sequent 
           (subst_sequent sigma (nth (assumptions r) i i_less)).
  Proof.
    intros rules r sigma i i_less H H0 H1.
    apply rank_sequent_subst_nth_assumptions; auto.
    apply rank_renaming.
    trivial.
  Qed.


  (** **** Lemma 4.3 *)
  Lemma simple_one_step_cut_free_complete :
    forall(nonempty_v : V)(LS : lambda_structure L T)
          (rules : set (sequent_rule V L))(osr : one_step_rule_set rules),
      one_step_cut_free_complete nonempty_v LS rules osr <->
      forall(X : Type)(coval : X -> set V)
            (gamma : sequent V L)(gamma_nonempty : gamma <> [])
            (gamma_simple : simple_modal_sequent gamma),
        mod_seq_val_valid LS coval gamma gamma_nonempty
               (simple_modal_sequent_is_prop_modal_prop gamma gamma_simple) ->
          exists(r : sequent_rule V L)(sigma : lambda_subst V L),
            rules_r # rules r /#\
            renaming_sigma # renaming sigma /#\
            multi_subset (subst_sequent sigma (conclusion r)) gamma /\
            forall(i : nat)(i_less : i < length (assumptions r)),
              prop_seq_val_valid nonempty_v coval 
                (subst_sequent sigma (nth (assumptions r) i i_less))
                (propositional_renamed_one_step_assumptions rules r sigma 
                   i i_less osr rules_r renaming_sigma).
  Proof.
    intros nonempty_v LS rules osr.
    split.
      intros H X coval gamma gamma_nonempty gamma_simple H0.
      unfold one_step_cut_free_complete, one_step_generic_complete in *.
      specialize (H X coval gamma gamma_nonempty gamma_simple H0).
      clear H0.
      destruct H.
      clear H.
      destruct x.
        clear gamma_nonempty.
        unfold valid_subst_n_conclusions in *.
        decompose [ex and or dep_and] v; clear v.
        rename x into r_subst.
        assert (H1 := a).
        unfold simple_mod_weaken_rule in H1.
        decompose [ex and or dep_and] H1; clear H1.
        rename x into rbase, x0 into sigma, x1 into delta.
        exists rbase. 
        exists (limit_subst v_eq (prop_var_sequent (conclusion rbase)) sigma).
        constructor 1 with (a := H3).
        assert (renaming (limit_subst v_eq 
                             (prop_var_sequent (conclusion rbase)) sigma)).
          clear - osr gamma_simple H H3 H7.
          apply renaming_limit_subst_simple_modal_sequent.
              trivial.
            apply one_step_rule_simple_modal_conclusion.
            apply osr.
            trivial.
          eapply simple_modal_sequent_append_left.
          eapply simple_modal_sequent_list_reorder.
            eexact H7.
          subst gamma.
          trivial.
        constructor 1 with (a := H1).
        split.
          rewrite subst_sequent_limit_eq.
              exists delta.
              subst gamma.
              apply list_reorder_symm.
              trivial.
            trivial.
          apply incl_refl.
        intros i i_less.
        generalize (propositional_renamed_one_step_assumptions rules rbase
                      (limit_subst v_eq (prop_var_sequent (conclusion rbase)) 
                         sigma) i i_less osr H3 H1).
        rewrite subst_sequent_limit_eq.
            rewrite nth_map_inv.
            generalize (nth_map_inv_tcc (sequent V L) (sequent V L) 
                           (subst_sequent sigma) (assumptions rbase) i i_less).
            rewrite <- H5.
            intros l p.
            erewrite prop_seq_val_valid_tcc_irr.
            apply H0.
          trivial.
        clear - H3 osr.
        apply one_step_rule_incl_prop_var_sequent.
        apply osr.
        trivial.
      exfalso.
      eapply G_n_set_no_simple_modal_conclusion.
        eexact g.
      trivial.
    intros H.
    unfold one_step_cut_free_complete, one_step_generic_complete in *.
    intros X coval gamma gamma_nonempty gamma_simple H0.
    specialize (H X coval gamma gamma_nonempty gamma_simple H0).
    clear H0.
    decompose [ex and or dep_and] H; clear H.
    rename x into r, x0 into sigma, a into rules_r, a0 into renaming_sigma.
    assert (valid_subst_n_conclusions nonempty_v rules osr coval gamma).
      unfold valid_subst_n_conclusions in *.
      set (sr := {| assumptions := map (subst_sequent sigma) (assumptions r); 
                    conclusion := gamma |}).
      exists sr.
      assert (simple_mod_weaken_rule rules sr).
        unfold simple_mod_weaken_rule in *.
        unfold multi_subset in *.
        destruct H0.
        rename x into delta.
        exists r, sigma, delta.
        repeat split; auto.
            apply rank_renaming.
            trivial.
          eapply simple_modal_sequent_append_right.
          eapply simple_modal_sequent_list_reorder.
            apply list_reorder_symm.
            eexact H.
          trivial.
        apply list_reorder_symm.
        trivial.
      constructor 1 with (a := H).
      split.
        trivial.
      intros i i_less.
      subst sr.
      simpl in *.
      generalize 
        (simple_mod_weaken_rule_assumptions rules
           {| assumptions := map (subst_sequent sigma) (assumptions r);
              conclusion := gamma |} 
           osr H i i_less).
      clear H0 H.
      simpl.
      rewrite nth_map.
      intros p.
      eapply prop_seq_val_valid_tcc_irr.
      apply H1.
    exists (assume _ _ _ H).
    trivial.
  Qed.


End One_step_conditions.

Implicit Arguments one_step_sound [V L T].
Implicit Arguments valid_subst_n_conclusions [V L X].
Implicit Arguments simple_mod_weaken_rule_assumptions [V L].
Implicit Arguments one_step_complete [V L T].
Implicit Arguments one_step_cut_free_complete [V L T].
Implicit Arguments simple_one_step_cut_free_complete [V L T].
