(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: renaming.v,v 1.20 2013/04/10 11:17:16 tews Exp $
 *)

(** ** Renaming

      Here, a renaming is a substitution with the special property
      that there are only propositional variables in its codomain.
      This permits to reuse all substitution lemmas on renamings.
*)


Require Export modal_formulas propositional_formulas.

Section Renaming.

  Variable V : Type.
  Variable L : modal_operators.

  (** need a decidable equality for the limit_subst results *)
  Variable v_eq : eq_type V.


  (***************************************************************************)
  (** *** Definition *)
  (***************************************************************************)

  Definition renaming(sigma : lambda_subst V L) : Prop :=
    forall(v : V), prop_form (sigma v).

  Lemma renaming_id_subst : renaming (id_subst).
  Proof.
    unfold renaming, id_subst in *.
    intros v.
    simpl.
    trivial.
  Qed.

  Lemma rank_renaming : forall(sigma : lambda_subst V L),
    renaming sigma -> rank_subst 1 sigma.
  Proof.
    intros sigma H v.
    apply rank_formula_prop_form.
    apply H.
  Qed.

  Lemma rank_sequent_renaming :
    forall(s : sequent V L)(n : nat)(sigma : lambda_subst V L),
      renaming sigma ->
      rank_sequent n s ->
        rank_sequent n (subst_sequent sigma s).
  Proof.
    intros s n sigma H H0.
    eapply rank_sequent_subst_add.
          eexact H0.
        apply rank_renaming.
        trivial.
      trivial.
    apply add_minus_diag.
  Qed.

  Lemma propositional_renaming :
    forall(s : sequent V L)(sigma : lambda_subst V L),
      renaming sigma ->
      propositional_sequent s ->
        propositional_sequent (subst_sequent sigma s).
  Proof.
    intros s.
    apply rank_sequent_renaming.
  Qed.

  Lemma renaming_subst_compose :
    forall(sigma rho : lambda_subst V L),
      renaming sigma ->
      renaming rho ->
        renaming (subst_compose sigma rho).
  Proof.
    unfold renaming, subst_compose in *.
    intros sigma rho H H0 v.
    specialize (H0 v).
    destruct (rho v); try contradiction.
    rewrite subst_form_char.
    apply H.
  Qed.


  (**************************************************************************)
  (** *** Renaming on prop_sequent's  *)
  (**************************************************************************)

  Lemma prop_form_renaming :
    forall(sigma : lambda_subst V L)(f : lambda_formula V L),
      renaming sigma ->
      neg_form_maybe prop_form f ->
        neg_form_maybe prop_form (subst_form sigma f).
  Proof.
    intros sigma f H H0.
    destruct f; try contradiction.
      rewrite subst_form_char.
      specialize (H v).
      destruct (sigma v); try contradiction.
      trivial.
    simpl in H0.
    destruct f; try contradiction.
    rewrite subst_form_char.
    rewrite subst_form_char.
    specialize (H v).
    destruct (sigma v); try contradiction.
    trivial.
  Qed.

  Lemma prop_sequent_renaming :
    forall(sigma : lambda_subst V L)(s : sequent V L),
      renaming sigma ->
      prop_sequent s ->
        prop_sequent (subst_sequent sigma s).
  Proof.
    intros sigma s H H0.
    apply every_nth_map.
    intros i i_less.
    apply prop_form_renaming.
      trivial.
    apply H0.
  Qed.


  (**************************************************************************)
  (** *** A substitution on simple_modal_sequent's is a renaming  *)
  (**************************************************************************)

  Lemma prop_form_subst_simple_modal_form :
    forall(f : lambda_formula V L)(sigma : lambda_subst V L)(v : V),
      In v (prop_var_formula f) ->
      simple_modal_form f ->
      simple_modal_form (subst_form sigma f) ->
        prop_form (sigma v).
  Proof.
    intros f sigma v H H0 H1.
    destruct f; try contradiction.
    rewrite subst_form_char in H1.
    rewrite prop_var_formula_char in H.
    simpl in *.
    rewrite list_of_counted_list_map in H1.
    apply In_flatten in H.
    decompose [ex and or dep_and] H; clear H.
    rename x into n, a into n_less, b into H.
    rewrite nth_map in H.
    rewrite every_nth_map in H1.
    specialize (H0 n (nth_map_tcc _ _ _ n_less)).
    specialize (H1 n (nth_map_tcc _ _ _ n_less)).
    simpl in *.
    destruct (nth (list_of_counted_list c) n _); try contradiction.
    rewrite subst_form_char in H1.
    simpl in *.
    destruct H.
      subst v0.
      trivial.
    contradiction.
  Qed.

  Lemma prop_form_subst_neg_simple_modal_form :
    forall(f : lambda_formula V L)(sigma : lambda_subst V L)(v : V),
      In v (prop_var_formula f) ->
      neg_form_maybe simple_modal_form f ->
      neg_form_maybe simple_modal_form (subst_form sigma f) ->
        prop_form (sigma v).
  Proof.
    intros f sigma v H H0 H1.
    destruct f; try contradiction.
      rewrite subst_form_char in H1.
      rewrite prop_var_formula_char in H.
      simpl in *.
      eapply prop_form_subst_simple_modal_form; eauto.
    eapply prop_form_subst_simple_modal_form; eauto.
  Qed.

  Lemma renaming_limit_subst_simple_modal_sequent :
    forall(s : sequent V L)(sigma : lambda_subst V L),
      simple_modal_sequent s ->
      simple_modal_sequent (subst_sequent sigma s) ->
        renaming (limit_subst v_eq (prop_var_sequent s) sigma).
  Proof.
    unfold renaming, limit_subst in *.
    intros s sigma H H0 v.
    destruct (member v_eq v (prop_var_sequent s)) eqn:H1.
      apply member_In in H1.
      apply In_prop_var_sequent in H1.
      destruct H1 as [f].
      destruct H1.
      eapply prop_form_subst_neg_simple_modal_form.
          eexact H1.
        eapply every_nth_In.
          eexact H.
        trivial.
      eapply every_nth_In.
        eexact H0.
      apply in_map.
      trivial.
    simpl.
    trivial.
  Qed.


  (**************************************************************************)
  (** *** Renaming on simple_modal_sequents  *)
  (**************************************************************************)

  Lemma simple_modal_form_renaming :
    forall(sigma : lambda_subst V L)(f : lambda_formula V L),
      renaming sigma ->
      simple_modal_form f ->
        simple_modal_form (subst_form sigma f).
  Proof.
    clear. 
    intros sigma f H H0.
    destruct f as [ | | | op args ]; try contradiction.
    rewrite subst_form_char.
    simpl in *.
    rewrite list_of_counted_list_map.
    induction (list_of_counted_list args).
      apply every_nth_empty.
    clear op args.
    simpl.
    apply every_nth_cons.
      apply every_nth_head in H0.
      clear IHl.
      destruct a; try contradiction.
      rewrite subst_form_char.
      apply H.
    apply IHl.
    eapply every_nth_tail.
    eexact H0.
  Qed.

  Lemma neg_simple_modal_form_renaming :
    forall(sigma : lambda_subst V L)(f : lambda_formula V L),
      renaming sigma ->
      neg_form_maybe simple_modal_form f ->
        neg_form_maybe simple_modal_form (subst_form sigma f).
  Proof.
    intros sigma f H H0.
    destruct f; try contradiction.
      rewrite subst_form_char.
      simpl in *.
      apply simple_modal_form_renaming; trivial.
    rewrite subst_form_char.
    unfold neg_form_maybe in *.
    rewrite <- (subst_form_char sigma (lf_modal op c)).
    apply simple_modal_form_renaming; trivial.
  Qed.

  Lemma simple_modal_sequent_renaming :
    forall(sigma : lambda_subst V L)(s : sequent V L),
      renaming sigma ->
      simple_modal_sequent s ->
        simple_modal_sequent (subst_sequent sigma s).
  Proof.
    induction s.
      trivial.
    intros H H0.
    simpl.
    apply simple_modal_sequent_cons.
      apply neg_simple_modal_form_renaming.
        trivial.
      eapply simple_modal_sequent_head.
      eexact H0.
    apply IHs.
      trivial.
    eapply simple_modal_sequent_tail.
    eexact H0.
  Qed.


  (**************************************************************************)
  (** *** Renamings from functions on V *)
  (**************************************************************************)

  Definition rename_of_fun(f : V -> V) : lambda_subst V L :=
    fun(v : V) => lf_prop (f v).

  Lemma renaming_rename_of_fun : forall(f : V -> V),
    renaming (rename_of_fun f).
  Proof.
    intros f v.
    simpl.
    trivial.
  Qed.

  Lemma rename_of_id : rename_of_fun id = id_subst.
  Proof.
    trivial.
  Qed.

End Renaming.

Implicit Arguments renaming [V L].
Implicit Arguments rename_of_fun [V L].
Implicit Arguments renaming_rename_of_fun [V L].
