(* 
 * Formalized Cut Elimination in Coalgebraic Logics
 * 
 * Copyright (C) 2013 - 2013 Hendrik Tews
 * 
 * This file is part of my formalization of "Cut Elimination in 
 * Coalgebraic Logics" by Dirk Pattinson and Lutz Schroeder.
 * 
 * The formalization is free software: you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation, either version 3 of the
 * License, or (at your option) any later version.
 * 
 * The formalization is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this or one of the parent
 * directories for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with the formalization in the file COPYING. 
 * If not, see <http://www.gnu.org/licenses/>.
 * 
 * $Id: some_neg.v,v 1.10 2013/04/10 11:17:16 tews Exp $
 *)

(** ** A special some predicate for lists

      [some_neg] asserts that some element of a list [l] fulfills a
      predicate [P], with the twist that it asserts [P] for singleton
      lists and [~~P] if the length of [l] is different than 1.

      [some_neg] corresponds precisely to the semantics of sequents.
      For sequents with more than one formula, the formulas are or-ed,
      but because disjunction is not in the object logic, 
      [~ (~ _ /\ ~ _)] is used. This leads to the double negation for
      sequents with more than one formula.

      I first define the dependent version [some_neg_dep]. Dependent
      here means that the predicate [P] can only be applied when a
      certain condition is true. [some_neg_dep] gets this condition
      for the whole list in the form of an [every_nth] predicate. 

      The simple independent version and all its results are directly
      derived from the dependent version.
 *)

Require Export reorder.


  (***************************************************************************)
  (** ***  Dependent version  *)
  (***************************************************************************)

Section Some_neg_dep.

  Variable A : Type.
  Variable aprop : A -> Prop.
  Variable P : forall(a : A), aprop a -> Prop.

  Hypothesis P_tcc_irr : forall(a : A)(ta1 ta2 : aprop a),
    P a ta1 -> P a ta2.


  (***************************************************************************)
  (** **** Definition *)
  (***************************************************************************)

  Definition a_list_prop(l : list A) : Prop :=
    every_nth aprop l.

  Definition some_neg_dep(l : list A)(dep : a_list_prop l) : Prop :=
    (len # length l = 1 /#\ 
         P (nth l 0 (nth_head_tcc l len)) (dep 0 (nth_head_tcc l len)))
    \/
    (length l <> 1 /\
     ~ ~ exists(n : nat),
           n_less # n < length l /#\ P (nth l n n_less) (dep n n_less)).

  Lemma some_neg_dep_tcc_irr : forall(l : list A)(dep1 dep2 : a_list_prop l),
    some_neg_dep l dep1 -> some_neg_dep l dep2.
  Proof.
    intros l dep1 dep2 H.
    unfold some_neg_dep in *.
    decompose [ex and or dep_and] H; clear H.
      left.
      apply dep_conj with (a := a).
      eapply P_tcc_irr.
      eexact b.
    right.
    split; trivial.
    intros H3.
    apply H2; clear H2; intros H2.
    apply H3; clear H3.
    decompose [ex and or dep_and] H2; clear H2.
    exists x.
    apply dep_conj with (a := a).
    eapply P_tcc_irr.
    eexact b.
  Qed.

  Lemma some_neg_dep_empty : forall(dep : a_list_prop []),
    some_neg_dep [] dep -> False.
  Proof.
    clear. 
    intros dep H.
    unfold some_neg_dep in *.
    decompose [ex and or dep_and] H; clear H.
      discriminate.
    apply H2; clear H2; intros H2.
    decompose [ex and or dep_and] H2; clear H2.
    simpl in *.
    omega.
  Qed.

  Lemma some_neg_dep_singleton : forall(a : A)(dep : a_list_prop [a]),
    some_neg_dep [a] dep -> P a (dep 0 (lt_0_Sn 0)).
  Proof.
    intros a dep H.
    unfold some_neg_dep in *.
    decompose [ex and or dep_and] H; clear H.
      simpl in *.
      eapply P_tcc_irr.
      eexact b.
    exfalso.
    auto.
  Qed.

  Lemma some_neg_dep_singleton_rev : 
    forall(a : A)(dep : a_list_prop [a])(ta : aprop a),
      P a ta -> some_neg_dep [a] dep.
  Proof.
    intros a dep ta H.
    unfold some_neg_dep in *.
    left.
    apply dep_conj with (a := eq_refl).
    simpl.
    eapply P_tcc_irr.
    eexact H.
  Qed.


  (***************************************************************************)
  (** **** Introduction and Elimination *)
  (***************************************************************************)

  Lemma some_neg_dep_nth_intro :
    forall(l : list A)(dep : a_list_prop l)
          (n : nat)(n_less : n < length l)(tn : aprop (nth l n n_less)),
      P (nth l n n_less) tn -> some_neg_dep l dep.
  Proof.
    intros l dep n n_less tn H.
    destruct l.
      simpl in *.
      omega.
    destruct l.
      left.
      apply dep_conj with (a := eq_refl).
      destruct n.
        simpl in *.
        eapply P_tcc_irr.
        eexact H.
      simpl in n_less.
      omega.
    right.
    split.
      discriminate.
    intros H0; apply H0; clear H0.
    eexists.
    eexists.
    eapply P_tcc_irr.
    eexact H.
  Qed.

  Definition a_list_prop_head : 
    forall{a : A}{l : list A}, a_list_prop (a :: l) -> aprop a :=
      fun _ l dep => dep 0 (lt_0_Sn (length l)).

  Lemma a_list_prop_tail : 
    forall{a : A}{l : list A}, a_list_prop (a :: l) -> a_list_prop l.
  Proof.
    intros a l dep n n_less.
    assert (dep_sn := dep (S n) (lt_n_S _ _ n_less)).
    simpl in dep_sn.
    erewrite nth_tcc_irr.
    eexact dep_sn.
  Qed.

  Lemma some_neg_dep_cons_case_elim :
    forall(a : A)(l : list A)(dep : a_list_prop (a :: l)),
      some_neg_dep (a :: l) dep ->
        ((l = [] /\ P a (a_list_prop_head dep)) \/
         (l <> [] /\ ~(~P a (a_list_prop_head dep) /\ 
                       ~some_neg_dep l (a_list_prop_tail dep)))).
  Proof.
    intros a l dep H.
    destruct l as [| a' ].
      left.
      split; trivial.
      unfold some_neg_dep in *.
      decompose [ex and or dep_and] H; clear H.
        simpl in *.
        eapply P_tcc_irr.
        eexact b.
      exfalso.
      auto.
    right.
    split.
      discriminate.
    intros H0.
    destruct H0.
    unfold some_neg_dep in H.
    decompose [ex and or dep_and] H; clear H.
      discriminate.
    apply H4; clear H4; intros H4.
    decompose [ex and or dep_and] H4; clear H4.
    rename x into n, a0 into n_less, b into H4.
    destruct n.
      simpl in H4.
      apply H0.
      eapply P_tcc_irr.
      eexact H4.
    revert H4.
    generalize (dep (S n) n_less).
    rewrite nth_tail.
    intros t H4.
    eapply some_neg_dep_nth_intro in H4.
    apply H1.
    eexact H4.
  Qed.

  Lemma some_neg_dep_cons_cons_elim :
    forall(a a' : A)(l : list A)(dep : a_list_prop (a :: a' :: l)),
      some_neg_dep (a :: a' :: l) dep ->
        ~(~P a (a_list_prop_head dep) /\ 
          ~some_neg_dep (a' :: l) (a_list_prop_tail dep)).
  Proof.
    intros a a' l dep H.
    apply some_neg_dep_cons_case_elim in H.
    decompose [and or] H; clear H.
      discriminate.
    trivial.
  Qed.

  Lemma some_neg_dep_head : 
    forall(a : A)(t0 : aprop a)(l : list A)(dep : a_list_prop (a :: l)), 
      P a t0 -> some_neg_dep (a :: l) dep.
  Proof.
    intros a t0 l dep H.
    eapply some_neg_dep_nth_intro with (n := 0)(n_less := lt_0_Sn (length l)).
    simpl.
    eexact H.
  Qed.

  (** some_neg_dep_tail is further below *)

  Lemma some_neg_dep_length_case_intro :
    forall(l : list A)(dep : a_list_prop l),
      (forall(len : length l = 1), 
         P (nth l 0 (nth_head_tcc l len)) (dep 0 (nth_head_tcc l len))) ->
      (length l <> 1 -> ~~some_neg_dep l dep) ->
        some_neg_dep l dep.
  Proof.
    intros l dep H H0.
    assert (length l = 1 \/ length l <> 1).
      omega.
    destruct H1.
      left.
      apply dep_conj with (a := H1).
      apply H.
    right.
    split; trivial.
    intros H2.
    apply H0; clear H0.
      trivial.
    intros H0.
    unfold some_neg_dep in *.
    decompose [ex and or dep_and] H0; clear H0.
      contradiction.
    contradiction.
  Qed.

  Lemma some_neg_dep_cons_case_intro : 
    forall(a : A)(l : list A)(dep : a_list_prop (a :: l)),
      (l = [] -> P a (a_list_prop_head dep)) ->
      (l <> [] -> ~~some_neg_dep (a :: l) dep) ->
        some_neg_dep (a :: l) dep.
  Proof.
    intros a l dep H H0.
    apply some_neg_dep_length_case_intro.
      intros len.
      simpl in *.
      eapply P_tcc_irr.
      apply H.
      destruct l.
        trivial.
      discriminate.
    intros H1.
    apply H0.
    destruct l.
      exfalso.
      auto.
    discriminate.
  Qed.

  Lemma some_neg_dep_long_neg_intro : forall(l : list A)(dep : a_list_prop l),
    length l <> 1 -> ~~some_neg_dep l dep -> some_neg_dep l dep.
  Proof.
    intros l dep H H0.
    apply some_neg_dep_length_case_intro.
      intros len.
      contradiction.
    auto.
  Qed.


  (***************************************************************************)
  (** **** Append Properties *)
  (***************************************************************************)

  Lemma dep_append_left : 
    forall{l1 l2 : list A}, a_list_prop (l1 ++ l2) -> a_list_prop l1.
  Proof.
    intros l1 l2 dep n n_less.
    assert (n < length (l1 ++ l2)).
      rewrite app_length.
      apply lt_plus_trans.
      trivial.
    rewrite <- nth_append_left with (n_less_app := H).
    apply dep.
  Qed.

  Lemma dep_append_right : 
    forall{l1 l2 : list A}, a_list_prop (l1 ++ l2) -> a_list_prop l2.
  Proof.
    intros l1 l2 dep n n_less.
    assert (n + length l1 < length (l1 ++ l2)).
      rewrite app_length.
      rewrite plus_comm.
      apply plus_lt_compat_l.
      trivial.
    specialize (dep _ H).
    rewrite nth_append_right with (n_greater := le_plus_r _ _ ) in dep.
    revert dep.
    generalize (nth_append_right_tcc l1 l2 (n + length l1) H
                                     (le_plus_r n (length l1))).
    rewrite add_minus_diag.
    intros l dep.
    erewrite nth_tcc_irr.
    eexact dep.
  Qed.

  Lemma some_neg_dep_append : 
    forall(l1 l2 : list A)(dep : a_list_prop (l1 ++ l2)),
      some_neg_dep (l1 ++ l2) dep ->
        ~ ( ~ (some_neg_dep l1 (dep_append_left dep)) /\ 
            ~ (some_neg_dep l2 (dep_append_right dep))).
  Proof.
    intros l1 l2 dep H H0.
    destruct H0.
    destruct l1 as [| a1 ].
      apply H1.
      eapply some_neg_dep_tcc_irr.
      eexact H.
    destruct l2 as [| a2 ].
      revert H.
      generalize dep.
      rewrite app_nil_r in *.
      intros dep' H.
      apply H0.
      eapply some_neg_dep_tcc_irr.
      eexact H.
    unfold some_neg_dep in H.
    decompose [ex and or dep_and] H; clear H.
      clear - a.
      rewrite app_length in a.
      simpl in a.
      omega.
    apply H4; clear H4; intros H4.
    decompose [ex and or dep_and] H4; clear H4.
    rename x into n, a into n_less, b into H4.
    apply (split_nat_case_lt n (length (a1 :: l1))).
      intros H.
      apply H0; clear H0 H1.
      revert H4.
      generalize (dep n n_less).
      rewrite nth_append_left with (n_less_l1 := H).
      intros tn H4.
      eapply some_neg_dep_nth_intro in H4.
      eexact H4.
    intros H.
    apply H1; clear H0 H1.
    revert H4.
    generalize (dep n n_less).
    rewrite nth_append_right with (n_greater := H).
    intros tn H4.
    eapply some_neg_dep_nth_intro in H4.
    eexact H4.
  Qed.

  Lemma some_neg_dep_append_right : 
    forall(l1 l2 : list A)(dep : a_list_prop (l1 ++ l2)),
      some_neg_dep l1 (dep_append_left dep) -> some_neg_dep (l1 ++ l2) dep.
  Proof.
    intros l1 l2 dep H.
    destruct l2 as [| a' ].
      generalize dep.
      rewrite app_nil_r.
      intros dep'.
      eapply some_neg_dep_tcc_irr.
      eexact H.
    destruct l1 as [| a ].
      apply some_neg_dep_empty in H.
      contradiction.
    unfold some_neg_dep in H.
    decompose [ex and or dep_and] H; clear H.
      simpl in *.
      eapply some_neg_dep_nth_intro with (n := 0)
                         (n_less := lt_0_Sn (length (l1 ++ a' :: l2))).
      eexact b.
    apply some_neg_dep_long_neg_intro.
      rewrite app_length.
      simpl in *.
      omega.
    intros H0.
    apply H2; clear H2; intros H2.
    apply H0; clear H0.
    decompose [ex and or dep_and] H2; clear H2.
    rename x into n, a0 into n_less, b into H2.
    assert (n < length ((a :: l1) ++ a' :: l2)).
      rewrite app_length.
      apply lt_plus_trans.
      trivial.
    revert H2.
    generalize (dep_append_left dep n n_less).
    rewrite <- nth_append_left with (n_less_app := H).
    intros tn H2.
    eapply some_neg_dep_nth_intro in H2.
    eexact H2.
  Qed.

  Lemma some_neg_dep_append_left : 
    forall(l1 l2 : list A)(dep : a_list_prop (l1 ++ l2)),
      some_neg_dep l2 (dep_append_right dep) -> some_neg_dep (l1 ++ l2) dep.
  Proof.
    intros l1 l2 dep H.
    destruct l1 as [| a ].
      eapply some_neg_dep_tcc_irr.
      eexact H.
    destruct l2 as [| a' ].
      apply some_neg_dep_empty in H.
      contradiction.
    unfold some_neg_dep in H.
    decompose [ex and or dep_and] H; clear H.
      assert (length (a :: l1) < length ((a :: l1) ++ a' :: l2)).
        rewrite app_length.
        omega.
      apply some_neg_dep_nth_intro with 
            (n := length (a :: l1)) (n_less := H) (tn := dep _ H).
      generalize (dep (length (a :: l1)) H).
      rewrite nth_append_right with (n_greater := le_refl _).
      generalize (nth_append_right_tcc (a :: l1) (a' :: l2) 
                     (length (a :: l1)) H (le_refl (length (a :: l1)))).
      rewrite minus_diag.
      intros l t0.
      eapply P_tcc_irr.
      eexact b.
    apply some_neg_dep_long_neg_intro.
      rewrite app_length.
      simpl.
      omega.
    intros H3.
    apply H2; clear H2; intros H2.
    apply H3; clear H3.
    decompose [ex and or dep_and] H2; clear H2.
    rename x into n, a0 into n_less, b into H2.
    assert (n + length (a :: l1) < length ((a :: l1) ++ a' :: l2)).
      rewrite plus_comm.
      rewrite app_length.
      apply plus_le_lt_compat; trivial.
    apply some_neg_dep_nth_intro with (n_less := H)(tn := dep _ H).
    generalize (dep (n + length (a :: l1)) H).
    rewrite nth_append_right with (n_greater := le_plus_r _ _).
    generalize (nth_append_right_tcc (a :: l1) (a' :: l2) 
                  (n + length (a :: l1)) H (le_plus_r n (length (a :: l1)))).
    rewrite add_minus_diag.
    intros l.
    erewrite nth_tcc_irr.
    intros tn.
    eapply P_tcc_irr.
    eexact H2.
  Qed.

  Lemma some_neg_dep_tail : 
    forall(a : A)(l : list A)(dep : a_list_prop (a :: l)),
    some_neg_dep l (a_list_prop_tail dep) -> some_neg_dep (a :: l) dep.
  Proof.
    intros a l dep H.
    generalize dep.
    rewrite append_single_rev.
    intros dep'.
    apply some_neg_dep_append_left.
    eapply some_neg_dep_tcc_irr.
    eexact H.
  Qed.


  (***************************************************************************)
  (** *** Reorder Properties *)
  (***************************************************************************)

  Lemma some_neg_dep_reorder : 
    forall(l1 l2 : list A)(dep1 : a_list_prop l1)(dep2 : a_list_prop l2),
      list_reorder l1 l2 ->
      some_neg_dep l1 dep1 ->
        some_neg_dep l2 dep2.
  Proof.
    intros l1 l2 dep1 dep2 H H0.
    destruct l1 as [| a1 ].
      apply some_neg_dep_empty in H0.
      contradiction.
    destruct l1 as [| a1' ].
      apply list_reorder_singleton in H.
      subst l2.
      eapply some_neg_dep_tcc_irr.
      eexact H0.
    apply some_neg_dep_long_neg_intro.
      rewrite list_reorder_length with (l2 := a1 :: a1' :: l1).
        discriminate.
      apply list_reorder_symm.
      trivial.
    unfold some_neg_dep in H0.
    decompose [ex and or dep_and] H0; clear H0.
      discriminate.
    intros H4.
    apply H3; clear H3; intros H3.
    apply H4; clear H4.
    decompose [ex and or dep_and] H3; clear H3.
    rename x into n, a into n_less, b into H3.
    assert(H0 := list_reorder_occurence _ _ n n_less H).
    decompose [ex and or dep_and] H0; clear H0.
    rename x into n', a into n'_less, b into H4.
    revert H3.
    generalize (dep1 n n_less).
    rewrite H4.
    intros tn H3.
    eapply some_neg_dep_nth_intro in H3.
    eexact H3.
  Qed.

End Some_neg_dep.

Implicit Arguments a_list_prop [A].
Implicit Arguments some_neg_dep [A].
Implicit Arguments a_list_prop_head [aprop A a l].
Implicit Arguments a_list_prop_tail [aprop A a l].


Lemma some_neg_dep_map : 
  forall{A B : Type}(f : A -> B)(aprop : A -> Prop)(bprop : B -> Prop)
        (prop_fun : forall(a : A), aprop a -> bprop (f a))
        (P : forall(b : B), bprop b -> Prop)(l : list A)
        (adep : a_list_prop aprop l)(bdep : a_list_prop bprop (map f l)),
    (forall (b : B) (tb1 tb2 : bprop b), P b tb1 -> P b tb2) ->
    some_neg_dep aprop 
         (fun(a : A)(ap : aprop a) => P (f a) (prop_fun a ap)) l adep ->
      some_neg_dep bprop P (map f l) bdep.
Proof.
  intros A B f aprop bprop prop_fun P l adep bdep H H0.
  unfold some_neg_dep in H0.
  decompose [ex and or dep_and] H0; clear H0.
    revert b.
    generalize (prop_fun (nth l 0 (nth_head_tcc l a)) 
                         (adep 0 (nth_head_tcc l a))).
    rewrite nth_map_inv.
    intros b H0.
    eapply some_neg_dep_nth_intro; trivial.
    eexact H0.
  apply some_neg_dep_long_neg_intro.
    rewrite map_length.
    trivial.
  intros H4.
  apply H3; clear H3; intros H3.
  decompose [ex and or dep_and] H3; clear H3.
  revert b.
  generalize (prop_fun (nth l x a) (adep x a)).
  rewrite nth_map_inv.
  intros b H5.
  apply H4.
  eapply some_neg_dep_nth_intro; trivial.
  eexact H5.
Qed.


  (***************************************************************************)
  (** *** Simple, independent version *)
  (***************************************************************************)

Section Some_neg.

  Variable A : Type.
  Variable P : A -> Prop.

  Definition noprop(a : A) : Prop := True.

  Definition lift_p : forall(a : A), noprop a -> Prop := fun a _ => P a.

  Lemma lift_p_tcc_irr : forall(a : A)(ta1 ta2 : noprop a), 
    lift_p a ta1 -> lift_p a ta2.
  Proof.
    trivial.
  Qed.

  Definition list_noprop(l : list A) : a_list_prop noprop l := fun _ _ => I.

  Definition some_neg(l : list A) : Prop := 
    some_neg_dep noprop lift_p l (list_noprop _).

  (** All properties are phrased as definitions in the following,
     because their proof simply uses the dependent proof with some
     additional arguments.
  *)

  Definition some_neg_empty : 
    some_neg [] -> False := 
    some_neg_dep_empty A noprop lift_p (list_noprop _).

  Definition some_neg_singleton_for : 
    forall(a : A), some_neg [a] -> P a :=
    fun a => some_neg_dep_singleton 
                  A noprop lift_p lift_p_tcc_irr a (list_noprop _).

  Definition some_neg_singleton_back : 
    forall(a : A), P a -> some_neg [a] :=
    fun a => some_neg_dep_singleton_rev 
                  A noprop lift_p lift_p_tcc_irr a (list_noprop _) I.

  Lemma some_neg_singleton : forall(a : A),
      some_neg [a] <-> P a.
  Proof.
    split.
      apply some_neg_singleton_for.
    apply some_neg_singleton_back.
  Qed.

  Definition some_neg_nth_intro :
    forall(l : list A)(n : nat)(n_less : n < length l),
      P (nth l n n_less) -> some_neg l :=
    fun l n n_less => some_neg_dep_nth_intro 
           A noprop lift_p lift_p_tcc_irr l (list_noprop _) n n_less I.

  Definition some_neg_cons_case_elim :
    forall(a : A)(l : list A),
      some_neg (a :: l) ->
        ((l = [] /\ P a) \/
         (l <> [] /\ ~(~P a /\ ~some_neg l))) :=
    fun a l => some_neg_dep_cons_case_elim
                 A noprop lift_p lift_p_tcc_irr a l (list_noprop _).

  Definition some_neg_head : 
    forall(a : A)(l : list A), P a -> some_neg (a :: l) :=
    fun a l => some_neg_dep_head 
                    A noprop lift_p lift_p_tcc_irr a I l (list_noprop _).

  Definition some_neg_length_case_intro :
    forall(l : list A),
      (forall(len : length l = 1), P (nth l 0 (nth_head_tcc l len))) ->
      (length l <> 1 -> ~~some_neg l) ->
        some_neg l :=
    fun l => some_neg_dep_length_case_intro A noprop lift_p l (list_noprop _).

  Definition some_neg_cons_case_intro : 
    forall(a : A)(l : list A),
      (l = [] -> P a) ->
      (l <> [] -> ~~some_neg (a :: l)) ->
        some_neg (a :: l) :=
    fun a l => some_neg_dep_cons_case_intro
                    A noprop lift_p lift_p_tcc_irr a l (list_noprop _).

  Definition some_neg_long_neg_intro : 
    forall(l : list A), length l <> 1 -> ~~some_neg l -> some_neg l :=
    fun l => some_neg_dep_long_neg_intro A noprop lift_p l (list_noprop _).

  Definition some_neg_append : 
    forall(l1 l2 : list A),
      some_neg (l1 ++ l2) -> ~ ( ~ (some_neg l1) /\ ~ (some_neg l2)) :=
    fun l1 l2 => some_neg_dep_append
                      A noprop lift_p lift_p_tcc_irr l1 l2 (list_noprop _).

  Definition some_neg_append_right : 
    forall(l1 l2 : list A), some_neg l1 -> some_neg (l1 ++ l2) :=
    fun l1 l2 => some_neg_dep_append_right
                      A noprop lift_p lift_p_tcc_irr l1 l2 (list_noprop _).

  Definition some_neg_append_left : 
    forall(l1 l2 : list A), some_neg l2 -> some_neg (l1 ++ l2) :=
    fun l1 l2 => some_neg_dep_append_left
                      A noprop lift_p lift_p_tcc_irr l1 l2 (list_noprop _).

  Definition some_neg_tail : 
    forall(a : A)(l : list A), some_neg l -> some_neg (a :: l) :=
    fun a l => some_neg_dep_tail
                    A noprop lift_p lift_p_tcc_irr a l (list_noprop _).

  Definition some_neg_reorder : 
    forall(l1 l2 : list A), list_reorder l1 l2 -> some_neg l1 -> some_neg l2 :=
    fun l1 l2 => some_neg_dep_reorder
          A noprop lift_p lift_p_tcc_irr l1 l2 (list_noprop _) (list_noprop _).

End Some_neg.

Implicit Arguments some_neg [A].
