(*
 * Memcheck -- ocaml runtime type checking
 *
 * Copyright (C) 2006, Hendrik Tews, all right reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License in file COPYING in this directory
 * for more details.
 *
 * $Id: memcheck.ml,v 1.12 2016/10/14 19:24:08 tews Exp $
 *
 * Description: 
 * 
 * Implementation of Memcheck.
 *)


(* module Hashtbl = Myhashtbl *)

type type_tag = int
type block_tag = int

type ocaml_type =
  | Int_type of type_tag
  | Float_type of type_tag
  | String_type of type_tag
  | Int32_type of type_tag
  | Nativeint_type of type_tag
  | Type_parameter of int
  | Array_type of type_tag * ocaml_type
  | Tuple_type of type_tag * ocaml_type list
  | Static_variant of type_tag * string list * (string * ocaml_type list) list
  | Record_type of type_tag * (string * ocaml_type) list
  | Type_constructor_use of ocaml_type_constructor ref

and ocaml_type_constructor =
    (* Recursive_application(rec_type_constructor_nummer, arguments) *)
  | Recursive_application of int * ocaml_type list
  | Resolved_application of ocaml_type_descr * ocaml_type list
  | Applied of ocaml_type

and ocaml_type_descr =
    (* Type_constructor_use(type_constructor_id, arity, type_expression) *)
  | Type_constructor_def of string * int * ocaml_type



let invalid_type_tag = -1


(*****************************************************************************
 *
 * fixed builtin types
 * 
 ****************************************************************************)

let bool_type_constr_descr = 
  Type_constructor_def(
    "builtin bool", 0,
    Static_variant(invalid_type_tag, ["false"; "true"], []))

let bool_type_descr = 
  Type_constructor_use(ref(Resolved_application(bool_type_constr_descr, [])))


let int_type_constr_descr = Type_constructor_def("builtin int", 0, 
						 Int_type invalid_type_tag)

let int_type_descr = 
  Type_constructor_use(ref(Resolved_application(int_type_constr_descr, [])))


let int32_type_constr_descr = 
  Type_constructor_def("builtin int32", 0, Int32_type(invalid_type_tag))

let int32_type_descr = 
  Type_constructor_use(ref(Resolved_application(int32_type_constr_descr, [])))


let nativeint_type_constr_descr = 
  Type_constructor_def("builtin nativeint", 0, Nativeint_type(invalid_type_tag))

let nativeint_type_descr = 
  Type_constructor_use(
    ref(Resolved_application(nativeint_type_constr_descr, [])))


let float_type_constr_descr = 
  Type_constructor_def("builtin float", 0, Float_type(invalid_type_tag))

let float_type_descr = 
  Type_constructor_use(ref(Resolved_application(float_type_constr_descr, [])))


let string_type_constr_descr = 
  Type_constructor_def("builtin string", 0, String_type(invalid_type_tag))

let string_type_descr = 
  Type_constructor_use(ref(Resolved_application(string_type_constr_descr, [])))



let array_type_constr_descr = 
  Type_constructor_def("builtin array", 1, 
		       Array_type(invalid_type_tag, Type_parameter 0))

let array_type_descr content = 
  Type_constructor_use(
    ref(Resolved_application(array_type_constr_descr, [content])))



let rec tuple_param_tuple max n =
  if n >= max then []
  else
    Type_parameter(n) :: (tuple_param_tuple max (n+1))

let tuple_type_constr_descr size =
  Type_constructor_def(
    Printf.sprintf "builtin %d tuple" size,
    size,
    Tuple_type(invalid_type_tag, tuple_param_tuple size 0))


let tuple_type_descr tuple = 
  Type_constructor_use(ref(
    Resolved_application(
      tuple_type_constr_descr (List.length tuple), 
      tuple)))



(* the list type is recursive, thus it needs to be fixed !!! *)
let list_type_constr_descr =
  Type_constructor_def(
    "builtin list", 1,
    Static_variant(
      invalid_type_tag,
      ["[]"], 
      [("_ :: _", 
	[Type_parameter(0); 
	 Type_constructor_use(
	   ref(Recursive_application(0, [Type_parameter 0])))])]))

let list_type_descr content = 
  Type_constructor_use(ref(Resolved_application(list_type_constr_descr, [content])))



let option_type_constr_descr =
  Type_constructor_def(
    "builtin option", 1,
    Static_variant(
      invalid_type_tag,
      ["None"], 
      [("Some", [Type_parameter(0)])]))

let option_type_descr content = 
  Type_constructor_use(ref(
    Resolved_application(option_type_constr_descr, [content])))


(* type 'a ref = { mutable contents: 'a } *)
let ref_type_constr_descr =
  Type_constructor_def(
    "builtin ref", 1,
    Record_type(invalid_type_tag, ["contents", Type_parameter 0]))

let ref_type_descr content = 
  Type_constructor_use(ref(Resolved_application(ref_type_constr_descr, [content])))




(*****************************************************************************
 *****************************************************************************
 *
 * routines for ocaml types
 *
 * fix_type
 * 
 ****************************************************************************)


exception Invalid_ocaml_type of string

(*
 * fix_type <recursive type def list>
 *
 * fix_type will solve the recursion in a recursive type definition. 
 * It will update reference cells to replace Recursive_application nodes 
 * with type_def sequence numbers with Resolved_application nodes, pointing 
 * to the right type def.
 *)

let rec fix_type rec_type_list = function
  | Int_type _
  | Float_type _
  | String_type _
  | Int32_type _
  | Nativeint_type _
  | Type_parameter _ -> ()

  | Array_type(_, contents) -> fix_type rec_type_list contents

  | Tuple_type(_, typs) -> List.iter (fix_type rec_type_list) typs

  | Static_variant(_typ_id, _consts, vars) ->
	List.iter 
	  (fun (_name, args) -> List.iter (fix_type rec_type_list) args)
	  vars

  | Record_type(_, fields) ->
      List.iter (fun (_name, typ) -> fix_type rec_type_list typ) fields

  | Type_constructor_use(constr_ref) ->
      (match !constr_ref with
	 | Recursive_application(n, args) ->
	     (try
		let ty_constr = List.nth rec_type_list n in
		let arity = match ty_constr with
		  | Type_constructor_def(_name, n, _typ) -> n
		in
		  assert(arity = List.length args);
		  constr_ref := 
		    Resolved_application(
		      ty_constr,
		      (List.iter (fix_type rec_type_list) args;
		       args)
		    )
	      with
		| Failure "nth" -> assert false
	     )
	 | Resolved_application(_type_constr, args) ->
	     List.iter (fix_type rec_type_list) args
	 | Applied _ -> assert false
      )


let fix_rec_type_def rec_type_list =
  List.iter
    (function 
       | Type_constructor_def(_id, _n, typ) -> fix_type rec_type_list typ)
    rec_type_list



(*******************************************************************************
 *
 * fixing builtin types
 *
 ******************************************************************************)

let _ = fix_rec_type_def [list_type_constr_descr]


(*****************************************************************************
 *
 * substitution
 * 
 ****************************************************************************)

(* subst_type(arguments, typ) will substitute the arguments for 
 * (Type_parameter n) placeholders inside type typ. As a side effect it 
 * will construct a copy of typ, such that updates in its reference cells
 * will not affect the original type expression (from inside a 
 * type constructor def).
 *)

let rec subst_type arguments typ =
  match typ with
    | Int_type type_id
    | Float_type type_id
    | String_type type_id
    | Int32_type type_id
    | Nativeint_type type_id -> 
	assert(type_id = invalid_type_tag);
	typ

    | Type_parameter n ->
	if n >= List.length arguments then
	  raise (Invalid_ocaml_type "Unbound type parameter");
	List.nth arguments n

    | Array_type(type_id, content) ->
	assert(type_id = invalid_type_tag);
	Array_type(type_id, subst_type arguments content)

    | Tuple_type(type_id, type_list) ->
	assert(type_id = invalid_type_tag);
	Tuple_type(type_id, List.map (subst_type arguments) type_list)

    | Static_variant(type_id, consts, vars) ->
	assert(type_id = invalid_type_tag);
	Static_variant(
	  type_id,
	  consts,
	  List.map
	    (fun (name, args) -> (name, List.map (subst_type arguments) args)) 
	    vars
	)

    | Record_type(type_id, fields) ->
	assert(type_id = invalid_type_tag);
	Record_type(
	  type_id,
	  List.map
	    (fun (name, typ) -> (name, subst_type arguments typ))
	    fields
	)

    | Type_constructor_use constref ->
	(match !constref with
	   | Resolved_application(constr_def, args) ->
	       Type_constructor_use(
		 ref(
		   Resolved_application(
		     constr_def,
		     List.map (subst_type arguments) args)))

	   | Recursive_application _ ->
	       raise(Invalid_ocaml_type "unresolved recursive type")
	   | Applied _ ->
	       raise(Invalid_ocaml_type "fixed type constructor application")
	)



(*****************************************************************************
 *
 * type checking
 * 
 *****************************************************************************)

  (* construct a hash table with physical equality *)
module Obj_t_hash = struct
  type t = Obj.t
  let equal = (==)
    (* default hash function performs purely on my tests. Increasing 
     * the "useful information" does help, but still puts thousends 
     * blocks into the same bucket. (Not sure whether those blocks 
     * are structurely equal, but I doubt it.) The hash function 
     * is therefore responsible for the decreasing performance on 
     * large inputs
     *)
  let hash x = Hashtbl.hash_param 30 500 x
end

module Eqhash = Hashtbl.Make(Obj_t_hash)


  (* state of the type checking functions *)
type type_check_state = {
  mutable max_tag : int;		(* next unique tag for types *)
  mutable max_obj_tag : int;		(* next unique tag for blocks *)
    (* hash type constructor applications *)
  constr_hash : (string * int list, ocaml_type) Hashtbl.t;
    (* hash visited blocks *)
  val_hash : (type_tag * block_tag) Eqhash.t;
    (* verbosity info extracted from the flags passed to check *)
  verbose_blocks : bool;
  verbose_types : bool;
  verbose_stat : bool;
  mutable time : float;
  mutable spinner : int;
  verbose_spin : bool;
  verbose_type_ids : bool;
  verbose_trace : bool;
    (* the stack of message being printed as trace on error *)
  trace : string Stack.t;
  start_indentation : int;
  mutable current_indentation : int;
  oc : out_channel
}
  

    (* read an Obj.t into a nativeint *)
let nativeint_of_value value =
  Nativeint.logor
    (Nativeint.shift_left (Nativeint.of_int (Obj.magic value : int)) 1)
    (if Obj.is_int value then Nativeint.one else Nativeint.zero)


    (* indentation related functions *)
let get_indent state = String.make state.current_indentation ' '

let indent state = state.current_indentation <- state.current_indentation + 2

let unindent state =
  state.current_indentation <- state.current_indentation -2;
  assert(state.current_indentation >= 0)


   (* message printing *)

    (* it is really: fpf state printf_format arguments ... 
     * output an indented message 
     *)
let fpf state = 
  output_string state.oc (get_indent state);
  Printf.fprintf state.oc

    (* Do a trace message: either directly output the message or 
     * add it to the trace stack. Increase indentation level.
     *)
let trace_ind state =
  Printf.ksprintf
    (fun msg ->
       let msg = (get_indent state) ^ msg
       in
	 if state.verbose_blocks then begin
	   output_string state.oc msg;
	   flush state.oc;
	   indent state;
	 end else if state.verbose_trace then begin
	   Stack.push msg state.trace;
	   indent state;
	 end
	 else ()
    )

    (* output the trace stack *)
let print_trace state =
  let trace = ref []
  in
    output_string state.oc "Recorded trace:\n";
    Stack.iter (fun s -> trace := s :: !trace) state.trace;
    List.iter (fun s -> output_string state.oc s) !trace

    (* Output indented failure indication, possibly prefixed with a 
     * trace.
     *)
let failure state =
  Printf.ksprintf
    (fun msg ->
       let msg = (get_indent state) ^ msg
       in
	 if state.verbose_blocks then begin
	   output_string state.oc msg;
	   flush state.oc;
	 end else if state.verbose_trace then begin
	   print_trace state;
	   output_string state.oc msg;
	   flush state.oc;
	 end
	 else ()
    )


    (* depending on ok, give a ok message or do a failure *)
let trace_or_failure state msg ok =
  if ok then begin
    if state.verbose_blocks then fpf state "%s OK\n%!" msg
  end
  else
    failure state "%s FAIL\n" msg

    (* Decrease indentation and pop the trace stack. *)
let trace_unindent state =
  if state.verbose_blocks then 
    unindent state
  else if state.verbose_trace then begin
    ignore(Stack.pop state.trace);
    unindent state;
  end 
  else ()  


    (* Output a be alive message. *)
let do_spin state =
  state.spinner <- state.spinner +1;
  if (state.spinner mod 100_000) = 0 then 
    let old_time = state.time in
    let new_time = Unix.gettimeofday()
    in
      Printf.fprintf state.oc "%s%10d values checked, time diff %g\n%!"
	(String.make state.start_indentation ' ')
	state.spinner
	(new_time -. old_time);
      state.time <- new_time


(* type checking utility functions *)

    (* extract tag from type *)
let get_type_id = function
  | Int_type type_id
  | Float_type type_id
  | String_type type_id
  | Int32_type type_id
  | Nativeint_type type_id
  | Array_type(type_id, _)
  | Tuple_type(type_id, _)
  | Static_variant(type_id, _, _)
  | Record_type(type_id, _)
      -> type_id

  | Type_parameter _
  | Type_constructor_use _
      -> assert false


    (* tag_type taggs a type expression with the given tag,
     * this should be uniquely identify this type.
     *)
let tag_type new_tag typ = 
  assert(get_type_id typ == invalid_type_tag);
  match typ with
    | Int_type _ -> Int_type new_tag
    | Float_type _ -> Float_type new_tag
    | String_type _ -> String_type new_tag
    | Int32_type _ -> Int32_type new_tag
    | Nativeint_type _ -> Nativeint_type new_tag
    | Array_type(_, contents) -> Array_type(new_tag, contents)
    | Tuple_type(_, tuples) -> Tuple_type(new_tag, tuples)
    | Static_variant(_, consts, vars) -> Static_variant(new_tag, consts, vars)
    | Record_type(_, fields) -> Record_type(new_tag, fields)

    | Type_parameter _
    | Type_constructor_use _ 
      -> assert false


    (* custom pointers *)
let int32_custom_prt = Obj.field (Obj.repr 0l) 0
let nativeint_custom_prt = Obj.field (Obj.repr 0n) 0


    (* check that a type expression is in head normal form *)
let is_head_normal typ =
  match typ with
    | Int_type _
    | Float_type _
    | String_type _
    | Int32_type _
    | Nativeint_type _
    | Array_type _
    | Tuple_type _
    | Static_variant _
    | Record_type _
	-> true

    | Type_constructor_use _ -> false

    | Type_parameter _ ->
	raise(Invalid_ocaml_type "free type parameter")


    (* expand type constructor application until we get something concrete *)
let rec make_head_normal state typ =
  match typ with
    | Int_type _
    | Float_type _
    | String_type _
    | Int32_type _
    | Nativeint_type _
    | Array_type _
    | Tuple_type _
    | Static_variant _
    | Record_type _
	-> typ

    | Type_constructor_use constr_ref ->
	(match !constr_ref with
	   | Applied type_expr ->
	       assert(is_head_normal type_expr);
	       type_expr

	   | Resolved_application(constr_def, arguments) ->
	       let res = 
		 do_type_application state constr_ref arguments constr_def
	       in
		 assert(is_head_normal res);
		 res

	   | Recursive_application _ ->
	       raise(Invalid_ocaml_type "unresolved recursive type")
	)

    | Type_parameter _ ->
	raise(Invalid_ocaml_type "free type parameter")


  (* Perform a type constructor application inside a type expression.
   * If we've seen this constructor with the same arguments already
   * then get the resulting type from the constr_hash. The reason for this 
   * is not efficiency but to ensure a unique tagging of type expressions. 
   * If we haven't seen it, perform a substitution, make the result 
   * head normal, give it a new tag and remember it in the hash. And 
   * not to forget, the type expression in which this constructor application 
   * sits in, is updated in place, this time purely for efficiency.
   *)
and do_type_application state constr_ref arguments type_def = 
  if state.verbose_spin then do_spin state;
  if state.verbose_types then begin
    fpf state "perform type application\n%!";
    indent state;
  end;
  let new_type =
    match type_def with
      | Type_constructor_def(constr_id, arity, constr_expr) ->
	  if arity <> List.length arguments then begin
	    if state.verbose_types then
	      fpf state 
		"Arity error. \
                 Constructor takes %d arguments and %d provided\n%!"
		arity (List.length arguments);
	    raise(Invalid_ocaml_type "constructor application mismatch")
	  end;

	  if state.verbose_types then begin
	    fpf state "normalize arguments\n%!";
	    indent state;
	  end;
      
	  let norm_args = 
	    List.map (make_head_normal state) arguments in
	  let _ = if state.verbose_types then unindent state in
	  let arg_ids = List.map get_type_id norm_args in
	  let hash_key = (constr_id, arg_ids)
	  in
	    try
	      let res_type = Hashtbl.find state.constr_hash hash_key
	      in
		if state.verbose_types then begin
		  fpf state "Known type application (%s).\n" constr_id;
		  fpf state "Retrieve type %d from cache\n%!"
		    (get_type_id res_type);
		end;
		constr_ref := Applied res_type;
		res_type
	    with
	      | Not_found ->
		  let _ = 
		    if state.verbose_types then 
		      fpf state 
			"New type application (%s)\n%sSubstitute arguments\n%!"
			constr_id
			(get_indent state);
		  in
		  let subst_args = subst_type norm_args constr_expr in
		  let _ =
		    if state.verbose_types then begin
		      fpf state "Normalize type expression\n%!";
		      indent state
		    end
		  in
		    (* For type equations subst_args starts with a 
		     * type constructor application. In such a case we 
		     * need do the application but don't have to tag the 
		     * result (done already by the inner application).
		     * Otherwise we have to tag the type itself.
		     *)
		  let (debug_intro, type_tag, res_type) = 
		    if is_head_normal subst_args then
		      let new_tag = state.max_tag in
		      let _ = state.max_tag <- state.max_tag + 1 
		      in
			("new type", new_tag, tag_type new_tag subst_args)
		    else
		      let norm_type = make_head_normal state subst_args in
		      let type_tag = get_type_id norm_type
		      in
			assert(type_tag <> invalid_type_tag);
			("type eq ", type_tag, norm_type)
		  in
		    if state.verbose_types then begin
		      unindent state;
		      output_string state.oc (get_indent state);
		    end else
		      output_string state.oc 
			(String.make state.start_indentation ' ');
		    if state.verbose_types || state.verbose_type_ids then begin
		      Printf.fprintf state.oc 
			"%s %d = %s applied to (%s)\n%!" 
			debug_intro
			type_tag constr_id
			(String.concat ", " 
			   (List.map string_of_int arg_ids));
		    end;
		    Hashtbl.add state.constr_hash hash_key res_type;
		    constr_ref := Applied res_type;
		    res_type
  in
    if state.verbose_types then unindent state;
    new_type



(* utility functions used together with check_block *)

let identity x = x

let array_field_name i _ =
  Printf.sprintf "element %d" i

let record_field_name _ field_list = fst(List.hd field_list)

let record_field_type field_list = snd(List.hd field_list)


    (* Check that the fields of the block at value have types 
     * according to element_types. The size of the block has been 
     * checked before, here I assume 
     *   List.length element_types == Obj.size value
     * The obvious loop is encoded in a recursion over an increasing index.
     * In principle check_block is recursive with internal_check and 
     * internal_check_type. However, I want to use it in a polymorphic way,
     * therefore I have to take it out of the recursion, providing 
     * the recursively called internal_check as argument.
     *)
let rec check_block internal_check 
    state value size element_types current_type next_iteration field_name index 
    =
  if index = size then true
  else begin
    if state.verbose_blocks || state.verbose_trace then 
      trace_ind state "%s\n%!" (field_name index element_types);
    let field_res = 
      internal_check state (Obj.field value index) (current_type element_types)
    in
      if state.verbose_blocks || state.verbose_trace then 
	trace_unindent state;
      if field_res then
	check_block internal_check state value size
	  (next_iteration element_types) 
	  current_type next_iteration field_name (index +1)
      else
	false
  end



    (* The main work is done by internal_check and internal_check_type. 
     * The first one normalizes type constructor and checks for 
     * shared blocks. It calls internal_check_type to do the real type 
     * checking.
     *)
let rec internal_check state value = function
  | Type_constructor_use constr_ref ->
      (match !constr_ref with
	 | Applied type_expr ->
	     if state.verbose_types then
	       fpf state "going through optimized type constructor\n%!";
	     internal_check state value type_expr

	 | Resolved_application(constr_def, arguments) ->
	     internal_check state value 
	       (do_type_application state constr_ref arguments constr_def)
	       
	 | Recursive_application _ -> 
	     raise(Invalid_ocaml_type "unresolved recursive typeconstructor")
      )

  | Type_parameter _ ->
      raise(Invalid_ocaml_type "free type parameter")


    (* extract the type_tag now and check whether we've seen 
     * this node already
     *)
  | (   Int_type type_id
      | Float_type type_id
      | String_type type_id
      | Int32_type type_id
      | Nativeint_type type_id
      | Array_type(type_id, _)
      | Tuple_type(type_id, _)
      | Static_variant(type_id, _, _)
      | Record_type(type_id, _) ) as type_expr
    ->
      (* let cur_ind = state.current_indentation in *)
      if state.verbose_spin then do_spin state;
	(* let res = *)
      if Obj.is_block value 
      then
	try 
	  let (o_type_id, block_id) = Eqhash.find state.val_hash value
	  in
	    if o_type_id == type_id
	    then begin
	      (* seen this block with the same type *)
	      if state.verbose_blocks then
		fpf state 
		  "visited value %d already with the same type %d; OK\n%!"
		  block_id type_id;
	      true
	    end
	    else begin
	      (* seen this block but with a different type *)
	      if state.verbose_blocks || state.verbose_trace then
		failure state 
		  "type confilict on value %d: now %d and previously %d; FAIL\n"
		  block_id type_id o_type_id;
	      false
	  end
	with
	   | Not_found -> 
	       (* not seen this block, register it and check its type *)
	       let block_id = state.max_obj_tag
	       in
		 state.max_obj_tag <- state.max_obj_tag + 1;
	       if state.verbose_blocks then
		 fpf state "visit value %d for the first time with type %d\n%!" 
		   block_id type_id;
	       Eqhash.add state.val_hash value (type_id, block_id);
	       internal_check_type state value type_expr
      else 
	(* not a block, have to check its type *)
	internal_check_type state value type_expr
	(* 
         * in
	 *   if cur_ind <> state.current_indentation then
	 *     Printf.fprintf state.oc "indentation mismatch %d <> %d\n%!"
	 *       cur_ind state.current_indentation;
	 *   res
         *)


    (* Check the value against its supposed type. Call internal_check
     * recursively when needed.
     *)
and internal_check_type state value = function
  | Int_type _ ->
      let res = Obj.is_int value
      in
	if state.verbose_blocks || state.verbose_trace then
	  trace_or_failure state "int" res;
	res

  | Float_type _ -> 
      let res = 
	Obj.is_block value &&
	  Obj.tag value == Obj.double_tag &&
	  (* 64 bit !! *)
	  Obj.size value == 2
      in
	if state.verbose_blocks || state.verbose_trace then
	  trace_or_failure state "float %s\n%!" res;
	res
  
  | String_type _ ->
      let res =
	Obj.is_block value &&
	  Obj.tag value == Obj.string_tag &&
	  (* 64 bit !! *)
	  (let last_word = 
	     nativeint_of_value (Obj.field value (Obj.size value -1))
	   in
	     match Nativeint.logand last_word 0xFF000000n 
	     with
	       | 0x0n -> true
	       | 0x01000000n -> Nativeint.logand last_word 0x00FF0000n = 0n
	       | 0x02000000n -> Nativeint.logand last_word 0x00FFFF00n = 0n
	       | 0x03000000n -> Nativeint.logand last_word 0x00FFFFFFn = 0n
	       | _ -> false
	  )
      in
	if state.verbose_blocks || state.verbose_trace then
	  trace_or_failure state "string %s\n%!" res;
	res

  | Int32_type _ ->
      let res = 
	Obj.is_block value &&
	  Obj.tag value == Obj.custom_tag &&
	  Obj.field value 0 == int32_custom_prt
      in
	if state.verbose_blocks || state.verbose_trace then
	  trace_or_failure state "int32 %s\n%!" res;
	res

  | Nativeint_type _ ->
      let res = 
	Obj.is_block value &&
	  Obj.tag value == Obj.custom_tag &&
	  Obj.field value 0 == nativeint_custom_prt
      in
	if state.verbose_blocks || state.verbose_trace then
	  trace_or_failure state "nativeint %s\n%!" res;
	res


  | Array_type(_, field_type) ->
      let top_res = 
	Obj.is_block value && 
	  Obj.tag value == 0 &&
	  Obj.size value >= 0
      in
	if top_res then begin
	  if state.verbose_blocks || state.verbose_trace then 
	    trace_ind state
	      "Array of size %d top level OK\n%!" (Obj.size value);
	  let res = 
	    check_block internal_check state value (Obj.size value) 
	      field_type identity identity array_field_name 0
	  in
	    if state.verbose_blocks || state.verbose_trace then 
	      trace_unindent state;
	    res
	end
	else begin
	  if state.verbose_blocks then
	    failure state "Array FAIL\n%!";
	  false
	end
	  

  | Tuple_type(_, element_types) ->
      let top_res = 
	Obj.is_block value &&
	  Obj.tag value == 0 &&
	  (Obj.size value) == (List.length element_types)
      in
	if top_res then begin
	  if state.verbose_blocks || state.verbose_trace then 
	    trace_ind state "Tuple %d top level OK\n%!" (Obj.size value);
	  let res = 
	    check_block internal_check state value 
	      (Obj.size value) element_types List.hd List.tl array_field_name 0
	  in
	    if state.verbose_blocks || state.verbose_trace then
	      trace_unindent state;
	    res
	end
	else begin
	  if state.verbose_blocks then
	    failure state "Tuple FAIL\n%!";
	  false
	end

  | Static_variant(_, consts, vars) ->
      if Obj.is_int value
      then begin
	let i = (Obj.magic value : int)
	in
	  if i < List.length consts then begin
	    if state.verbose_blocks then
	      fpf state "const variant %s OK\n%!" (List.nth consts i);
	    true
	  end
	  else begin
	    if state.verbose_blocks || state.verbose_trace then
	      failure state "invalid const variant %d FAIL\n%!" i;
	    false
	  end
      end
      else  (* value is a block *)
	let tag = Obj.tag value in
	let size = Obj.size value
	in
	  if tag < List.length vars then 
	    let (constr_name, field_types) = List.nth vars tag
	    in
	      if size == List.length field_types then begin
		if state.verbose_blocks || state.verbose_trace then 
		  trace_ind state "variant %s top level OK\n%!" constr_name;
		let res = 
		  check_block internal_check state value
		    size field_types List.hd List.tl array_field_name 0
		in
		  if state.verbose_blocks || state.verbose_trace then
		    trace_unindent state;
		  res
	      end  
	      else begin  (* size wrong *)
		if state.verbose_blocks || state.verbose_trace then 
		  failure state "block size wrong (%d) for variant %s; FAIL\n%!"
		    size constr_name;
		false
	      end
	  else begin  (* tag too big *)
	    if state.verbose_blocks || state.verbose_trace then
	      failure state "inapropriate variable constructor tag; FAIL\n%!";
	    false
	  end

  | Record_type(_, fields) ->
      let top_res = 
	Obj.is_block value &&
	  Obj.tag value == 0 &&
	  (Obj.size value) == (List.length fields)
      in
	if top_res then begin
	  if state.verbose_blocks || state.verbose_trace then
	    trace_ind state "record top level OK\n%!";
	  let res =
	    check_block internal_check state value (Obj.size value) fields 
	      record_field_type List.tl record_field_name 0
	  in
	    if state.verbose_blocks || state.verbose_trace then
	      trace_unindent state;
	    res
	end
	else begin
	  if state.verbose_blocks || state.verbose_trace then 
	    failure state "record top level FAIL\n%!";
	  false
	end

  | Type_parameter _ ->
      raise(Invalid_ocaml_type "free type parameter")

  | Type_constructor_use _ -> assert false



(*****************************************************************************
 *
 * user interface
 * 
 ****************************************************************************)

     (* see mli for doc *)
type check_flags =
  | Channel of out_channel
  | Verbose_blocks
  | Verbose_types
  | Verbose_statistics
  | Verbose_spinner
  | Verbose_type_ids
  | Verbose_trace
  | Start_indent of int

    (* Parse/record flags. This sucks, need to rewrite it some day. *)
let create_state flags =
  let outc = ref None in
  let vblocks = ref false in
  let vtypeid = ref false in
  let vtypes = ref false in
  let vstat = ref false in
  let vspin = ref false in
  let vtrace = ref false in
  let vind = ref 0 
  in
    List.iter
      (function
	 | Channel o -> outc := Some o
	 | Verbose_blocks -> vblocks := true
	 | Verbose_type_ids -> vtypeid := true
	 | Verbose_types -> vtypes := true
	 | Verbose_statistics -> vstat := true
	 | Verbose_spinner -> vspin := true
	 | Verbose_trace -> vtrace := true
	 | Start_indent i -> vind := i)
      flags;
    if !outc = None then begin
      vblocks := false;
      vtypeid := false;
      vtypes := false;
      vstat := false;
      vspin := false;
      vtrace := false;
    end;
    {
      max_tag = 0;
      max_obj_tag = 0;
      constr_hash = (Hashtbl.create 953);
      val_hash = (Eqhash.create 10061);
      verbose_blocks = !vblocks;
      verbose_types = !vtypes;
      verbose_stat = !vstat;
      time = Unix.gettimeofday();
      spinner = 0;
      verbose_spin = !vspin;
      verbose_type_ids = !vtypeid;
      verbose_trace = !vtrace;
      trace = Stack.create ();
      start_indentation = !vind;
      current_indentation = !vind;
      oc = match !outc with
	| None -> stdout
	| Some c -> c;
    } 
    

    (* Interface function: Check if value has type type_expr.
     * Create the necessary hashes. Perform an empty substitution 
     * on the type_expr in order to check for stray type parameters
     * and to create a private copy of the type expression such that 
     * updateing reference cells in it does not infect the outside program.
     *)
let check flags value type_expr =
  let state = create_state flags in
  let _ = 
      if state.verbose_types then begin
	fpf state "Enter typecheck, copy type expression\n%!";
	indent state;
      end
  in let my_type_expr = subst_type [] type_expr
  in let _ = 
      if state.verbose_types then
	unindent state;
      if state.verbose_blocks then begin
	assert(
	  if state.current_indentation = state.start_indentation 
	  then
	    true
	  else begin
	    Printf.fprintf state.oc "current ind %d <> start ind %d\n%!"
	      state.current_indentation state.start_indentation;
	    false
	  end
	);
	fpf state "Enter internal check\n%!";
	indent state;
      end
  in let res = internal_check state (Obj.repr value) my_type_expr
  in
    if state.verbose_blocks then begin
      unindent state;
      assert(
	if state.current_indentation = state.start_indentation 
	then
	  true
	else begin
	  Printf.fprintf state.oc "current ind %d <> start ind %d\n%!"
	    state.current_indentation state.start_indentation;
	  false
	end
      );
    end;
    if state.verbose_stat then begin
      fpf state "%s. Check stat: %d types, %d blocks, %d type appl.\n%!"
	(if res then "OK" else "FAIL")
	state.max_tag
	(Eqhash.length state.val_hash)
	(Hashtbl.length state.constr_hash);
      (* 
       * fpf state "constr hash stat %!";
       * Hashtbl.print_stat state.constr_hash;
       * fpf state "val hash stat %!";
       * Eqhash.print_stat state.val_hash;
       *)
    end;
    res
