(* 
 * 
 *               Camlp4 quotations in original syntax
 * 
 *                 Copyright (C) 2005  Hendrik Tews
 * 
 *   This library is free software; you can redistribute it and/or
 *   modify it under the terms of the GNU Library General Public
 *   License as published by the Free Software Foundation; either
 *   version 2 of the License, or (at your option) any later version.
 * 
 *   This library is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *   Library General Public License in the file LICENCE in this or one
 *   of the parent directories for more details.
 * 
 *   Time-stamp: <Thursday 20 July 06 0:06:02 tews@debian>
 * 
 *   $Id: qo_MLast.ml,v 1.29 2006/07/19 22:25:06 tews Exp $
 * 
 *)


value ocamlp4_version = "0.3";

(****************************************************************************
 *      BEGIN RELOC SECTION
 *
 *      the following is copied from camlp4/camlp4/reloc.ml 
 *      to make this module independent from non-installed libraries
 *
 *)

value shift_pos n p =
   { (p) with Lexing.pos_cnum = p.Lexing.pos_cnum + n }
;

value adjust_pos globpos local_pos =
{
  Lexing.pos_fname = globpos.Lexing.pos_fname;
  Lexing.pos_lnum = globpos.Lexing.pos_lnum + local_pos.Lexing.pos_lnum - 1;
  Lexing.pos_bol = 
      if local_pos.Lexing.pos_lnum <= 1 then
        globpos.Lexing.pos_bol
      else
        local_pos.Lexing.pos_bol + globpos.Lexing.pos_cnum;
  Lexing.pos_cnum = local_pos.Lexing.pos_cnum + globpos.Lexing.pos_cnum
};

value adjust_loc gpos (p1, p2) =
   (adjust_pos gpos p1, adjust_pos gpos p2)
;

(*
 *      End reloc section
 ****************************************************************************)


value (gram, q_position) =
  let (lexer,pos) = Plexer.make_lexer () in
  (Grammar.gcreate lexer, pos)
;


module Qast =
struct
  type t =
      [ Node of string and list t
	| List of list t
	| Tuple of list t
	| Option of option t
	| Int of string
	| Str of string
	| Bool of bool
	| Cons of t and t
	| Apply of string and list t
	| Record of list (string * t)
	| Loc
	| Antiquot of MLast.loc and string ]
  ;
    value _loc =
	let nowhere =
	  {(Lexing.dummy_pos) with Lexing.pos_lnum = 1; Lexing.pos_cnum = 0 } in
	  (nowhere,nowhere);

    value rec to_expr =
      fun
      [ Node n al ->
	  List.fold_left (fun e a -> <:expr< $e$ $to_expr a$ >>)
	    <:expr< MLast.$uid:n$ >> al
      | List al ->
	  List.fold_right (fun a e -> <:expr< [$to_expr a$ :: $e$] >>) al
	    <:expr< [] >>
      | Tuple al -> <:expr< ($list:List.map to_expr al$) >>
      | Option None -> <:expr< None >>
      | Option (Some a) -> <:expr< Some $to_expr a$ >>
      | Int s -> <:expr< $int:s$ >>
      | Str s -> <:expr< $str:s$ >>
      | Bool True -> <:expr< True >>
      | Bool False -> <:expr< False >>
      | Cons a1 a2 -> <:expr< [$to_expr a1$ :: $to_expr a2$] >>
      | Apply f al ->
	  List.fold_left (fun e a -> <:expr< $e$ $to_expr a$ >>)
	    <:expr< $lid:f$ >> al
      | Record lal -> <:expr< {$list:List.map to_expr_label lal$} >>
      | Loc -> <:expr< $lid:Stdpp.loc_name.val$ >>
      | Antiquot loc s ->
          let (bolpos,lnum, _) = Pcaml.position.val in
          let (bolposv,lnumv) = (bolpos.val, lnum.val) in
          let zero_pos () = do { bolpos.val := 0; lnum.val := 1 } in
          let restore_pos () = do { bolpos.val := bolposv; lnum.val := lnumv } in
          let e =
            try
              let _ = zero_pos() in
              let result = Grammar.Entry.parse Pcaml.expr_eoi (Stream.of_string s) in
              let _ = restore_pos() in
              result
            with
            [ Stdpp.Exc_located (bp, ep) exc ->
                do { restore_pos() ; raise (Stdpp.Exc_located (adjust_loc (fst loc) (bp,ep)) exc) }
            | exc -> do { restore_pos(); raise exc } ]
          in
          <:expr< $anti:e$ >> ]
    and to_expr_label (l, a) = (<:patt< MLast.$lid:l$ >>, to_expr a);
    value rec to_patt =
      fun
      [ Node n al ->
	  List.fold_left (fun e a -> <:patt< $e$ $to_patt a$ >>)
	    <:patt< MLast.$uid:n$ >> al
      | List al ->
	  List.fold_right (fun a p -> <:patt< [$to_patt a$ :: $p$] >>) al
	    <:patt< [] >>
      | Tuple al -> <:patt< ($list:List.map to_patt al$) >>
      | Option None -> <:patt< None >>
      | Option (Some a) -> <:patt< Some $to_patt a$ >>
      | Int s -> <:patt< $int:s$ >>
      | Str s -> <:patt< $str:s$ >>
      | Bool True -> <:patt< True >>
      | Bool False -> <:patt< False >>
      | Cons a1 a2 -> <:patt< [$to_patt a1$ :: $to_patt a2$] >>
      | Apply _ _ -> failwith "bad pattern"
      | Record lal -> <:patt< {$list:List.map to_patt_label lal$} >>
      | Loc -> <:patt< _ >>
      | Antiquot loc s ->
          let (bolpos,lnum, _) = Pcaml.position.val in
          let (bolposv,lnumv) = (bolpos.val, lnum.val) in
          let zero_pos () = do { bolpos.val := 0; lnum.val := 1 } in
          let restore_pos () = do { bolpos.val := bolposv; lnum.val := lnumv } in
          let p =
            try
              let _ = zero_pos() in
              let result = Grammar.Entry.parse Pcaml.patt_eoi (Stream.of_string s) in
              let _ = restore_pos() in
              result
             with
            [ Stdpp.Exc_located (bp, ep) exc ->
                do { restore_pos() ; raise (Stdpp.Exc_located (adjust_loc (fst loc) (bp, ep)) exc) }
            | exc -> do { restore_pos(); raise exc } ]
          in
          <:patt< $anti:p$ >> ]
    and to_patt_label (l, a) = (<:patt< MLast.$lid:l$ >>, to_patt a);

end
;


value antiquot k (bp, ep) x =
  let shift =
    if k = "" then String.length "$"
    else String.length "$" + String.length k + String.length ":"
  in
  Qast.Antiquot (shift_pos shift bp, shift_pos (-1) ep) x
;

value sig_item = Grammar.Entry.create gram "signature item";
value str_item = Grammar.Entry.create gram "structure item";
value ctyp = Grammar.Entry.create gram "type";
value poly_type = Grammar.Entry.create gram "poly type";
value patt = Grammar.Entry.create gram "pattern";
value expr = Grammar.Entry.create gram "expression";

value module_type = Grammar.Entry.create gram "module type";
value module_expr = Grammar.Entry.create gram "module expression";
value class_type = Grammar.Entry.create gram "class type";
value class_expr = Grammar.Entry.create gram "class expr";
value class_sig_item = Grammar.Entry.create gram "class signature item";
value class_str_item = Grammar.Entry.create gram "class structure item";
value let_binding = Grammar.Entry.create gram "let_binding";
value type_declaration = Grammar.Entry.create gram "type_declaration";
value type_kind = Grammar.Entry.create gram "type_kind";
value with_constr = Grammar.Entry.create gram "with_constr";
value a_seq = Grammar.Entry.create gram "a_seq";
value a_tup = Grammar.Entry.create gram "a_tup";
value a_list = Grammar.Entry.create gram "a_list";
value a_opt = Grammar.Entry.create gram "a_opt";
value a_UIDENT = Grammar.Entry.create gram "a_UIDENT";
value a_LIDENT = Grammar.Entry.create gram "a_LIDENT";
value a_INT = Grammar.Entry.create gram "a_INT";
value a_INT32 = Grammar.Entry.create gram "a_INT32";
value a_INT64 = Grammar.Entry.create gram "a_INT64";
value a_NATIVEINT = Grammar.Entry.create gram "a__NATIVEINT";
value a_FLOAT = Grammar.Entry.create gram "a_FLOAT";
value a_STRING = Grammar.Entry.create gram "a_STRING";
value a_CHAR = Grammar.Entry.create gram "a_CHAR";


(* 
 * converts optional keywords like rec, mutable, and virtual 
 * into booleans (present or not)
 * 
 * pa_o loc 53
 *)

value o2b =
  fun
  [ Qast.Option (Some _) -> Qast.Bool True
  | Qast.Option None -> Qast.Bool False
  | x -> x ]
;




(* 
 * Negate a number in string presentation. 
 * Because it treats negative numbers correctly you can have 
 * pattern like <:expr<--5>>, can't you?
 *)
value neg_string n =
  let len = String.length n in
  if len > 0 && n.[0] = '-' then String.sub n 1 (len - 1)
  else "-" ^ n
;


(* 
 * mkumin transforms "-5" into an integer node, rather then 
 * leaving it an application of "-" to "5". 
 * 
 * pa_o loc 59
 *)

value mkumin _ f arg =
  match arg with
  [ Qast.Node (("ExInt" | "ExInt32" | "ExInt64" | "ExNativeInt") as exi)
      [Qast.Loc; Qast.Str n] when int_of_string n > 0 ->
        let n = neg_string n in
        Qast.Node exi [Qast.Loc; Qast.Str n]
  | Qast.Node "ExFlo" [Qast.Loc; Qast.Str n] when float_of_string n > 0.0 ->
      let n = neg_string n in
      Qast.Node "ExFlo" [Qast.Loc; Qast.Str n]
  | _ ->
      match f with
      [ Qast.Str f ->
          let f = "~" ^ f in
          Qast.Node "ExApp"
            [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str f]; arg]
      | _ -> assert False ] ]
;

value mkuminpat _ f is_int s =
  let s =
    match s with
    [ Qast.Str s -> Qast.Str (neg_string s)
    | s -> failwith "bad unary minus" ]
  in
  match is_int with
  [ Qast.Bool True -> Qast.Node "PaInt" [Qast.Loc; s]
  | Qast.Bool False -> Qast.Node "PaFlo" [Qast.Loc; s]
  | _ -> assert False ]
; 


(* 
 * mklistexp translates a list of elements e1; e2; ... ; en  
 * into the list e1 :: e2 :: ... :: en :: []
 * 
 * the original mklistexp that is in pa_o, pa_r, and q_Mlast takes 
 * three arguments: 
 *  - the location, 
 *  - last -- the tail list (actually a tail list option), and 
 *  - a list of elements
 * In the pa_o, pa_r versions the location is used and adopted in 
 * the recursion. In q_Mlast the location is ignored.
 * In the revised syntax one has [ e1; ... ; tail], where tail is a list, 
 * this tail comes in as last argument. Obviously, in the original syntax 
 * the last arg is alway None.
 * 
 * Here we need a version of mklistexp that ignores the location and never 
 * gets a tail list. I take the liberty to hard-code this.
 *
 * pa_o loc 79
 *)

value rec mklistexp = fun
    [ Qast.List [] -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "[]"]
    | Qast.List [e1 :: el] ->
        Qast.Node "ExApp"
          [Qast.Loc;
           Qast.Node "ExApp"
             [Qast.Loc; Qast.Node "ExUid" [Qast.Loc; Qast.Str "::"]; e1];
           mklistexp (Qast.List el)]

    (* this should never happen (I believe)
     * the original code does nothing, ie a -> a,
     * however I prefer to fail. Let's see, if this gets ever triggered!
     *)
    | a -> do {match a with
		   [ Qast.Node _ _ -> prerr_endline "Node"
		   | Qast.List _ -> prerr_endline "List"
		   | Qast.Tuple _ -> prerr_endline "Tuple"
		   | Qast.Option _ -> prerr_endline "Option"
		   | Qast.Int _ -> prerr_endline "Int"
		   | Qast.Str _ -> prerr_endline "Str"
		   | Qast.Bool _ -> prerr_endline "Bool"
		   | Qast.Cons _ _ -> prerr_endline "Cons"
		   | Qast.Apply _ _ -> prerr_endline "Apply"
		   | Qast.Record _ -> prerr_endline "Record"
		   | Qast.Loc -> prerr_endline "Loc"
		   | Qast.Antiquot _ _ -> prerr_endline "Antiquot"
		   ];
		 flush stderr;
		 assert False}
    ]
;


(* 
 * mklistpat is similar to mklistexp, only for patterns. 
 * It translates a Qast list to the appropriate "::" applications.
 * The version here is simpler than the ones in pa_o or q_MLast.
 * 
 * pa_o loc 91
 *)

value rec mklistpat = fun
  [ Qast.List [] -> Qast.Node "PaUid" [Qast.Loc; Qast.Str "[]"]
  | Qast.List [p1 :: pl] ->
      Qast.Node "PaApp"
        [Qast.Loc;
         Qast.Node "PaApp"
           [Qast.Loc; Qast.Node "PaUid" [Qast.Loc; Qast.Str "::"]; p1];
           mklistpat (Qast.List pl)]
    (* fail here *)
  | a -> assert False ]
;

(* 
 * rewrites a right associative expr_ident like A.(B.(C.d))
 * into a left associative form, ie ((A.B).C).d
 *)
value mkexprident _loc ids = match ids with
  [ [] -> Stdpp.raise_with_loc _loc (Stream.Error "illegal long identifier")
  | [ id :: ids ] ->
      let rec loop m = fun
        [ [ id :: ids ] -> loop 
	    (* <:expr< $m$ . $id$ >>  *)
	    (Qast.Node "ExAcc" [Qast.Loc; m; id])
	    ids
        | [] -> m ]
      in
	loop id ids ]
;


(* 
 * built a special assert(false) node, if e is false,
 * otherwise a usual assert node
 *)
value mkassert e =
  match e with
  [ Qast.Node "ExUid" [_; Qast.Str "False"] -> Qast.Node "ExAsf" [Qast.Loc]
  | _ -> Qast.Node "ExAsr" [Qast.Loc; e] ]
;



(* 
 * val mkcurriedconstr : string -> Qast.t -> Qast.t -> Qast.t
 * 
 * The last argument args is considered a list encoded as Qast.t 
 * (with the List and Cons constructors). 
 *   mkcurriedconstr astconstr expr args 
 * applies each element of this args list to 
 * expr with the help of astconstr, 
 * building a curriefied application of astconstr nodes.
 * 
 * Keep in mind that the final nil in the list of elements in a 
 * tuple quotation is encoded as Qast.List _ [], 
 * see pa_expand fun quote_expr
 * 
 *)

value rec mkcurriedconstr constr e = fun 
  [ Qast.List args -> 
      List.fold_left (fun e1 e2 -> Qast.Node constr [Qast.Loc; e1; e2]) e args
  | Qast.Cons hd tl ->
      mkcurriedconstr constr (Qast.Node constr [Qast.Loc; e; hd]) tl
  | _ -> assert False]
;

(****************************************************************************
 *
 *      utility functions from pa_o
 *
 ****************************************************************************)

(* 
 * val is_operator : string -> bool
 * Recognizes an infix operator. Recognition is done via two hash tables,
 * the first with the ascii infix operators the second with the starting 
 * symbols of the symbolic operators.
 * 
 * pa_o loc 103
 *)

value is_operator =
  let ht = Hashtbl.create 73 in
  let ct = Hashtbl.create 73 in
  do {
    List.iter (fun x -> Hashtbl.add ht x True)
      ["asr"; "land"; "lor"; "lsl"; "lsr"; "lxor"; "mod"; "or"];
    List.iter (fun x -> Hashtbl.add ct x True)
      ['!'; '&'; '*'; '+'; '-'; '/'; ':'; '<'; '='; '>'; '@'; '^'; '|'; '~';
       '?'; '%'; '.'; '$'];
    fun x ->
      do {
	try Hashtbl.find ht x with
	    [ Not_found -> try Hashtbl.find ct x.[0] with _ -> False ]
      }}
;


(* 
 * parses two tokens: <infix op> ")"
 * If these tokens are recognized they are deleted from the stream and 
 * the operator is returned. Otherwise this parser fails (to try the 
 * next alternative)
 * 
 * pa_o loc 118
 *)
value operator_rparen =
  Grammar.Entry.of_parser gram "operator_rparen"
    (fun strm ->
       match Stream.npeek 2 strm with
       [ [("", s); ("", ")")] when is_operator s ->
           do { Stream.junk strm; Stream.junk strm; s }
       | _ -> raise Stream.Failure ]
    )
;


(* 
 * val symbolchar : string -> int -> bool
 * 
 * symbolchar s i  returns true if s consists only of symbols from position i
 * Used to recognize infix operators for which the start has been recognized 
 * already.
 * 
 * pa_o loc 136
 *)
value symbolchar =
  let list =
    ['!'; '$'; '%'; '&'; '*'; '+'; '-'; '.'; '/'; ':'; '<'; '='; '>'; '?';
     '@'; '^'; '|'; '~']
  in
  let rec loop s i =
    if i == String.length s then True
    else if List.mem s.[i] list then loop s (i + 1)
    else False
  in
  loop
;


(* 
 * recognizes and, if found, discards a prefix operator
 * 
 * pa_o loc 149
 *)
value prefixop =
  let list = ['!'; '?'; '~'] in
  let excl = ["!="; "??"] in
  Grammar.Entry.of_parser gram "prefixop"
    (parser
       [: `("", x)
           when
             not (List.mem x excl) && String.length x >= 2 &&
             List.mem x.[0] list && symbolchar x 1 :] ->
         x)
;


(* 
 * recognize and discard an infixes with least precedence
 * 
 * pa_o loc 161
 *)
value infixop0 =
  let list = ['='; '<'; '>'; '|'; '&'; '$'] in
  let excl = ["<-"; "||"; "&&"] in
  Grammar.Entry.of_parser gram "infixop0"
    (parser
       [: `("", x)
           when
             not (List.mem x excl) && String.length x >= 2 &&
             List.mem x.[0] list && symbolchar x 1 :] ->
         x)
;


(* 
 * recognize and discard concatenation infixes starting with @ or ^
 * 
 * pa_o loc 173
 *)
value infixop1 =
  let list = ['@'; '^'] in
  Grammar.Entry.of_parser gram "infixop1"
    (parser
       [: `("", x)
           when
             String.length x >= 2 && List.mem x.[0] list &&
             symbolchar x 1 :] ->
         x)
;


(* 
 * recognize and discard an additive infix operator starting with + or -
 * 
 * pa_o loc 184
 *)
value infixop2 =
  let list = ['+'; '-'] in
  Grammar.Entry.of_parser gram "infixop2"
    (parser
       [: `("", x)
           when
             x <> "->" && String.length x >= 2 && List.mem x.[0] list &&
             symbolchar x 1 :] ->
         x)
;

(* 
 * recognize and discard an multiplicative infix operator starting with *,/,%
 * 
 * pa_o loc 195
 *)
value infixop3 =
  let list = ['*'; '/'; '%'] in
  Grammar.Entry.of_parser gram "infixop3"
    (parser
       [: `("", x)
           when
             String.length x >= 2 && List.mem x.[0] list &&
             symbolchar x 1 :] ->
         x)
;

(* 
 * recognize and discard an exponential infix operator starting with **
 * 
 * pa_o loc 206
 *)
value infixop4 =
  Grammar.Entry.of_parser gram "infixop4"
    (parser
       [: `("", x)
           when
             String.length x >= 3 && x.[0] == '*' && x.[1] == '*' &&
             symbolchar x 2 :] ->
         x)
;


(* 
 * After we have parsed "type =" the function test_constr_decl 
 * checks (nondestructively) if a (traditional) variant type follows.
 *
 * permit list antiqotations for a type constructor list antiquotation
 * 
 * pa_o loc 216
 *)
value test_constr_decl =
  Grammar.Entry.of_parser gram "test_constr_decl"
    (fun strm ->
       match Stream.npeek 1 strm with
       [ [("UIDENT", _)] ->
           match Stream.npeek 2 strm with
           [ [_; ("", ".")] -> raise Stream.Failure
           | [_; ("", "(")] -> raise Stream.Failure
           | [_ :: _] -> ()
           | _ -> raise Stream.Failure ]
       | [("", "|")] -> ()
       | [("ANTIQUOT", s)] 
	   when (String.length s) > 5 && (String.sub s 0 5) = "list:" -> ()
       | _ -> raise Stream.Failure
    ])
;


(* 
 * return the n-th element of the stream, nondestructively
 * 
 * rewritten using List.nth
 * 
 * pa_o loc 230
 *)
value stream_peek_nth n strm =
      try
	let v = List.nth (Stream.npeek n strm) (n-1)
	in
          do{
	  (* Printf.fprintf stderr "peek %d: |%s|%s|\n" n (fst v) (snd v); *)
	  Some(v)
	  }
      with
	  [ Failure "nth" -> None ]
;


(* 
 * Check for class valued function types. Need to jump over arbitrary 
 * type expressions and matching parenthesis until there is a "->" token.
 * 
 * added jumping over antiquotations
 * 
 * pa_o loc 238
 *)
value test_ctyp_minusgreater =
  Grammar.Entry.of_parser gram "test_ctyp_minusgreater"
    (fun strm ->
       let rec skip_simple_ctyp n =
         match stream_peek_nth n strm with
         [ Some ("", "->") -> n
         | Some ("", "[" | "[<") ->
             skip_simple_ctyp (ignore_upto "]" (n + 1) + 1)
         | Some ("", "(") -> skip_simple_ctyp (ignore_upto ")" (n + 1) + 1)
         | Some
             ("",
              "as" | "'" | ":" | "*" | "." | "#" | "<" | ">" | ".." | ";" |
              "_") ->
             skip_simple_ctyp (n + 1)
         | Some ("QUESTIONIDENT" | "LIDENT" | "UIDENT" | "ANTIQUOT", _) ->
             skip_simple_ctyp (n + 1)
         | Some _ | None -> raise Stream.Failure ]
       and ignore_upto end_kwd n =
         match stream_peek_nth n strm with
         [ Some ("", prm) when prm = end_kwd -> n
         | Some ("", "[" | "[<") ->
             ignore_upto end_kwd (ignore_upto "]" (n + 1) + 1)
         | Some ("", "(") -> ignore_upto end_kwd (ignore_upto ")" (n + 1) + 1)
         | Some _ -> ignore_upto end_kwd (n + 1)
         | None -> raise Stream.Failure ]
       in
       match Stream.peek strm with
       [ Some (("", "[") | ("LIDENT" | "UIDENT" | "ANTIQUOT", _)) -> 
	   skip_simple_ctyp 1
       | Some ("", "object") -> raise Stream.Failure
       | _ -> 1 ])
;


(* 
 * look ahead magic: don't consume any tokens and
 *  - deliver unit if there is a record field name followed by an "=" 
 *  - otherwise trigger a parse error
 * thus this function can distinguish record update from record construction
 * 
 * changes:
 * - the original in pa_o uses "where rec", my brain preferes "let rec"
 * - accept antiquotations as field name to permit { $id$ = ... }
 * - accept { $list:...$ }
 *
 * pa_o loc 272
 *)
value test_label_eq =
    let rec test_for_eq lev strm =
	                     (* recursively peek the stream deeper and deeper *)
      match stream_peek_nth lev strm with
	  [ Some (("ANTIQUOT",_) | ("UIDENT", _) | 
		      ("LIDENT", _) | ("", ".")) ->
	      test_for_eq (lev + 1) strm
	  | Some ("", "=") -> ()
	  | _ -> raise Stream.Failure ]
    in
    let test strm =
      match (stream_peek_nth 1 strm, stream_peek_nth 2 strm) with
	[ (Some("ANTIQUOT",s), Some("","}")) 
	    when String.sub s 0 5 = "list:"-> ()
	| _ -> test_for_eq 1 strm]
    in				       (* construct a grammar entry from test *)
      Grammar.Entry.of_parser gram "test_label_eq" test
;


(* 
 * Look ahead to distinguish type expressions from polymorphic method and 
 * record label types ("'a 'b . ...").
 * 
 * permit list antiqotations for a type variable list antiquotation
 *
 * pa_o loc 282
 *)
value test_typevar_list_dot =
  Grammar.Entry.of_parser gram "test_typevar_list_dot"
    (let rec test lev strm =
       match stream_peek_nth lev strm with
       [ Some ("", "'") -> test2 (lev + 1) strm
       | Some ("", ".") -> ()
       | _ -> raise Stream.Failure ]
    and test2 lev strm =
       match stream_peek_nth lev strm with
       [ Some ("UIDENT" | "LIDENT", _) -> test (lev + 1) strm
       | _ -> raise Stream.Failure ]
     in
      fun strm -> 
	match stream_peek_nth 1 strm with
	  [ Some ("", "'") -> test2 2 strm
	  | Some ("", ".") -> ()
	  | Some ("ANTIQUOT", s) 
	      when (String.length s) > 5 && (String.sub s 0 5) = "list:" -> ()
	  | _ -> raise Stream.Failure ]
    )
;


(* 
 * Different constructors are used for virtual and non-virtual methods. 
 * The oracle test_virtual_method checks for a virtual method. It has
 * to skip over a potential "private" declaration.
 * 
 * not present in the original source
 *)
value test_virtual_method =
  Grammar.Entry.of_parser gram "test_virtual_method"
    (let test_2 strm = 
       match stream_peek_nth 2 strm with
	 [ Some ("", "virtual") -> ()
	 | _ -> raise Stream.Failure]
     in
       fun strm ->
	 match stream_peek_nth 1 strm with
	     [ Some ("", "virtual") -> ()
	     | Some ("", "private") -> test_2 strm
	     | Some ("ANTIQUOT", s) 
		 when (String.length s) > 4 && (String.sub s 0 4) = "opt:" 
		   -> test_2 strm
	     | _ -> raise Stream.Failure
	     ]
    )
;

(* 
 * Constructor arity
 * Tries to figure out, if in the expression or pattern Someconstr(a,b,c) 
 * the constructor is applied to one (triple) or to three arguments.
 * AFAIK you would need to know the typedefinition in order to
 * decide that. However, the reference constr_arity is never changed, 
 * so only Some and Match_Failure are magically treated right. 
 * Other constructors with more than one argument are internally 
 * currified and thus treated as n-ary constructors. This gives a 
 * type error when printing to revised syntax and parsing again.
 * 
 * I simply copy the code and hope for enlightenment in the future.
 * 
 * pa_o loc 297
 *)

value constr_arity = ref [("Some", 1); ("Match_Failure", 1)];


(* pa_o loc 299 *)

value rec is_expr_constr_call  =
  fun
  [ (* <:expr< $uid:_$ >> *)
    Qast.Node "ExUid" _ -> True

  | (* <:expr< $uid:_$.$e$ >>  *)
      Qast.Node "ExAcc" [_; Qast.Node "ExUid" _; e] -> is_expr_constr_call e
  
  | (* <:expr< $e$ $_$ >> *) 
    Qast.Node "ExApp" [_; e; _] -> is_expr_constr_call e
  
  | _ -> False 
]
;


(* pa_o loc 307 *)

value rec constr_expr_arity _loc = 
  fun
  [ (* <:expr< $uid:c$ >> *) 
    Qast.Node "ExUid" [_; Qast.Str c] ->
      try List.assoc c constr_arity.val with [ Not_found -> 0 ]
  
  | (* <:expr< $uid:_$.$e$ >> *) 
    Qast.Node "ExAcc" [_; Qast.Node "ExUid" _; e] -> constr_expr_arity _loc e
  
  | (* <:expr< $e$ $_$ >> *) 
    Qast.Node "ExApp" [_; e; _] ->
      if is_expr_constr_call e then
        Stdpp.raise_with_loc _loc (Stream.Error "currified constructor")
      else 1
  | _ -> 1 ]
;


(* pa_o loc 319 *)

value rec is_patt_constr_call =
  fun
  [ (* <:patt< $uid:_$ >> *) 
    Qast.Node "PaUid" _ -> True

  | (* <:patt< $uid:_$.$p$ >> *) 
    Qast.Node "PaAcc" [_; Qast.Node "PaUid" _; p] -> is_patt_constr_call p

  | (* <:patt< $p$ $_$ >> *)
    Qast.Node "PaApp" [_; p; _] -> is_patt_constr_call p

  | _ -> False ]
;




(* pa_o loc 327 *)

value rec constr_patt_arity _loc =
  fun
  [ (* <:patt< $uid:c$ >> *) 
    Qast.Node "PaUid" [_; Qast.Str c] ->
      try List.assoc c constr_arity.val with [ Not_found -> 0 ]

  | (* <:patt< $uid:_$.$p$ >> *)
    Qast.Node "PaAcc" [_; Qast.Node "ExUid" _; p] -> constr_patt_arity _loc p

  | (* <:patt< $p$ $_$ >> *)
    Qast.Node "PaApp" [_; p; _ ] ->
      if is_patt_constr_call p then
        Stdpp.raise_with_loc _loc (Stream.Error "currified constructor")
      else 1
  | _ -> 1 ]
;


(* 
 * extract the statements from a sequential composition 
 * in order to prepend another statement
 * this is also used in for and while-loops 
 * 
 * pa_o loc 339
 *)
value get_seq =
  fun
  [ (* <:expr< do { $list:el$ } >> -> el *)
    Qast.Node "ExSeq" [_; el] -> el
  | e -> Qast.List [e] ]
;

(* 
 * Choose a type variable that does not occur in tpl. Needed for
 * abstract types, because internally (and in revised) they have
 * the form  type t = 'a, where the type variable on the right is 
 * fresh.
 * 
 * pa_o loc 345
 *)
value choose_tvar qast_tpl =
  let qast_fst = fun
    [ Qast.Tuple [Qast.Str tyvar; x] -> tyvar
				     (* should report an error here *)
    | Qast.Tuple [Qast.Antiquot _ _; _] -> assert False 
    | _ -> assert False
    ] 
  in
  let rec qast_list_of = fun 
    [ Qast.List ql -> List.map qast_fst ql
    | Qast.Cons hd tl -> [qast_fst hd :: qast_list_of tl]
    | Qast.Antiquot _ _ -> assert False	(* should report an error here *)
    | _ -> assert False
    ]
  in
  let tpl = qast_list_of qast_tpl
  in
  let rec find_alpha v =
    let s = String.make 1 v in
    if List.mem s tpl then
      if v = 'z' then None else find_alpha (Char.chr (Char.code v + 1))
    else Some (String.make 1 v)
  in
  let rec make_n n =
    let v = "a" ^ string_of_int n in
    if List.mem v tpl then make_n (succ n) else v
  in
  match find_alpha 'a' with
  [ Some x -> x
  | None -> make_n 1 ]
;


(* 
 * recognize the function syntactic suguar in let bindings, 
 * ie. recognize id <patt> ... which is an abbreviation 
 * of id = fun <patt> ...  
 * If recognized, patt_lid returns Some(id, patt_list) 
 * otherwise None
 * 
 * pa_o loc 361
 *)
value rec patt_lid =
  fun
  [ (* <:patt< $p1$ $p2$ >> *) 
    Qast.Node "PaApp" [_; p1; p2] ->
      match p1 with
      [ (* <:patt< $lid:i$ >> *) 
	Qast.Node "PaLid" [_; i] -> Some (i, [p2])
      | _ ->
          match patt_lid p1 with
          [ Some (i, pl) -> Some (i, [p2 :: pl])
          | None -> None ] ]
  | _ -> None ]
;



(* 
 * Translate the bigarray get syntax e.{...}
 * For one, two, and three arguments call Bigarray.Array{1,2,3}.get
 * otherwise call Bigarray.Genarray.get with an index array.
 * 
 * The original function is quite simple, because it can match on 
 * ExTup to get the list of indices (using quotations of course).
 * Here it is much more difficult: We have to destruct the Qast.t,
 * going down Cons'es and count and collect the indices. 
 * We might even hit an antiqotation. In this case we produce a 
 * Genarray.get, which might be less efficient, but is correct in 
 * expressions. For patterns <:expr<a.{ $list:_$ }>> matches then 
 * of course only Genarray.get's.
 * 
 * We need to distinguish three cases: The indices in e.{...}
 * - are a tuple with one, two or three elements 
 *      -> use Bigarray.Array{1,2,3}
 * - are a tuple with more than three elements or it is an antiquotation
 *      -> use Bigarray.Genarray
 * - is not a typle
 *      -> use Bigarray.Array1
 * Because you cannot distinguish these three cases with the option type
 * we use our own datatype bigarray_args. The UseGenarray constructor 
 * is only used when diving in the Qast.t structure. The Genarray 
 * constructor is only used afterwards with a Qast.t that is either
 * a list valued antiquotation or a Qast list of the indices with 
 * more than three elements.
 * 
 * Keep in mind that the final nil in the list of elements in a 
 * tuple quotation is encoded as Qast.List _ [], 
 * see pa_expand fun quote_expr
 * 
 * po_o loc 373
 *)

(* 
 * ba_acc_term builds the Qast term corresponding to 
 * <:expr< Bigarray.$array_mod$.$acc$ $arr$ arg1 arg2 ...>>
 *)
value ba_acc_term array_mod acc arr args =
  List.fold_left
    (fun t arg -> Qast.Node "ExApp" [Qast.Loc; t; arg])
    (Qast.Node "ExApp"
       [Qast.Loc;
	Qast.Node "ExAcc"
	  [Qast.Loc;
	   Qast.Node "ExAcc"
	     [Qast.Loc;
	      Qast.Node "ExUid" [Qast.Loc; Qast.Str "Bigarray"];
	      Qast.Node "ExUid" [Qast.Loc; Qast.Str array_mod]];
	   Qast.Node "ExLid" [Qast.Loc; Qast.Str acc]];
	arr])
    args
;

type bigarray_args = 
    [Arrayn of list Qast.t | Genarray of Qast.t | UseGenarray];

value bigarray_get _loc arr arg =
  let rec qast_head n = fun
    [ Qast.Cons hd tl -> 
	if n = 0 then UseGenarray 
	else match qast_head (n-1) tl with
	  [ UseGenarray -> UseGenarray
	  | Arrayn tl -> Arrayn [hd :: tl]
	  | Genarray _ -> assert False]
    | Qast.List tl -> 
	if List.length tl <= n then Arrayn tl
	else UseGenarray
    | Qast.Antiquot _ _ -> 
      do {assert(n = 3);
	  UseGenarray}
    | _ -> assert False] 
  in let coords =
    match arg with
	[ Qast.Node "ExTup" [_ ; els ] -> 
	    match qast_head 3 els with
		[ UseGenarray -> Genarray els
		| _ as coords -> coords]
	| Qast.Node "ExTup" _ -> assert False
	| _ -> Arrayn [arg] ]
  in
  match coords with
  [ Arrayn args -> 
      (* 
       * <:expr< Bigarray.Array1.get $arr$ $c1$ >>
       * <:expr< Bigarray.Array2.get $arr$ $c1$ $c2$ >>
       * <:expr< Bigarray.Array3.get $arr$ $c1$ $c2$ $c3$ >>
       *)
      let array_mod = match List.length args with
	  [ 1 -> "Array1"
	  | 2 -> "Array2"
	  | 3 -> "Array3"
	  | _ -> assert False]
      in
	ba_acc_term array_mod "get" arr args

  | Genarray args ->
      (* <:expr< Bigarray.Genarray.get $arr$ [| $list:coords$ |] >> *)
      ba_acc_term "Genarray" "get" arr 
	 [Qast.Node "ExArr" [Qast.Loc; args]]

  | UseGenarray -> assert False 
  ]
;

(* 
 * rewrite the bigarray syntactic suguar e.{...} <- ... 
 * into an application of Bigarray...set
 * 
 * pa_o loc 386
 *)
value bigarray_set _loc var newval =
  match var with
  [ (* <:expr< Bigarray.Array1.get $arr$ $c1$ >> *) 
    Qast.Node "ExApp"
      [Qast.Loc;
       Qast.Node "ExApp"
	 [Qast.Loc;
	  Qast.Node "ExAcc"
	    [Qast.Loc;
	     Qast.Node "ExAcc"
	       [Qast.Loc;
		Qast.Node "ExUid" [Qast.Loc; Qast.Str "Bigarray"];
		Qast.Node "ExUid" [Qast.Loc; Qast.Str "Array1"]];
	     Qast.Node "ExLid" [Qast.Loc; Qast.Str "get"]];
	  arr];
       c1]
    -> 
      (* Some <:expr< Bigarray.Array1.set $arr$ $c1$ $newval$ >> *)
      Some( ba_acc_term "Array1" "set" arr [c1; newval])

  | (* <:expr< Bigarray.Array2.get $arr$ $c1$ $c2$ >> *)
    Qast.Node "ExApp"
      [Qast.Loc;
       Qast.Node "ExApp"
	 [Qast.Loc;
	  Qast.Node "ExApp"
	    [Qast.Loc;
	     Qast.Node "ExAcc"
	       [Qast.Loc;
		Qast.Node "ExAcc"
		  [Qast.Loc;
		   Qast.Node "ExUid" [Qast.Loc; Qast.Str "Bigarray"];
		   Qast.Node "ExUid" [Qast.Loc; Qast.Str "Array2"]];
		Qast.Node "ExLid" [Qast.Loc; Qast.Str "get"]];
	     arr];
	  c1];
       c2]
    ->
      (* Some <:expr< Bigarray.Array2.set $arr$ $c1$ $c2$ $newval$ >> *)
      Some( ba_acc_term "Array2" "set" arr [c1; c2; newval])

  | (* <:expr< Bigarray.Array3.get $arr$ $c1$ $c2$ $c3$ >> *) 
    Qast.Node "ExApp"
      [Qast.Loc;
       Qast.Node "ExApp"
	 [Qast.Loc;
	  Qast.Node "ExApp"
	    [Qast.Loc;
	     Qast.Node "ExApp"
	       [Qast.Loc;
		Qast.Node "ExAcc"
		  [Qast.Loc;
		   Qast.Node "ExAcc"
		     [Qast.Loc;
		      Qast.Node "ExUid" [Qast.Loc; Qast.Str "Bigarray"];
		      Qast.Node "ExUid" [Qast.Loc; Qast.Str "Array3"]];
		   Qast.Node "ExLid" [Qast.Loc; Qast.Str "get"]];
		arr];
	     c1];
	  c2];
       c3]
    ->
      (* Some <:expr< Bigarray.Array3.set $arr$ $c1$ $c2$ $c3$ $newval$ >> *)
      Some( ba_acc_term "Array3" "set" arr [c1; c2; c3; newval] )

  | (* <:expr< Bigarray.Genarray.get $arr$ [| $list:coords$ |] >> *) 
    Qast.Node "ExApp"
      [Qast.Loc;
       Qast.Node "ExApp"
	 [Qast.Loc;
	  Qast.Node "ExAcc"
	    [Qast.Loc;
	     Qast.Node "ExAcc"
	       [Qast.Loc;
		Qast.Node "ExUid" [Qast.Loc; Qast.Str "Bigarray"];
		Qast.Node "ExUid" [Qast.Loc; Qast.Str "Genarray"]];
	     Qast.Node "ExLid" [Qast.Loc; Qast.Str "get"]];
	  arr];
       args]
    ->
      (* 
       * Some <:expr< Bigarray.Genarray.set $arr$ 
       * 	[| $list:coords$ |] $newval$ >>
       *)
      Some( ba_acc_term "Genarray" "set" arr [args; newval] )
  | _ -> None ]
;



EXTEND
  GLOBAL: ctyp poly_type str_item expr patt let_binding type_declaration
    module_expr module_type sig_item with_constr class_type type_kind
    class_expr class_str_item class_sig_item;
  
  module_expr:
    [ [ "functor"; "("; i = a_UIDENT; ":"; t = module_type; ")"; "->";
        me = SELF ->
          Qast.Node "MeFun" [Qast.Loc; i; t; me]
      |	"struct"; st = SLIST0 [ s = str_item; OPT ";;" -> s ]; "end" ->
          Qast.Node "MeStr" [Qast.Loc; st] ]
    | [ me1 = SELF; me2 = SELF -> Qast.Node "MeApp" [Qast.Loc; me1; me2] ]
    | [ me1 = SELF; "."; me2 = SELF ->
          Qast.Node "MeAcc" [Qast.Loc; me1; me2] ]
    | "simple"
      [ i = a_UIDENT -> Qast.Node "MeUid" [Qast.Loc; i]
      | "("; me = SELF; ":"; mt = module_type; ")" ->
          Qast.Node "MeTyc" [Qast.Loc; me; mt]
      | "("; me = SELF; ")" -> me ] ]
  ;


  str_item:
    [ "top"
	[ 
	  "declare"; st = SLIST0 [ s = str_item; OPT ";;" -> s ]; "end" ->
            Qast.Node "StDcl" [Qast.Loc; st]

	| "exception"; ctl = constructor_declaration; b = rebind_exn ->
            let (c, tl) =
              match ctl with
		[ Qast.Tuple [loc; c; tl] -> (c, tl)
		| _ -> assert False ]
            in
              Qast.Node "StExc" [Qast.Loc; c; tl; b]

	| "external"; i = a_LIDENT; ":"; t = ctyp; "="; pd = SLIST1 a_STRING ->
            Qast.Node "StExt" [Qast.Loc; i; t; pd]

	| "external"; "("; op = operator_rparen; ":"; t = ctyp; "=";
              pd = SLIST1 a_STRING ->
		Qast.Node "StExt" 
		  [Qast.Loc; Qast.Str op; t; pd]

        | "include"; me = module_expr -> Qast.Node "StInc" [Qast.Loc; me]

	| "module"; i = a_UIDENT; mb = module_binding ->
            Qast.Node "StMod" [Qast.Loc; i; mb]

        | "module"; "rec"; nmtmes = SLIST1 module_rec_binding SEP "and" ->
            Qast.Node "StRecMod" [Qast.Loc; nmtmes]

        | "module"; "type"; i = a_UIDENT; "="; mt = module_type ->
            Qast.Node "StMty" [Qast.Loc; i; mt]

	| "open"; i = mod_ident -> Qast.Node "StOpn" [Qast.Loc; i]

	| "type"; tdl = SLIST1 type_declaration SEP "and" ->
            Qast.Node "StTyp" [Qast.Loc; tdl]

	  (* 
           * pa_o has two rules here: one for let ... in and 
	   * one for let ...  However, left factorization does not seem
	   * to work with SOPT and SLIST1 (it does with OPT and LIST1).
	   * Therefore I left-facorize the rule myself.
	   * 
	   * Further, pa_o rewrites "let _ = e" to "e", but I believe thats 
	   * only done to read the result of the printers 
	   * into the same ast again. I don't do that here.
           *)

	| "let"; r = SOPT "rec"; l = SLIST1 let_binding SEP "and";
	  inex = OPT [ "in"; x = expr -> x]
	    ->
	      match inex with [
		None -> Qast.Node "StVal" [Qast.Loc; o2b r; l] 

	      | Some x -> 
		  Qast.Node "StExp"
		    [Qast.Loc;
		     Qast.Node "ExLet" [Qast.Loc; o2b r; l; x]]
	      ]

        | "let"; "module"; m = a_UIDENT; mb = module_binding; "in"; e = expr ->
	    Qast.Node "StExp"
	      [Qast.Loc;
               Qast.Node "ExLmd" [Qast.Loc; m; mb; e]]

	| e = expr -> Qast.Node "StExp" [Qast.Loc; e] ]
    ]
  ;
  
  rebind_exn:
    [ [ "="; sl = mod_ident -> sl
      | -> Qast.List [] ] ]
  ;

  
  module_binding:
    [ RIGHTA
      [ "("; m = a_UIDENT; ":"; mt = module_type; ")"; mb = SELF ->
          Qast.Node "MeFun" [Qast.Loc; m; mt; mb]
      | ":"; mt = module_type; "="; me = module_expr ->
          Qast.Node "MeTyc" [Qast.Loc; me; mt]
      | "="; me = module_expr -> me ] ]
  ;

  module_rec_binding:
    [ [ m = a_UIDENT; ":"; mt = module_type; "="; me = module_expr ->
          Qast.Tuple [m; me; mt] ] ]
  ;
  module_type:
    [ [ "functor"; "("; i = a_UIDENT; ":"; t = SELF; ")"; "->"; mt = SELF ->
          Qast.Node "MtFun" [Qast.Loc; i; t; mt] ]
    | [ mt = SELF; "with"; wcl = SLIST1 with_constr SEP "and" ->
          Qast.Node "MtWit" [Qast.Loc; mt; wcl] ]
    | [ "sig"; sg = SLIST0 [ s = sig_item; OPT ";;" -> s ]; "end" ->
          Qast.Node "MtSig" [Qast.Loc; sg] ]
    | [ m1 = SELF; "("; m2 = SELF; ")"-> Qast.Node "MtApp" [Qast.Loc; m1; m2] ]
    | [ m1 = SELF; "."; m2 = SELF -> Qast.Node "MtAcc" [Qast.Loc; m1; m2] ]
    | "simple"
      [ i = a_UIDENT -> Qast.Node "MtUid" [Qast.Loc; i]
      | i = a_LIDENT -> Qast.Node "MtLid" [Qast.Loc; i]
      | "("; mt = SELF; ")" -> mt ] ]
  ;
  sig_item:
    [ "top"
      [ "declare"; st = SLIST0 [ s = sig_item; OPT ";;" -> s ]; "end" ->
          Qast.Node "SgDcl" [Qast.Loc; st]
      | "exception"; ctl = constructor_declaration ->
          let (c, tl) =
            match ctl with
            [ Qast.Tuple [loc; c; tl] -> (c, tl)
            | _ -> assert False ]
          in
          Qast.Node "SgExc" [Qast.Loc; c; tl]
      | "external"; i = a_LIDENT; ":"; t = ctyp; "="; pd = SLIST1 a_STRING ->
          Qast.Node "SgExt" [Qast.Loc; i; t; pd]
      | "external"; "("; op = operator_rparen; ":"; t = ctyp; "=";
            pd = SLIST1 a_STRING ->
              Qast.Node "SgExt" [Qast.Loc; Qast.Str op; t; pd]
      | "include"; mt = module_type -> Qast.Node "SgInc" [Qast.Loc; mt]
      | "module"; i = a_UIDENT; mt = module_declaration ->
          Qast.Node "SgMod" [Qast.Loc; i; mt]
      | "module"; "rec"; mds = SLIST1 module_rec_declaration SEP "and" ->
          Qast.Node "SgRecMod" [Qast.Loc; mds]
      | "module"; "type"; i = a_UIDENT; "="; mt = module_type ->
          Qast.Node "SgMty" [Qast.Loc; i; mt]
      | "module"; "type"; i = a_UIDENT ->
	  Qast.Node "SgMty" 
	    [Qast.Loc; i; 
	     Qast.Node "MtQuo" [Qast.Loc; Qast.Str "abstract"]]
      | "open"; i = mod_ident -> Qast.Node "SgOpn" [Qast.Loc; i]
      | "type"; tdl = SLIST1 type_declaration SEP "and" ->
          Qast.Node "SgTyp" [Qast.Loc; tdl]
      | "val"; i = a_LIDENT; ":"; t = ctyp ->
          Qast.Node "SgVal" [Qast.Loc; i; t] 
      | "val"; "("; op = operator_rparen; ":"; t = ctyp ->
          Qast.Node "SgVal" [Qast.Loc; Qast.Str op; t] ] ]
  ;
  module_declaration:
    [ RIGHTA
      [ ":"; mt = module_type -> mt
      | "("; i = a_UIDENT; ":"; t = module_type; ")"; mt = SELF ->
          Qast.Node "MtFun" [Qast.Loc; i; t; mt] ] ]
  ;
  module_rec_declaration:
    [ [ m = a_UIDENT; ":"; mt = module_type -> Qast.Tuple [m; mt] ] ]
  ;

  with_constr:
    [ [ "type"; tpl = type_parameters; i = mod_ident; "="; t = ctyp ->
          Qast.Node "WcTyp" [Qast.Loc; i; tpl; t]
      | "module"; i = mod_ident; "="; me = module_expr ->
          Qast.Node "WcMod" [Qast.Loc; i; me] ] ]
  ;
  
  expr:
    [ "top" RIGHTA
      [ e1 = SELF; ";"; e2 = SELF -> 
	  Qast.Node "ExSeq" [Qast.Loc; Qast.Cons e1 (get_seq e2) ]
      | e1 = SELF; ";" -> e1 ]

    | "expr1"
      [ "let"; r = SOPT "rec"; l = SLIST1 let_binding SEP "and"; "in";
        x = expr LEVEL "top" ->
          Qast.Node "ExLet" [Qast.Loc; o2b r; l; x]
      | "let"; "module"; m = a_UIDENT; mb = module_binding; "in"; 
	e = expr LEVEL "top" ->
          Qast.Node "ExLmd" [Qast.Loc; m; mb; e]
      | "function"; OPT "|"; l = SLIST1 match_case SEP "|" ->
	  Qast.Node "ExFun" [Qast.Loc; l]
      | "fun"; p = patt LEVEL "simple"; e = fun_def ->
	  Qast.Node "ExFun" 
	    [Qast.Loc; Qast.List [Qast.Tuple [p; Qast.Option None; e]]]
      | "match"; e = SELF; "with"; OPT "|"; l = SLIST1 match_case SEP "|" ->
	  Qast.Node "ExMat" [Qast.Loc; e; l]
      | "try"; e = SELF; "with"; OPT "|"; l = SLIST1 match_case SEP "|" ->
	  Qast.Node "ExTry" [Qast.Loc; e; l]
      | "if"; e1 = SELF; "then"; e2 = expr LEVEL "expr1";
        "else"; e3 = expr LEVEL "expr1" ->
	  Qast.Node "ExIfe" [Qast.Loc; e1; e2; e3]
      | "if"; e1 = SELF; "then"; e2 = expr LEVEL "expr1" ->
	  Qast.Node "ExIfe" 
	    [Qast.Loc; e1; e2; Qast.Node "ExUid" [Qast.Loc; Qast.Str "()"]]
      | "for"; i = a_LIDENT; "="; e1 = SELF; df = direction_flag; e2 = SELF;
        "do"; e = SELF; "done" ->
	  Qast.Node "ExFor" [Qast.Loc; i; e1; e2; df; get_seq e]
      | "while"; e1 = SELF; "do"; e2 = SELF; "done" ->
	  Qast.Node "ExWhi" [Qast.Loc; e1; get_seq e2]
      | "object"; cspo = SOPT class_self_patt; cf = class_structure; "end" ->
          Qast.Node "ExObj" [Qast.Loc; cspo; cf ]
      ]
    | "tuple"
	[ e = SELF; ","; el = SLIST1 (expr LEVEL ":=") SEP "," ->
            Qast.Node "ExTup" [Qast.Loc; Qast.Cons e el]]
	

    | ":=" NONA
      [ e1 = SELF; ":="; e2 = expr LEVEL "expr1" ->
	  Qast.Node "ExAss" 
	    [Qast.Loc; 
	     Qast.Node "ExAcc" 
	       [Qast.Loc; e1; Qast.Node "ExLid" [Qast.Loc; Qast.Str "val"]];
	     e2]
      | e1 = SELF; "<-"; e2 = expr LEVEL "expr1" ->
          match bigarray_set _loc e1 e2 with
              [ Some e -> e
              | None -> Qast.Node "ExAss" [Qast.Loc; e1; e2]
	      ] ]


    | "||" RIGHTA
	[ e1 = SELF; "or"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "or"]; e1];
               e2]
	| e1 = SELF; "||"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "||"]; e1];
               e2] ]
    | "&&" RIGHTA
	[ e1 = SELF; "&"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "&"]; e1];
               e2]
	| e1 = SELF; "&&"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "&&"]; e1];
               e2] ]
    | "<" LEFTA
	[ e1 = SELF; "<"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "<"]; e1];
               e2]
	| e1 = SELF; ">"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str ">"]; e1];
               e2]
	| e1 = SELF; "<="; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "<="]; e1];
               e2]
	| e1 = SELF; ">="; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str ">="]; e1];
               e2]
	| e1 = SELF; "="; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "="]; e1];
               e2]
	| e1 = SELF; "<>"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "<>"]; e1];
               e2]
	| e1 = SELF; "=="; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "=="]; e1];
               e2]
	| e1 = SELF; "!="; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "!="]; e1];
               e2] 
	| e1 = SELF; op = infixop0; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str op]; e1];
               e2] ]

    | "^" RIGHTA
	[ e1 = SELF; "^"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
               [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "^"]; e1];
               e2]
	| e1 = SELF; "@"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "@"]; e1];
               e2] 
	| e1 = SELF; op = infixop1; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str op]; e1];
               e2] ]
  
    | RIGHTA
	[ e1 = SELF; "::"; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExUid" [Qast.Loc; Qast.Str "::"]; e1];
               e2] ]
    | "+" LEFTA
	[ e1 = SELF; "+"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "+"]; e1];
               e2]
	| e1 = SELF; "-"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "-"]; e1];
               e2]
	| e1 = SELF; op = infixop2; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str op]; e1];
               e2] ]
  
    | "*" LEFTA
	[ e1 = SELF; "*"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "*"]; e1];
               e2]
	| e1 = SELF; "/"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "/"]; e1];
               e2]
	| e1 = SELF; "%"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "%"]; e1];
               e2]
	| e1 = SELF; "land"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "land"]; e1];
               e2]
	| e1 = SELF; "lor"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "lor"]; e1];
               e2]
	| e1 = SELF; "lxor"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "lxor"]; e1];
               e2]
	| e1 = SELF; "mod"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "mod"]; e1];
               e2] 
	| e1 = SELF; op = infixop3; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str op]; e1];
               e2] ]
 
    | "**" RIGHTA
	[ e1 = SELF; "**"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "**"]; e1];
               e2]
	| e1 = SELF; "asr"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "asr"]; e1];
               e2]
	| e1 = SELF; "lsl"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "lsl"]; e1];
               e2]
	| e1 = SELF; "lsr"; e2 = SELF ->
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "lsr"]; e1];
               e2] 
	| e1 = SELF; op = infixop4; e2 = SELF -> 
            Qast.Node "ExApp"
              [Qast.Loc;
               Qast.Node "ExApp"
		 [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str op]; e1];
               e2] ]

    | "unary minus" NONA
	[ "-"; e = SELF -> mkumin Qast.Loc (Qast.Str "-") e
	| "-."; e = SELF -> mkumin Qast.Loc (Qast.Str "-.") e ]

    | "apply" LEFTA
      [ e1 = SELF; e2 = SELF ->
          match constr_expr_arity _loc e1 with
          [ 1 -> Qast.Node "ExApp" [ Qast.Loc; e1; e2]
          | _ -> 
              match e2 with
              [ Qast.Node "ExTup" [_; args] -> mkcurriedconstr "ExApp" e1 args
		  (* protect against internal errors *)
	      | Qast.Node "ExTup" _ -> assert False
              | _ -> Qast.Node "ExApp" [Qast.Loc; e1; e2] ] ]

      | "assert"; e = SELF -> mkassert e
      | "lazy"; e = SELF -> Qast.Node "ExLaz" [Qast.Loc; e] ]
      
    | "." LEFTA
	[ e1 = SELF; "."; "("; e2 = SELF; ")" -> 
	    Qast.Node "ExAre" [Qast.Loc; e1; e2]
	| e1 = SELF; "."; "["; e2 = SELF; "]" -> 
	    Qast.Node "ExSte" [Qast.Loc; e1; e2]
	| e1 = SELF; "."; "{"; e2 = SELF; "}" -> bigarray_get _loc e1 e2
     	| e1 = SELF; "."; e2 = SELF -> Qast.Node "ExAcc" [Qast.Loc; e1; e2]]

    | "~-" NONA
      [ "!"; e = SELF -> 
	  Qast.Node "ExAcc" [ Qast.Loc; e; 
			      Qast.Node "ExLid" [ Qast.Loc; Qast.Str "val" ] ]
      | "~-"; e = SELF -> 
	  Qast.Node "ExApp"
	    [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "~-"]; e]
      | "~-."; e = SELF -> 
	  Qast.Node "ExApp"
	    [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str "~-."]; e]
      | f = prefixop; e = SELF -> 
	  Qast.Node "ExApp"
	    [Qast.Loc; Qast.Node "ExLid" [Qast.Loc; Qast.Str f]; e] ]

    | "simple"
	[ s = a_INT -> Qast.Node "ExInt" [Qast.Loc; s]
	| s = a_INT32 -> Qast.Node  "ExInt32" [Qast.Loc; s]
	| s = a_INT64 -> Qast.Node  "ExInt64" [Qast.Loc; s]
	| s = a_NATIVEINT -> Qast.Node  "ExNativeInt" [Qast.Loc; s]
	| s = a_FLOAT -> Qast.Node "ExFlo" [Qast.Loc; s]
	| s = a_STRING -> Qast.Node "ExStr" [Qast.Loc; s]
	| s = a_CHAR -> Qast.Node "ExChr" [Qast.Loc; s]
	| UIDENT "True" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str " True"]
	| UIDENT "False" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str " False"]
	| ids = expr_ident -> mkexprident _loc ids
	| s = "false" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "False"]
	| s = "true" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "True"]
        | "["; "]" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "[]"]
	(* | "["; el = expr1_semi_list; "]" -> mklistexp el *)
	| "["; el = SLIST1 expr LEVEL "expr1" SEP ";"; "]" -> mklistexp el
	| "[|"; el = SLIST0 expr LEVEL "expr1" SEP ";"; "|]" ->
            Qast.Node "ExArr" [Qast.Loc; el]
	| "{"; test_label_eq; lel = SLIST1 lbl_expr SEP ";"; "}" ->
            Qast.Node "ExRec" [Qast.Loc; lel; Qast.Option None]
	| "{"; e = expr LEVEL "."; "with"; 
	    lel = SLIST1 lbl_expr SEP ";"; "}" ->
              Qast.Node "ExRec" [Qast.Loc; lel; Qast.Option (Some e)]
        | "("; ")" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "()"]
	| "("; op = operator_rparen -> 
	    Qast.Node "ExLid" [Qast.Loc; Qast.Str op]
	| "("; e = SELF; ":"; t = ctyp; ")" -> 
            Qast.Node "ExTyc" [Qast.Loc; e; t]
	| "("; e = SELF; ")" -> e
	| "begin"; e = SELF; "end" -> e
	| "begin"; "end" -> Qast.Node "ExUid" [Qast.Loc; Qast.Str "()"]
	(* skip LOCATE and QUOTATION here *)
	]
    ]
  ;

  let_binding:
    [ [ p = patt; e = fun_binding ->
          match patt_lid p with
          [ Some (i, pl) ->
              let e =
                List.fold_left 
		  (fun e p -> 
		     (* <:expr< fun $p$ -> $e$ >> *)
		     Qast.Node "ExFun" 
		       [Qast.Loc; 
			Qast.List [Qast.Tuple [p; Qast.Option None; e]]]
		  ) 
		  e pl
              in
              Qast.Tuple 
		[(* <:patt< $lid:i$ >> *)
		  Qast.Node "PaLid" [Qast.Loc; i];
		  e]
          | None -> Qast.Tuple [p; e] ] ] ]
  ;
  fun_binding:
    [ RIGHTA
      [ p = patt LEVEL "simple"; e = SELF -> 
	  Qast.Node "ExFun"
	    [Qast.Loc;
	     Qast.List [Qast.Tuple [p; Qast.Option None; e]]]
      | "="; e = expr -> e
      | ":"; t = ctyp; "="; e = expr -> Qast.Node "ExTyc" [Qast.Loc; e; t]
      ] ]
  ;

  match_case:
    [ [ x1 = patt; w = SOPT [ "when"; e = expr -> e ]; "->"; x2 = expr ->
          Qast.Tuple[ x1; w; x2 ] ] ]
  ;


  lbl_expr:
    [ [ i = patt_label_ident; "="; e = expr LEVEL "expr1" -> 
	  Qast.Tuple [i; e] ] ]
  ;


  expr_ident:
    [ RIGHTA
      [ i = a_LIDENT -> [ Qast.Node "ExLid" [Qast.Loc; i] ]
      | i = a_UIDENT -> [ Qast.Node "ExUid" [Qast.Loc; i] ]
      | i = a_UIDENT; "."; j = SELF -> 
	  [(Qast.Node "ExUid" [Qast.Loc; i]) :: j ] ] ]
  ;

  fun_def:
    [ RIGHTA
	[ p = patt LEVEL "simple"; e = SELF -> 
	    Qast.Node "ExFun"
	      [Qast.Loc; Qast.List [Qast.Tuple [p; Qast.Option None; e]]]
	| "->"; e = expr -> e
	]]
  ;

  patt: 
    [ LEFTA
      [ p1 = SELF; "as"; i = a_LIDENT -> 
	  Qast.Node "PaAli" 
	    [Qast.Loc; p1; 
	     Qast.Node "PaLid" [Qast.Loc; i]]
      ]
    | LEFTA
      [ p1 = SELF; "|"; p2 = SELF -> Qast.Node "PaOrp" [Qast.Loc; p1; p2] ]
    | "tuple"
	  (* 
	   * In pa_o we find: pl = LIST1 NEXT SEP ",",
	   * however, NEXT does not work with SLIST1, therefore we
	   * refer to the next level by name.
	   *)
	[ p = SELF; ","; pl = SLIST1 (patt LEVEL "after-tuple") SEP "," ->
	    Qast.Node "PaTup" [Qast.Loc; Qast.Cons p pl]
	]
    | "after-tuple" NONA
      [ p1 = SELF; ".."; p2 = SELF -> Qast.Node "PaRng" [Qast.Loc; p1; p2] ]
    | RIGHTA
      [ p1 = SELF; "::"; p2 = SELF -> 
	  Qast.Node "PaApp"
	    [Qast.Loc;
	     Qast.Node "PaApp"
	       [Qast.Loc;
		Qast.Node "PaUid" [Qast.Loc; Qast.Str "::"];
		p1];
	     p2]]

    | LEFTA
      [ p1 = SELF; p2 = SELF ->
          match constr_patt_arity _loc p1 with
          [ 1 -> Qast.Node "PaApp" [Qast.Loc; p1; p2]
          | n ->
              let p2 =
                match p2 with
                [ Qast.Node "PatAny" [Qast.Loc] when n > 1 ->
		    let rec loop n =
		      if n = 0 
		      then Qast.List []
		      else 
			Qast.Cons
			  (Qast.Node "PaAny" [Qast.Loc])
			  (loop (n -1))
		    in 
		    let pl = loop n
                    in
		      Qast.Node "PaTup" [Qast.Loc; pl]
                | _ -> p2 ]
              in
		match p2 with
		  [ (* <:patt< ( $list:pl$ ) >> ->
                     * List.fold_left (fun p1 p2 -> <:patt< $p1$ $p2$ >>) p1 pl 
		     *)
		    Qast.Node "PaTup" [_; args] -> 
		      mkcurriedconstr "PaApp" p1 args

		      (* protect against internal errors *)
		  | Qast.Node "PaTup" _ -> assert False
		  | _ -> Qast.Node "PaApp" [Qast.Loc; p1; p2]
		  ] ] 
      ]
    | LEFTA
      [ p1 = SELF; "."; p2 = SELF -> Qast.Node "PaAcc" [Qast.Loc; p1; p2] ]
    | "simple"
      [ s = a_LIDENT -> Qast.Node "PaLid" [Qast.Loc; s]
      | s = a_UIDENT -> Qast.Node "PaUid" [Qast.Loc; s]
      | s = a_INT -> Qast.Node "PaInt" [Qast.Loc; s]
      | s = a_INT32 -> Qast.Node  "PaInt32" [Qast.Loc; s]
      | s = a_INT64 -> Qast.Node  "PaInt64" [Qast.Loc; s]
      | s = a_NATIVEINT -> Qast.Node  "PaNativeInt" [Qast.Loc; s]
      | "-"; s = a_INT -> mkuminpat Qast.Loc (Qast.Str "-") (Qast.Bool True) s
      | "-"; s = a_INT32 -> mkuminpat Qast.Loc (Qast.Str "-") (Qast.Bool True) s
      | "-"; s = a_INT64 -> mkuminpat Qast.Loc (Qast.Str "-") (Qast.Bool True) s
      | "-"; s = a_NATIVEINT -> mkuminpat Qast.Loc (Qast.Str "-") (Qast.Bool True) s
      | "-"; s = a_FLOAT ->
          mkuminpat Qast.Loc (Qast.Str "-") (Qast.Bool False) s
      | s = a_FLOAT -> Qast.Node "PaFlo" [Qast.Loc; s]
      | s = a_STRING -> Qast.Node "PaStr" [Qast.Loc; s]
      | s = a_CHAR -> Qast.Node "PaChr" [Qast.Loc; s]
      | UIDENT "True" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str " True"]
      | UIDENT "False" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str " False"]
      | s = "false" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str "False"]
      | s = "true" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str "True"]
      | "["; "]" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str "[]"]
      | "["; pl = SLIST1 patt SEP ";"; "]" -> mklistpat pl

      | "[|"; pl = SLIST0 patt SEP ";"; "|]" ->
          Qast.Node "PaArr" [Qast.Loc; pl]
      | "{"; lpl = SLIST1 lbl_patt SEP ";"; "}" ->
          Qast.Node "PaRec" [Qast.Loc; lpl]
      | "("; ")" -> Qast.Node "PaUid" [Qast.Loc; Qast.Str "()"]
      | "("; op = operator_rparen -> Qast.Node "PaLid" [Qast.Loc; Qast.Str op]
      | "("; p = patt; ":"; t = ctyp; ")" -> Qast.Node "PaTyc" [Qast.Loc; p; t]
      | "("; p = patt; ")" -> p
      | "_" -> Qast.Node "PaAny" [Qast.Loc]

      (* 
       * pa_o has also the following two rules here, I put them below 
       * where the other label stuff is
       * | "`"; s = ident -> <:patt< ` $s$ >>
       * | "#"; t = mod_ident -> <:patt< # $list:t$ >>
       *)
      ]
    ]
  ;

  lbl_patt:
    [ [ i = patt_label_ident; "="; p = patt -> Qast.Tuple [i; p] ] ]
  ;

  patt_label_ident:
    [ LEFTA
	[ p1 = SELF; "."; p2 = SELF -> Qast.Node "PaAcc" [Qast.Loc; p1; p2] ]
    | "simple" RIGHTA
	[ i = a_UIDENT -> Qast.Node "PaUid" [Qast.Loc; i]
	| i = a_LIDENT -> Qast.Node "PaLid" [Qast.Loc; i] ] ]
  ;



  (****************************************************************************
   *
   *      type declarations
   *
   ****************************************************************************)

  type_declaration:
    [ [ tpl = type_parameters; n = type_patt; "="; tk = type_kind;
        cl = SLIST0 constrain ->
          Qast.Tuple [n; tpl; tk; cl]
      | tpl = type_parameters; n = type_patt; cl = SLIST0 constrain ->
          Qast.Tuple [n; tpl; 
		      Qast.Node "TyQuo" [Qast.Loc; Qast.Str (choose_tvar tpl)]; 
		      cl] ] ]
  ;


  type_patt:
    [ [ n = a_LIDENT -> Qast.Tuple [Qast.Loc; n] ] ]
  ;

  constrain:
    [ [ "constraint"; t1 = ctyp; "="; t2 = ctyp -> Qast.Tuple [t1; t2] ] ]
  ;

  type_kind:
    [ [ "private"; tk = type_kind -> Qast.Node "TyPrv" [Qast.Loc; tk]
      | test_constr_decl; 
	OPT "|"; cdl = SLIST1 constructor_declaration SEP "|" -> 
	  Qast.Node "TySum" [Qast.Loc; cdl]
      | t = ctyp -> t
      | t = ctyp; "="; "private"; tk = type_kind ->
      	  Qast.Node "TyMan" 
      	    [Qast.Loc; t;
	     Qast.Node "TyPrv" [Qast.Loc; tk]]
      | t = ctyp; "="; "{"; ldl = SLIST1 label_declaration SEP ";"; "}" ->
	    Qast.Node "TyMan" 
	      [Qast.Loc; t;
	       Qast.Node "TyRec" [Qast.Loc; ldl]]
      | t = ctyp; "="; OPT "|"; cdl = SLIST1 constructor_declaration SEP "|" ->
	    Qast.Node "TyMan" 
	      [Qast.Loc; t;
	       Qast.Node "TySum" [Qast.Loc; cdl]]
      | "{"; ldl = SLIST1 label_declaration SEP ";"; "}" ->
	  Qast.Node "TyRec" [Qast.Loc; ldl]
      ] ]
  ;

  type_parameters:
    [ [ -> (* empty *) Qast.List []
      | tp = type_parameter -> Qast.List [tp]
      | "("; tpl = SLIST1 type_parameter SEP ","; ")" -> tpl ] ]
  ;

  type_parameter:
    [ [ "'"; i = ident ->
          Qast.Tuple [i; Qast.Tuple [Qast.Bool False; Qast.Bool False]]
      | "+"; "'"; i = ident ->
          Qast.Tuple [i; Qast.Tuple [Qast.Bool True; Qast.Bool False]]
      | "-"; "'"; i = ident ->
          Qast.Tuple [i; Qast.Tuple [Qast.Bool False; Qast.Bool True]] ] ]
  ;


    (* 
     * Note that the ptyp quotation uses poly_type as entry point in 
     * the grammar. So the ptyp quotation covers slightly more than 
     * the following ctyp grammar entry (but still much less than 
     * the ctyp of the revised syntax).
     *)

  ctyp:
    [ [ t1 = SELF; "as"; "'"; i = ident -> 
	  Qast.Node "TyAli" 
	    [Qast.Loc; t1; 
	     Qast.Node "TyQuo" [Qast.Loc; i]]
      ]
    | "arrow" RIGHTA
      [ t1 = SELF; "->"; t2 = SELF -> 
	  Qast.Node "TyArr" [Qast.Loc; t1; t2]
      ]
    | "star"
      [ t = SELF; "*"; tl = SLIST1 (ctyp LEVEL "ctyp1") SEP "*" ->
	  Qast.Node "TyTup" [Qast.Loc; Qast.Cons t tl]
      ]
    | "ctyp1"
      [ t1 = SELF; t2 = SELF -> 
	  Qast.Node "TyApp" [Qast.Loc; t2; t1]
      ]
    | "ctyp2"
      [ t1 = SELF; "."; t2 = SELF -> 
	  Qast.Node "TyAcc" [Qast.Loc; t1; t2]
      | t1 = SELF; "("; t2 = SELF; ")" -> 
	  Qast.Node "TyApp" [Qast.Loc; t1; t2]
      ]

    | "simple"
	[ "'"; i = ident -> Qast.Node "TyQuo" [Qast.Loc; i]
	| "_" -> Qast.Node "TyAny" [Qast.Loc]
	| i = a_LIDENT -> Qast.Node "TyLid" [Qast.Loc; i]
	| i = a_UIDENT -> Qast.Node "TyUid" [Qast.Loc; i]

	| "("; t = SELF; ","; tl = LIST1 ctyp SEP ","; ")";
            i = ctyp LEVEL "ctyp2" ->
          List.fold_left 
	    (fun c a -> 
	       (* <:ctyp< $c$ $a$ >> *)
	       Qast.Node "TyApp" [Qast.Loc; c; a]) 
	    i 
	    [t :: tl]


	| "("; t = SELF; ")" -> t 
	] ]
    ;

  constructor_declaration:
    [ [ ci = a_UIDENT; "of"; cal = SLIST1 ctyp LEVEL "ctyp1" SEP "*" ->
          Qast.Tuple [Qast.Loc; ci; cal]
      | ci = a_UIDENT -> Qast.Tuple [Qast.Loc; ci; Qast.List []] 
      ] ]
  ;

  label_declaration:
    [ [ i = a_LIDENT; ":"; t = poly_type -> 
	  Qast.Tuple [Qast.Loc; i; Qast.Bool False; t]
      | "mutable"; i = a_LIDENT; ":"; t = poly_type -> 
	  Qast.Tuple [Qast.Loc; i; Qast.Bool True; t]
      ] ]
  ;

  ident:
    [ [ i = a_LIDENT -> i
      | i = a_UIDENT -> i ] ]
  ;

  mod_ident:
    [ RIGHTA
      [ i = a_UIDENT -> Qast.List [i]
      | i = a_LIDENT -> Qast.List [i]
      | i = a_UIDENT; "."; j = SELF -> Qast.Cons i j ] ]
  ;

  (****************************************************************************
   *
   *      Objects and Classes
   *
   ****************************************************************************)

  str_item:
    [ [ "class"; cd = SLIST1 class_declaration SEP "and" ->
          Qast.Node "StCls" [Qast.Loc; cd]
      | "class"; "type"; ctd = SLIST1 class_type_declaration SEP "and" ->
          Qast.Node "StClt" [Qast.Loc; ctd] 
      ] ]
  ;
  sig_item:
    [ [ "class"; cd = SLIST1 class_description SEP "and" ->
          Qast.Node "SgCls" [Qast.Loc; cd]
      | "class"; "type"; ctd = SLIST1 class_type_declaration SEP "and" ->
          Qast.Node "SgClt" [Qast.Loc; ctd] ] ]
  ;
  class_declaration:
    [ [ vf = SOPT "virtual"; ctp = class_type_parameters; i = a_LIDENT;
	cfb = class_fun_binding ->
          Qast.Record
            [("ciLoc", Qast.Loc); ("ciVir", o2b vf); ("ciPrm", ctp);
             ("ciNam", i); ("ciExp", cfb)] ] ]
  ;
  class_fun_binding:
    [ [ "="; ce = class_expr -> (ce : Qast.t)
      (* todo
       * | ":"; ct = class_type; "="; ce = class_expr ->
       *     Qast.Node "CeTyc" [Qast.Loc; ce; ct]
       *)
      | p = patt LEVEL "simple"; cfb = SELF ->
	  Qast.Node "CeFun" [Qast.Loc; p; cfb] ] ]
  ;
  class_type_parameters:
    [ [ -> Qast.Tuple [Qast.Loc; Qast.List []]
      | "["; tpl = SLIST1 type_parameter SEP ","; "]" ->
          Qast.Tuple [Qast.Loc; tpl] ] ]
  ;

  class_fun_def:
    [ [ p = patt LEVEL "simple"; "->"; ce = class_expr ->
	  Qast.Node "CeFun" [Qast.Loc; p; ce]
      (* 
       * | p = labeled_patt; "->"; ce = class_expr ->
       *     <:class_expr< fun $p$ -> $ce$ >>
       *)
      | p = patt LEVEL "simple"; cfd = SELF ->
	  Qast.Node "CeFun" [Qast.Loc; p; cfd]
      (* 
       * | p = labeled_patt; cfd = SELF ->
       *     <:class_expr< fun $p$ -> $cfd$ >> 
       *)
      ] ]
  ;

  class_expr:
    [ "top"
      [ "fun"; cfd = class_fun_def -> cfd
      | "let"; rf = SOPT "rec"; lb = SLIST1 let_binding SEP "and"; "in";
        ce = SELF ->
          Qast.Node "CeLet" [Qast.Loc; o2b rf; lb; ce]
      ]
    | "apply" LEFTA
      [ ce = SELF; e = expr LEVEL "label" ->
          Qast.Node "CeApp" [Qast.Loc; ce; e] ]
    | "simple"
      [ "["; ctcl = SLIST1 ctyp SEP ","; "]"; ci = class_longident ->
          Qast.Node "CeCon" [Qast.Loc; ci; ctcl]
      | ci = class_longident -> 
	  Qast.Node "CeCon" [Qast.Loc; ci; Qast.List []]
      | "object"; cspo = SOPT class_self_patt; cf = class_structure; "end" ->
          (Qast.Node "CeStr" [Qast.Loc; cspo; cf] : Qast.t)
      | "("; ce = SELF; ":"; ct = class_type; ")" ->
          Qast.Node "CeTyc" [Qast.Loc; ce; ct]
      | "("; ce = SELF; ")" -> ce 
      ] ]
  ;

  class_structure:
    [ [ cf = SLIST0 class_str_item -> cf ] ]
  ;

  class_self_patt:
    [ [ "("; p = patt; ")" -> p
      | "("; p = patt; ":"; t = ctyp; ")" ->
          Qast.Node "PaTyc" [Qast.Loc; p; t] ] ]
  ;
  class_str_item:
    [ [ "declare"; st = SLIST0 class_str_item; "end" ->
          Qast.Node "CrDcl" [Qast.Loc; st]
      | "inherit"; ce = class_expr; pb = SOPT as_lident ->
          Qast.Node "CrInh" [Qast.Loc; ce; pb]
      | "val"; mf = SOPT "mutable"; lab = label; e = cvalue_binding ->
          Qast.Node "CrVal" [Qast.Loc; lab; o2b mf; e]
      (* 
       * 	pa_o has two rules for virtual methods
       * 	we combine them into one
       * 
       * | "method"; "virtual"; l = label; ":"; t = poly_type ->
       *     <:class_str_item< method virtual $l$ : $t$ >>
       *)
      | "method"; test_virtual_method;
	pf = SOPT "private"; "virtual"; l = label; 
	":"; t = poly_type ->
            Qast.Node "CrVir" [Qast.Loc; l; o2b pf; t]

      (* 
       * 	pa_o permits to swap virtual and private
       * 	the reference manual states "method private virtual" as the 
       * 	right order, so we exclude "method virtual private" here
       * 
       * | "method"; "virtual"; "private"; l = label; ":"; t = poly_type ->
       *     <:class_str_item< method virtual private $l$ : $t$ >>
       *)

      (* 
       * 	pa_o has four rules for non-virtual methods
       * 	we combine them into one, similarly to q_MLast
       * 
       * | "method"; "private"; l = label; ":"; t = poly_type; "="; e = expr ->
       *     MLast.CrMth loc l True e (Some t)
       * | "method"; "private"; l = label; sb = fun_binding ->
       *     MLast.CrMth loc l True sb None
       * | "method"; l = label; ":"; t = poly_type; "="; e = expr ->
       *     MLast.CrMth loc l False e (Some t)
       * | "method"; l = label; sb = fun_binding ->
       *     MLast.CrMth loc l False sb None
       *)
      | "method"; pf = SOPT "private"; l = label; topt = SOPT polyt; 
	e = fun_binding ->
          Qast.Node "CrMth" [Qast.Loc; l; o2b pf; e; topt]
      | "constraint"; t1 = ctyp; "="; t2 = ctyp ->
          Qast.Node "CrCtr" [Qast.Loc; t1; t2]
      | "initializer"; se = expr -> Qast.Node "CrIni" [Qast.Loc; se]
      ] ]
  ;

  as_lident:
    [ [ "as"; i = a_LIDENT -> i ] ]
  ;

  polyt:
    [ [ ":"; t = poly_type -> t ] ]
  ;

  cvalue_binding:
    [ [ "="; e = expr -> e
      | ":"; t = ctyp; "="; e = expr -> Qast.Node "ExTyc" [Qast.Loc; e; t]
      | ":"; t = ctyp; ":>"; t2 = ctyp; "="; e = expr ->
          Qast.Node "ExCoe" [Qast.Loc; e; Qast.Option (Some t); t2]
      | ":>"; t = ctyp; "="; e = expr ->
          Qast.Node "ExCoe" [Qast.Loc; e; Qast.Option None; t] 
      ] ]
  ;

  label:
    [ [ i = a_LIDENT -> i ] ]
  ;

  (****************************************************************************
   *
   *      Class types
   *
   ****************************************************************************)

  class_type:
    [ [ test_ctyp_minusgreater; t = ctyp LEVEL "star"; "->"; ct = SELF ->
          Qast.Node "CtFun" [Qast.Loc; t; ct]
      | cs = class_signature -> cs ] ]
  ;
  class_signature:
    [ [ "["; tl = SLIST1 ctyp SEP ","; "]"; id = clty_longident ->
	  Qast.Node "CtCon" [Qast.Loc; id; tl]
      | id = clty_longident -> Qast.Node "CtCon" [Qast.Loc; id; Qast.List []]
      | "object"; cst = SOPT class_self_type; csf = SLIST0 class_sig_item;
        "end" ->
	  Qast.Node "CtSig" [Qast.Loc; cst; csf] ] ]
  ;

  class_self_type:
    [ [ "("; t = ctyp; ")" -> t ] ]
  ;

  class_sig_item:
    [ [ "declare"; st = SLIST0 class_sig_item; "end" ->
          Qast.Node "CgDcl" [Qast.Loc; st]
      | "inherit"; cs = class_signature -> Qast.Node "CgInh" [Qast.Loc; cs]
      | "val"; mf = SOPT "mutable"; l = label; ":"; t = ctyp ->
          Qast.Node "CgVal" [Qast.Loc; l; o2b mf; t]
      | "method"; test_virtual_method; pf = SOPT "private"; 
	"virtual"; l = label; ":"; t = poly_type ->
          Qast.Node "CgVir" [Qast.Loc; l; o2b pf; t]
      | "method"; pf = SOPT "private"; l = label; ":"; t = poly_type ->
          Qast.Node "CgMth" [Qast.Loc; l; o2b pf; t]
      | "constraint"; t1 = ctyp; "="; t2 = ctyp ->
          Qast.Node "CgCtr" [Qast.Loc; t1; t2] ] ]
  ;

  class_description:
    [ [ vf = SOPT "virtual"; ctp = class_type_parameters; n = a_LIDENT; ":";
        ct = class_type ->
          Qast.Record
            [("ciLoc", Qast.Loc); ("ciVir", o2b vf); ("ciPrm", ctp);
             ("ciNam", n); ("ciExp", ct)] ] ]
  ;

  class_type_declaration:
    [ [ vf = SOPT "virtual"; ctp = class_type_parameters; n = a_LIDENT; "=";
        cs = class_signature ->
          Qast.Record
            [("ciLoc", Qast.Loc); ("ciVir", o2b vf); ("ciPrm", ctp);
             ("ciNam", n); ("ciExp", cs)] ] ]
  ;

  expr: LEVEL "simple"
    [ LEFTA
      [ "new"; i = class_longident -> Qast.Node "ExNew" [Qast.Loc; i] ] ]
  ;
  expr: LEVEL "."
    [ [ e = SELF; "#"; lab = label -> Qast.Node "ExSnd" [Qast.Loc; e; lab] ] ]
  ;
  expr: LEVEL "simple"
    [ [ "("; e = SELF; ":"; t = ctyp; ":>"; t2 = ctyp; ")" ->
          Qast.Node "ExCoe" [Qast.Loc; e; Qast.Option (Some t); t2]
      | "("; e = SELF; ":>"; t = ctyp; ")" ->
          Qast.Node "ExCoe" [Qast.Loc; e; Qast.Option None; t]
      | "{<"; fel = SLIST0 field_expr SEP ";"; ">}" ->
          Qast.Node "ExOvr" [Qast.Loc; fel] ] ]
  ;
  field_expr:
    [ [ l = label; "="; e = expr LEVEL "expr1" -> Qast.Tuple [l; e] ] ]
  ;


  ctyp: LEVEL "simple"
    [ [ "#"; id = class_longident -> Qast.Node "TyCls" [Qast.Loc; id]
      | "<"; (ml, v) = meth_list; ">" -> Qast.Node "TyObj" [Qast.Loc; ml; o2b v]
      | "<"; ">" -> Qast.Node "TyObj" [Qast.Loc; Qast.List []; Qast.Bool False ]
    ] ]
  ;
  
  meth_list:
    [ [ f = field; ";"; (ml, v) = SELF -> (Qast.Cons f ml, v)
      | f = field; ";" -> (Qast.Cons f (Qast.List []), Qast.Bool False)
      | f = field -> (Qast.Cons f (Qast.List []), Qast.Bool False)
      | ".." -> (Qast.List [], Qast.Bool True) ] ]
  ;
  field:
    [ [ lab = a_LIDENT; ":"; t = poly_type -> Qast.Tuple [lab; t] ] ]
  ;


  (****************************************************************************
   *
   *      polymorphic types
   *
   ****************************************************************************)

  typevar:
    [ [ "'"; i = ident -> i ] ]
  ;

  poly_type:
    [ [ test_typevar_list_dot; tpl = SLIST1 typevar; "."; t2 = ctyp ->
          Qast.Node "TyPol" [Qast.Loc; tpl; t2]
      | t = ctyp -> t 
      ] ]
  ;

  (****************************************************************************
   *
   *      Identifiers
   *
   ****************************************************************************)

  clty_longident:
    [ [ m = a_UIDENT; "."; l = SELF -> Qast.Cons m l
      | i = a_LIDENT -> Qast.List [i] ] ]
  ;
  class_longident:
    [ [ m = a_UIDENT; "."; l = SELF -> Qast.Cons m l
      | i = a_LIDENT -> Qast.List [i] ] ]
  ;


  (* HT: insert empty level "label" because class expressions refer to it *)
  expr: AFTER "apply"
    [ "label"
      [ 
      ] ]
  ;


  direction_flag:
    [ [ "to" -> Qast.Bool True
      | "downto" -> Qast.Bool False ] ]
  ;



(****************************************************************************
 ****************************************************************************
 *
 *                 Antiquotations for local entries
 *
 ****************************************************************************
 ****************************************************************************)


  (* give a singleton list back; it will be converted by mkexprident *)
  expr_ident:
    [ [ a = ANTIQUOT -> [ antiquot "" _loc a ] ] ]
  ;

  patt_label_ident: LEVEL "simple"
    [ [ a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;


  type_patt:
    [ [ a = ANTIQUOT "locstr" -> antiquot "locstr" _loc a ] ]
  ;

  mod_ident:
    [ [ a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;

  clty_longident:
    [ [ a = a_list -> a ] ]
  ;

  class_longident:
    [ [ a = a_list -> a ] ]
  ;

  direction_flag:
    [ [ a = ANTIQUOT "to" -> antiquot "to" _loc a ] ]
  ;

  class_signature:
    [ [ a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;

  meth_list:
    [ [ a = ANTIQUOT "list"; v = SOPT [ ";"; ".." -> Qast.Bool True ] -> 
	  (antiquot "list" _loc a, o2b v)
      ] ]
  ;

END;



(****************************************************************************
 ****************************************************************************
 *
 *                 Antiquotations for global entries
 *
 ****************************************************************************
 ****************************************************************************)


EXTEND
  module_expr: LEVEL "simple"
    [ [ a = ANTIQUOT "mexp" -> antiquot "mexp" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;
  
  str_item: LEVEL "top"
    [ [ a = ANTIQUOT "stri" -> antiquot "stri" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;
  

  module_type: LEVEL "simple"
    [ [ a = ANTIQUOT "mtyp" -> antiquot "mtyp" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;
  sig_item: LEVEL "top"
    [ [ a = ANTIQUOT "sigi" -> antiquot "sigi" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;
  expr: LEVEL "top"
    [ [ 
          (* the expr - sequence antiquotation, providing
	   *      <:expr< $seq:els$ >>: statement sequence
	   *)
	el = a_seq -> Qast.Node "ExSeq" [Qast.Loc; el]
      ] ];

  expr: LEVEL "tuple"
    [ [ 
          (* the expr - tuple antiquotation, providing
	   *      <:expr< $tup:el$ >>: tuple
	   *)
	el = a_tup -> Qast.Node "ExTup" [Qast.Loc; el]
      ] ];

  expr: LEVEL "simple"
    [ [ 
			(* the exp antiquotation in other nodes, eg str_item *)
	a = ANTIQUOT "exp" -> antiquot "exp" _loc a
				(* the expr antiquotation inside expressions *)
      | a = ANTIQUOT -> antiquot "" _loc a


      ] ]
  ;

  patt: LEVEL "tuple"
    [ [ 
          (* the pattern - tuple antiquotation, providing
	   *      <:expr< $tup:el$ >>: tuple
	   *)
	pl = a_tup -> Qast.Node "PaTup" [Qast.Loc; pl]
      ] ];

  patt: LEVEL "simple"
    [ [ a = ANTIQUOT "pat" -> antiquot "pat" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a


    ] ]
  ;

  ctyp: LEVEL "star"
    [ [
          (* the type - tuple antiquotation, providing
	   *      <:ctyp< $tup:el$ >> : tuple type
	   *)
	tl = a_tup -> Qast.Node "TyTup" [Qast.Loc; tl]
      ] ]
  ;

  ctyp: LEVEL "simple"
    [ [ a = ANTIQUOT "typ" -> antiquot "typ" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      (* | "("; tl = a_list; ")" -> Qast.Node "TyTup" [Qast.Loc; tl]  *)
      ] ]
  ;

  class_expr: LEVEL "simple"
    [ [ a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;

  class_str_item:
    [ [ a = ANTIQUOT "cli" -> antiquot "cli" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;

  class_sig_item:
    [ [ a = ANTIQUOT "clsi" -> antiquot "clsi" _loc a 
      | a = ANTIQUOT -> antiquot "" _loc a ] ]
  ;

  (* put the antiquotation into class_signature, which is part of 
   * class_type (see local antiquotations above)
   * 
   * class_type:
   *   [ [ a = ANTIQUOT -> antiquot "" _loc a ] ]
   * ;
   *)


  a_list:
    [ [ a = ANTIQUOT "list" -> antiquot "list" _loc a ] ]
  ;

  a_tup:
    [ [ a = ANTIQUOT "tup" -> antiquot "tup" _loc a ] ]
  ;

  a_seq:
    [ [ a = ANTIQUOT "seq" -> antiquot "seq" _loc a ] ]
  ;

  a_opt:
    [ [ a = ANTIQUOT "opt" -> antiquot "opt" _loc a ] ]
  ;

  a_UIDENT:
    [ [ a = ANTIQUOT "uid" -> antiquot "uid" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | i = UIDENT -> Qast.Str i ] ]
  ;
  a_LIDENT:
    [ [ a = ANTIQUOT "lid" -> antiquot "lid" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | i = LIDENT -> Qast.Str i ] ]
  ;
  a_INT:
    [ [ a = ANTIQUOT "int" -> antiquot "int" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = INT -> Qast.Str s ] ]
  ;
  a_INT32:
    [ [ a = ANTIQUOT "int32" -> antiquot "int32" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = INT32 -> Qast.Str s ] ]
  ;
  a_INT64:
    [ [ a = ANTIQUOT "int64" -> antiquot "int64" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = INT64 -> Qast.Str s ] ]
  ;
  a_NATIVEINT:
    [ [ a = ANTIQUOT "nativeint" -> antiquot "nativeint" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = NATIVEINT -> Qast.Str s ] ]
  ;
  a_FLOAT:
    [ [ a = ANTIQUOT "flo" -> antiquot "flo" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = FLOAT -> Qast.Str s ] ]
  ;
  a_STRING:
    [ [ a = ANTIQUOT "str" -> antiquot "str" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = STRING -> Qast.Str s ] ]
  ;
  a_CHAR:
    [ [ a = ANTIQUOT "chr" -> antiquot "chr" _loc a
      | a = ANTIQUOT -> antiquot "" _loc a
      | s = CHAR -> Qast.Str s ] ]
  ;
END;

value apply_entry e =
  let f s =
    let (bolpos,lnum,fname) = q_position in
    let (bolp,ln,_) = (bolpos.val, lnum.val, fname.val) in
    let zero_position() = do { bolpos.val := 0; lnum.val := 1 } in
    let restore_position() = do { bolpos.val := bolp; lnum.val := ln } in
    let _ = zero_position() in
    try
      let result =
        Grammar.Entry.parse e (Stream.of_string s) in
      let _ = restore_position() in
      result
    with exc -> do { restore_position(); raise exc } in
  let expr s = Qast.to_expr (f s) in
  let patt s = Qast.to_patt (f s) in
  Quotation.ExAst (expr, patt)
;


let sig_item_eoi = Grammar.Entry.create gram "signature item" in
do {
  EXTEND
    sig_item_eoi:
      [ [ x = sig_item; EOI -> x ] ]
    ;
  END;
  Quotation.add "sig_item" (apply_entry sig_item_eoi)
}; 

let str_item_eoi = Grammar.Entry.create gram "structure item" in
do {
  EXTEND
    str_item_eoi:
      [ [ x = str_item; EOI -> x ] ]
    ;
  END;
  Quotation.add "str_item" (apply_entry str_item_eoi)
};

(* 
 * In the revised syntax ctyp covers quantified types and constructions 
 * that appear only in type declarations. In the original syntax ctyp 
 * is much smaller. We therefore create two new quotations. 
 * First type_info for those parts of ctyp that really belong to type 
 * declarations (and not to type expressions). Second typedef for type 
 * declarations as a whole. Quantified types are included in the ctyp 
 * quotation, therefore ctyp is renamed to ptyp. This gives two 
 * quotation expanders (ptyp, type_info) for the same type (MLast.ctyp),
 * which is a bit odd. However, the type_info quotations simplifies 
 * paqo_o a lot.
 *
 * Further we include the quantified types in the ctyp quotation. 
 *)
let typedef_eoi = Grammar.Entry.create gram "type declaration" in
do {
  EXTEND
    typedef_eoi:
      [ [ x = type_declaration; EOI -> x ] ]
    ;
  END;
  Quotation.add "typedef" (apply_entry typedef_eoi)
};

let type_info_eoi = Grammar.Entry.create gram "type information" in
do {
  EXTEND
   type_info_eoi:
      [ [ x = type_kind; EOI -> x ] ]
    ;
  END;
  Quotation.add "type_info" (apply_entry type_info_eoi)
};

let poly_type_eoi = Grammar.Entry.create gram "type" in
do {
  EXTEND
    poly_type_eoi:
      [ [ x = poly_type; EOI -> x ] ]
    ;
  END;
  Quotation.add "ptyp" (apply_entry poly_type_eoi)
};

let patt_eoi = Grammar.Entry.create gram "pattern" in
do {
  EXTEND
    patt_eoi:
      [ [ x = patt; EOI -> x ] ]
    ;
  END;
  Quotation.add "patt" (apply_entry patt_eoi)
}; 

let expr_eoi = Grammar.Entry.create gram "expression" in
do {
  EXTEND
    expr_eoi:
      [ [ x = expr; EOI -> x ] ]
    ;
  END;
  Quotation.add "expr" (apply_entry expr_eoi)
};

let module_type_eoi = Grammar.Entry.create gram "module type" in
do {
  EXTEND
    module_type_eoi:
      [ [ x = module_type; EOI -> x ] ]
    ;
  END;
  Quotation.add "module_type" (apply_entry module_type_eoi)
};

let module_expr_eoi = Grammar.Entry.create gram "module expression" in
do {
  EXTEND
    module_expr_eoi:
      [ [ x = module_expr; EOI -> x ] ]
    ;
  END;
  Quotation.add "module_expr" (apply_entry module_expr_eoi)
};

let class_type_eoi = Grammar.Entry.create gram "class_type" in
do {
  EXTEND
    class_type_eoi:
      [ [ x = class_type; EOI -> x ] ]
    ;
  END;
  Quotation.add "class_type" (apply_entry class_type_eoi)
};

let class_expr_eoi = Grammar.Entry.create gram "class_expr" in
do {
  EXTEND
    class_expr_eoi:
      [ [ x = class_expr; EOI -> x ] ]
    ;
  END;
  Quotation.add "class_expr" (apply_entry class_expr_eoi)
};

let class_sig_item_eoi = Grammar.Entry.create gram "class_sig_item" in
do {
  EXTEND
    class_sig_item_eoi:
      [ [ x = class_sig_item; EOI -> x ] ]
    ;
  END;
  Quotation.add "class_sig_item" (apply_entry class_sig_item_eoi)
};

let class_str_item_eoi = Grammar.Entry.create gram "class_str_item" in
do {
  EXTEND
    class_str_item_eoi:
      [ [ x = class_str_item; EOI -> x ] ]
    ;
  END;
  Quotation.add "class_str_item" (apply_entry class_str_item_eoi)
};

let with_constr_eoi = Grammar.Entry.create gram "with constr" in
do {
  EXTEND
    with_constr_eoi:
      [ [ x = with_constr; EOI -> x ] ]
    ;
  END;
  Quotation.add "with_constr" (apply_entry with_constr_eoi)
};





value output_version () = 
 do {
   Printf.printf "pp_qo version %s\n" ocamlp4_version;
   exit 0
 };

Pcaml.add_option "-qo_MLast-version" (Arg.Unit output_version)
  ("output version (" ^ ocamlp4_version ^ ") and exit");



(*** Local Variables: ***)
(*** version-control: t ***)
(*** kept-new-versions: 5 ***)
(*** delete-old-versions: t ***)
(*** time-stamp-line-limit: 30 ***)
(*** End: ***)
