(* (C)        Jean-Franois Monin, 1999            *)
(* Centre National d'Etudes des Tlcommunications *)
(* $Id: splay.mli,v 1.2 2005-10-13 12:57:06 tews Exp $ *)
(* NOT THREAD SAFE : env needs a mutex *)
(* WARNING : compare must not raise any exception *)

module type InhabitedType =
  sig
    type t
    val default : t
  end

module type InhabitedOrderedType =
  sig
    type t
    val default : t
    val compare : t -> t -> int
  end

          (* The input signature of the functor [Splay.Make].
             [key] is the type of keys in the tree.
             [compare] is a total ordering function over the keys.
             This is a two-argument function [f] such that
             [f e1 e2] is zero if the elements [e1] and [e2] are equal,
             [f e1 e2] is strictly negative if [e1] is smaller than [e2],
             and [f e1 e2] is strictly positive if [e1] is greater than [e2].
             Example: a suitable ordering function is
             the generic structural comparison function [compare].
	  *)

module type S =
  sig
    type key
          (* The type of keys in the tree. *)
    type vt
          (* The type of values in the tree. *)
    type t
          (* The type of trees. *)
    exception Already_there
    val print : (key -> unit) -> (vt -> unit) -> t -> unit
	(* prints a tree *)
    val create: unit -> t
          (* The empty tree. *)
    val clear:  t -> unit
          (* clears the tree. *)
    val min_elt: t -> key * vt
        (* Return the smallest element of the given tree
           (with respect to the [Ord.compare] ordering), or raise
           [Not_found] if the tree is empty. *)
    val max_elt: t -> key * vt
        (* Same as [min_elt], but returns the largest element of the
           given tree. *)
    val find: t -> key -> vt
        (* [find s x] is an element y of [s] such that [compare x y = 0].
	   If [x] was not in [s], [Not_found] is raised. *)
    val mem: t -> key -> bool
        (* [mem s x] is true if there is an element y of [s]
	   such that [compare x y = 0]. *)
    val add: t -> key -> vt -> unit
        (* [add s k x] adds the element [k,x] to the tree [s],
           If [x] was already in [s], [Already_there] is raised. *)
    val remove: t -> key -> unit
        (* [remove s x] returns a tree containing all elements of [s],
           except [y] such that [compare x y = 0].
	   If [x] was not in [s], nothing is done (excepted rearrangement) *)
    val set: t -> key -> vt -> unit
        (* [set s k x] replaces the element [k,_] of the tree [s] by [k,x]
           If [x] was not already in [s], [k,x] is simply added.          *)
    val sub: t -> key -> key -> t
	(* [sub s c1 c2] returns a fresh tree of consecutive elements of [s] *)
        (* such that their key [c] satisfies  [c1 <= c < c2] *)
    val from: t -> key -> t
	(* [from s c1] returns a fresh tree of consecutive elements of [s] *)
        (* such that their key [c] satisfies  [c1 <= c] *)
    val floor: t -> key -> key * vt
	(* [floor s c] returns the greatest element with [key <= c] *)
    val ceil: t -> key -> key * vt
	(* [ceil s c] returns the smallest element with [key >= c] *)
    val prev: t -> key -> key * vt
	(* [prev s c] returns the greatest element with [key < c] *)
    val next: t -> key -> key * vt
	(* [next s c] returns the smallest element with [key > c] *)

    (* The following functions don't have any side effect on their tree arg *)

    val copy: t -> t
          (* creates a fresh copy *)
    val iter: (key -> vt -> unit) -> t -> unit
        (* [iter s f] applies [f] in increasing order to all elements of [s] *)
    val fold_left: ('a -> key -> vt -> 'a) -> 'a -> t -> 'a
        (* [fold_left f a s] computes [f (...(f (f a c1 x1) c2 x2)...) cn xn],
           where [c1,x1... cn,xn] are the elements of [s] in increasing order. *)
    val fold_right: (key -> vt -> 'b -> 'b) -> t -> 'b -> 'b
        (* [fold_right f s b] computes [f c1 x1 (... (f c2 x2 (f cn xn b))...)],
           where [c1,x1... cn,xn] are the elements of [s] in increasing order. *)
    val cardinal: t -> int
        (* Return the number of elements of a tree. *)
    val is_empty: t -> bool
        (* Test whether a tree is empty or not. *)
    val to_list: t -> (key * vt) list
        (* Returns the list of all elements of the given tree
           in increasing order wrt [Ord.compare] *)
    val to_stream : t -> (key * vt) Stream.t
        (* Returns the stream of all elements of the given tree
           in increasing order wrt [Ord.compare] *)
    val filter: (key -> vt -> bool) -> t ->  t
        (* [filter p s] returns a tree of all elements [c,v] of [s]
	   such that [p c v], in increasing order wrt [Ord.compare] *)
(*     val map: (key -> vt -> 'b) -> t ->  'b t *)
        (* [map f s] returns an tree to [s] where values v 
	   are replaced with [f c v] *)

  end

module Make(Ord: InhabitedOrderedType) (Val: InhabitedType):
    (S with type key = Ord.t and type vt = Val.t)
        (* Functor building an implementation of the tree structure
           given a totally ordered type. *)
